#MULTI-DIMENSIONAL PROBLEM: SENSORY-MOTOR TASK - PLOTS 

#PACKAGES -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------

using Optim, Plots, DelimitedFiles, LinearAlgebra, Random, StatsBase, FiniteDifferences, LaTeXStrings , EasyFit, Printf, FFTW, Pkg, Noise, Clustering, Dierckx, BSplineKit, MultivariateStats, Flux, Combinatorics, Bigsimr, DataFrames, JLD

#FUNCTIONS -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------

#functions for Todorov's and numerical GD optimization, with useful functions to get model predictions.
cd(@__DIR__)#to go back to the directory of the script
include("../../MyFunctions/functions_sensory_motor_task.jl")

#Functions for full GD algorithms (1D and multiple dimensions)
cd(@__DIR__)#to go back to the directory of the script
include("../../MyFunctions/functions_full_GD_algorithms.jl")

#quality of figures
dpi_value = 500 # A higher dpi value will result in a sharper image but may also increase the file size.
default(dpi=dpi_value)

#TASK PARAMETERS, we follow (Todorov, 2005) -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------
dimension_of_state = 4 #We chose a system with a 4D state vector, instead of the 5D system of Todorov: see comments about that in the document "functions_Reaching_task_application".
dimension_of_control = 1 #1D control.
observation_dimension = 1 #We chose a 1D sensory feedback for simplicity and to give major emphasis to the visual stimuli (for which the multiplicative and internal noise 
#description could be more significant: eccentricity for multiplicative noise and memory leak for what concerns the internal noise - considering a case in which we need to
#remember the position of the target, like in the navigation task).

Δt = 0.010 #seconds
N_steps = 100 #we use the same time steps as the estimation task of (Todorov, 2005)
m = 1 #Kg, mass of the hand, modelled as a point mass
target_position_mean = 0.15 #meters; mean of the gaussian distribution from which the target position is extracted. We use here the params used in the estimation task of 
#(Todorov, 2005), so that the target is in the periphery and multiplicative sensory noise plays a role. Note that three more means are used: 0.05m and 0.25m.
target_position_std = 0.00 #Note that optimal controls and estimators could depende on the target position: consider putting zero variance of that to simplify the problem at
#the moment. In (Todorov, 2005) they use (0.05m)^2
tau_1 = 0.04 #seconds; tau_1 and tau_2 are the time constants of the second oreder low pass filter to map the control signal into the force acting on the hand
tau_2 = 0.04 #seconds
r = 0.00001 #as in the reaching atsk of (Todorov, 2005)
w_v = 0.2 #w_v and #w_f are computed by considering Todorov's choices in (Todorov, 2005) and rescaling them to our case in which target_displacement is 0.15 and N_steps = 100 
#(we thus just divided by 2 the values used in the paper)
w_f = 0.01 

#NOISE terms 
σ_ξ = 0.00 #we only use control dependent noise acting on the dynamics to mimic the reaching task of (Todorov, 2005).  
σ_ω_add_sensory_noise = 0.00 #we only use sensory dependent noise acting on the dynamics to mimic the estimation task of (Todorov, 2005).  
σ_ρ_mult_sensory_noise = 0.5 #as in the estimation task of (Todorov, 2005).
σ_ϵ_mult_control_noise = 0.5 #as in the reaching task of (Todorov, 2005).

σ_η_internal_noise = 0.0 #meters, as in the estimation task of (Todorov, 2005).
#we also consider here possible internal fluctuations on the other components of the state estimate 
σ_η_internal_noise_velocity = 0.0
σ_η_internal_noise_force = 0.0
σ_η_internal_noise_filtered_control = 0.0
#NOTE THAT IN THE PREVIOUS SIMULATIONS WE HAVE ALWAYS USED σ_η_internal_noise_velocity = σ_η_internal_noise_force = σ_η_internal_noise_filtered_control = 0.0.

σ_η_internal_noise_vec = [0.005,0.05,0.5] #Internal noise is acting only on the osberved variables (position, velocity: visual feedbacks; force: proprioception)

#Initial conditions (mean state and state estimate, and their covariances) -- Note that initial state and state estimate are considered to be uncorrelated at t=1 (initial time step)

#Mean initial state and state estimates -- Note that we assume <x_1> = <z_1>

#note that x_state_0 is the initial mean state of the vector
x_state_mean_0 = zeros(dimension_of_state) #meters. 
x_state_mean_0[1] = target_position_mean 
x_state_mean_0[2] = 0
x_state_mean_0[3] = 0
x_state_mean_0[4] = 0

x_state_estimate_mean_0 = zeros(dimension_of_state)
x_state_estimate_mean_0[:] .= x_state_mean_0[:]

#State covariance 
target_position_variance = target_position_std^2
velocity_initial_variance = 0
force_initial_variance = 0
filtered_control_initial_variance = 0

Σ_1_x = Diagonal(ones(dimension_of_state)) #intial covariance of the state

Σ_1_x[1,1] = target_position_variance
Σ_1_x[2,2] = velocity_initial_variance
Σ_1_x[3,3] = force_initial_variance
Σ_1_x[4,4] = filtered_control_initial_variance

#State estimate covariance 
target_position_estimate_initial_variance = 0
velocity_estimate_initial_variance = 0
force_estimate_initial_variance = 0
filtered_control_estimate_initial_variance = 0

Σ_1_z = Diagonal(ones(dimension_of_state)) #intial covariance of the state estimate 

Σ_1_z[1,1] = target_position_estimate_initial_variance
Σ_1_z[2,2] = velocity_estimate_initial_variance
Σ_1_z[3,3] = force_estimate_initial_variance
Σ_1_z[4,4] = filtered_control_estimate_initial_variance

#Initial condition of raw moments.
#S_0 is a 4x4 block matrix containing the inital condition for the propagation of the raw moments (mxm matrices, where m is the dimension of the state), and it's defined this way: 
#S_0 = [<x_1*transpose(x_1)>, <x_1*transpose(z_1)>; <z_1*transpose(x_1)>, <z_1*transpose(z_1)>], where x is the state and z the state estimate. 
#This variable is used as an argument for "raw_moments_propagation_for_full_GD", used for the control (and estimation) optimization of the full GD.

S_0 = [zeros(dimension_of_state, dimension_of_state) for _ in 1:2, _ in 1:2]

S_0[1,1] .= Σ_1_x .+ x_state_mean_0*transpose(x_state_mean_0)
S_0[1,2] .= x_state_estimate_mean_0*transpose(x_state_mean_0)
S_0[2,1] .= x_state_mean_0*transpose(x_state_estimate_mean_0)
S_0[2,2] .= Σ_1_z .+ x_state_estimate_mean_0*transpose(x_state_estimate_mean_0)

#Matrices of the problem 
#Define the dynamical system
A = zeros(dimension_of_state, dimension_of_state)
B = zeros(dimension_of_state, dimension_of_control)
H = zeros(observation_dimension, dimension_of_state)
D = zeros(observation_dimension, dimension_of_state)

Ω_ξ = Diagonal([0, σ_ξ^2, 0, 0]) #we consider the sqrt of the covariance matrix, so we don't need to call the function sqrt at each time step iteration
Ω_ω = σ_ω_add_sensory_noise^2 #Additive sensory noise (1D sensory feedback)
Ω_η = Diagonal([σ_η_internal_noise, σ_η_internal_noise_velocity^2, σ_η_internal_noise_force^2, σ_η_internal_noise_filtered_control^2]) #Internal noise is acting only on the osberved variables (position, velocity: visual feedbacks; force: proprioception)

A[1,:] = [1, Δt, 0, 0]
A[2,:] = [0, 1, Δt/m, 0]
A[3,:] = [0, 0, 1-Δt/tau_2, Δt/tau_2]
A[4,:] = [0, 0, 0, 1-Δt/tau_1]

B[:] = [0, 0, 0, Δt/tau_1]
C = σ_ϵ_mult_control_noise .* B
H[1,:] = [1, 0, 0, 0] #we're selecting only the position as observable 

D[1,:] = σ_ρ_mult_sensory_noise .* H[1,:]

#Matrix for cost function (quadratic form)
p_vec = [1; 0; 0; 0]
v_vec = [0; w_v; 0; 0]
f_vec = [0; 0; w_f; 0]

R_matrix = zeros(dimension_of_control, dimension_of_control, N_steps)
for i in 1:N_steps-1
    R_matrix[:,:,i] .= r/(N_steps-1)
end

Q_matrix = zeros(dimension_of_state, dimension_of_state, N_steps)
Q_N = p_vec*transpose(p_vec) .+ v_vec*transpose(v_vec) .+ f_vec*transpose(f_vec) #Matrix for the cost at the last time step 
Q_matrix[:,:,N_steps] .= Q_N[:,:]

#Load optimal solutions -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------

#Folder
#functions for Todorov's and numerical GD optimization, with useful functions to get model predictions.
cd(@__DIR__)#to go back to the directory of the script

folder_location = "../../Data/Multi_D_problems/sensory_motor_task"

#σ_η = 0.0 
#Tod
file_name_K_TOD_s_0 = "Reaching_task_NEURIPS_TOD_K_s_eta_0_0.jld"
file_path_K_TOD_s_0 = folder_location * "/" * file_name_K_TOD_s_0
K_TOD_s_0 = load(file_path_K_TOD_s_0, "K_TOD")

file_name_L_TOD_s_0 = "Reaching_task_NEURIPS_TOD_L_s_eta_0_0.jld"
file_path_L_TOD_s_0 = folder_location * "/" * file_name_L_TOD_s_0
L_TOD_s_0 = load(file_path_L_TOD_s_0, "L_TOD")

#GD
file_name_K_GD_s_0 = "Reaching_task_NEURIPS_GD_K_s_eta_0_0.jld"
file_path_K_GD_s_0 = folder_location * "/" * file_name_K_GD_s_0
K_GD_s_0 = load(file_path_K_GD_s_0, "K_GD")

file_name_L_GD_s_0 = "Reaching_task_NEURIPS_GD_L_s_eta_0_0.jld"
file_path_L_GD_s_0 = folder_location * "/" * file_name_L_GD_s_0
L_GD_s_0 = load(file_path_L_GD_s_0, "L_GD")

#σ_η = 0.005
#Tod
file_name_K_TOD_s_0_005 = "Reaching_task_NEURIPS_TOD_K_s_eta_0_005.jld"
file_path_K_TOD_s_0_005 = folder_location * "/" * file_name_K_TOD_s_0_005
K_TOD_s_0_005 = load(file_path_K_TOD_s_0_005, "K_TOD")

file_name_L_TOD_s_0_005 = "Reaching_task_NEURIPS_TOD_L_s_eta_0_005.jld"
file_path_L_TOD_s_0_005 = folder_location * "/" * file_name_L_TOD_s_0_005
L_TOD_s_0_005 = load(file_path_L_TOD_s_0_005, "L_TOD")

#GD
file_name_K_GD_s_0_005 = "Reaching_task_NEURIPS_GD_K_s_eta_0_005.jld"
file_path_K_GD_s_0_005 = folder_location * "/" * file_name_K_GD_s_0_005
K_GD_s_0_005 = load(file_path_K_GD_s_0_005, "K_GD")

file_name_L_GD_s_0_005 = "Reaching_task_NEURIPS_GD_L_s_eta_0_005.jld"
file_path_L_GD_s_0_005 = folder_location * "/" * file_name_L_GD_s_0_005
L_GD_s_0_005 = load(file_path_L_GD_s_0_005, "L_GD")

#σ_η = 0.05
#Tod
file_name_K_TOD_s_0_05 = "Reaching_task_NEURIPS_TOD_K_s_eta_0_05.jld"
file_path_K_TOD_s_0_05 = folder_location * "/" * file_name_K_TOD_s_0_05
K_TOD_s_0_05 = load(file_path_K_TOD_s_0_05, "K_TOD")

file_name_L_TOD_s_0_05 = "Reaching_task_NEURIPS_TOD_L_s_eta_0_05.jld"
file_path_L_TOD_s_0_05 = folder_location * "/" * file_name_L_TOD_s_0_05
L_TOD_s_0_05 = load(file_path_L_TOD_s_0_05, "L_TOD")

#GD
file_name_K_GD_s_0_05 = "Reaching_task_NEURIPS_GD_K_s_eta_0_05.jld"
file_path_K_GD_s_0_05 = folder_location * "/" * file_name_K_GD_s_0_05
K_GD_s_0_05 = load(file_path_K_GD_s_0_05, "K_GD")

file_name_L_GD_s_0_05 = "Reaching_task_NEURIPS_GD_L_s_eta_0_05.jld"
file_path_L_GD_s_0_05 = folder_location * "/" * file_name_L_GD_s_0_05
L_GD_s_0_05 = load(file_path_L_GD_s_0_05, "L_GD")

#σ_η = 0.5
#Tod
file_name_K_TOD_s_0_5 = "Reaching_task_NEURIPS_TOD_K_s_eta_0_5.jld"
file_path_K_TOD_s_0_5 = folder_location * "/" * file_name_K_TOD_s_0_5
K_TOD_s_0_5 = load(file_path_K_TOD_s_0_5, "K_TOD")

file_name_L_TOD_s_0_5 = "Reaching_task_NEURIPS_TOD_L_s_eta_0_5.jld"
file_path_L_TOD_s_0_5 = folder_location * "/" * file_name_L_TOD_s_0_5
L_TOD_s_0_5 = load(file_path_L_TOD_s_0_5, "L_TOD")

#GD
file_name_K_GD_s_0_5 = "Reaching_task_NEURIPS_GD_K_s_eta_0_5.jld"
file_path_K_GD_s_0_5 = folder_location * "/" * file_name_K_GD_s_0_5
K_GD_s_0_5 = load(file_path_K_GD_s_0_5, "K_GD")

file_name_L_GD_s_0_5 = "Reaching_task_NEURIPS_GD_L_s_eta_0_5.jld"
file_path_L_GD_s_0_5 = folder_location * "/" * file_name_L_GD_s_0_5
L_GD_s_0_5 = load(file_path_L_GD_s_0_5, "L_GD")

#MONTECARLO SIMULATIONS PARAMETERS -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------

num_trials_MC_runs = 50000
random_seed_MC = rand(1:1000000)

#EXPECTED COSTS -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------

σ_η_tmp = 0.0
println("\n------- (0) σ_η = $σ_η_tmp --------------------------\n")

Ω_η = Diagonal([0, 0, 0, 0]) #Internal noise is acting only on the osberved variables (position, velocity: visual feedbacks; force: proprioception)

#TODOROV'S APPROACH

#Montecarlo simulations
final_cost_average_TOD_s_0, final_cost_average_std_TOD_s_0, p_t_average_TOD_s_0, p_t_average_std_TOD_s_0, p_hat_t_average_TOD_s_0, p_hat_t_average_std_TOD_s_0, control_t_average_TOD_s_0, control_t_average_std_TOD_s_0, p_t_TOD_s_0, p_hat_t_TOD_s_0, control_t_TOD_s_0, accumulated_cost_t_TOD_s_0 = get_model_predictions_reaching_task_application_4D_state(num_trials_MC_runs, K_TOD_s_0, L_TOD_s_0, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η, random_seed_MC)

#Expected cost using moment's propagation
expected_cost_mom_prop_TOD_s_0 = moments_propagation_reaching_task_application_4D_state(K_TOD_s_0, L_TOD_s_0, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η)

println("TODOROV's OPTIMIZATION\n")
println("- Expected cost using moment propagation (Todorov): $expected_cost_mom_prop_TOD_s_0\n")
println("- Montecarlo - cost function (Todorov): $final_cost_average_TOD_s_0  (error: $final_cost_average_std_TOD_s_0)\n")

#GD APPROACH
#Montecarlo simulations
final_cost_average_GD_s_0, final_cost_average_std_GD_s_0, p_t_average_GD_s_0, p_t_average_std_GD_s_0, p_hat_t_average_GD_s_0, p_hat_t_average_std_GD_s_0, control_t_average_GD_s_0, control_t_average_std_GD_s_0, p_t_GD_s_0, p_hat_t_GD_s_0, control_t_GD_s_0, accumulated_cost_t_GD_s_0 = get_model_predictions_reaching_task_application_4D_state(num_trials_MC_runs, K_GD_s_0, L_GD_s_0, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η, random_seed_MC)

#Expected cost using moment's propagation
expected_cost_mom_prop_GD_s_0 = moments_propagation_reaching_task_application_4D_state(K_GD_s_0, L_GD_s_0, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η)

println("GD OPTIMIZATION\n")
println("- Expected cost using moment propagation (GD): $expected_cost_mom_prop_GD_s_0\n")
println("- Montecarlo - cost function (GD): $final_cost_average_GD_s_0  (error: $final_cost_average_std_GD_s_0)\n")

σ_η_tmp = σ_η_internal_noise_vec[1]
println("\n------- (1) σ_η = $σ_η_tmp --------------------------\n")

Ω_η = Diagonal([σ_η_internal_noise_vec[1]^2, 0, 0, 0]) #Internal noise is acting only on the osberved variables (position, velocity: visual feedbacks; force: proprioception)

#TODOROV'S APPROACH

#Montecarlo simulations
final_cost_average_TOD_s_0_005, final_cost_average_std_TOD_s_0_005, p_t_average_TOD_s_0_005, p_t_average_std_TOD_s_0_005, p_hat_t_average_TOD_s_0_005, p_hat_t_average_std_TOD_s_0_005, control_t_average_TOD_s_0_005, control_t_average_std_TOD_s_0_005, p_t_TOD_s_0_005, p_hat_t_TOD_s_0_005, control_t_TOD_s_0_005, accumulated_cost_t_TOD_s_0_005 = get_model_predictions_reaching_task_application_4D_state(num_trials_MC_runs, K_TOD_s_0_005, L_TOD_s_0_005, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η, random_seed_MC)

#Expected cost using moment's propagation
expected_cost_mom_prop_TOD_s_0_005 = moments_propagation_reaching_task_application_4D_state(K_TOD_s_0_005, L_TOD_s_0_005, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η)

println("TODOROV's OPTIMIZATION\n")
println("- Expected cost using moment propagation (Todorov): $expected_cost_mom_prop_TOD_s_0_005\n")
println("- Montecarlo - cost function (Todorov): $final_cost_average_TOD_s_0_005  (error: $final_cost_average_std_TOD_s_0_005)\n")

#GD APPROACH
#Montecarlo simulations
final_cost_average_GD_s_0_005, final_cost_average_std_GD_s_0_005, p_t_average_GD_s_0_005, p_t_average_std_GD_s_0_005, p_hat_t_average_GD_s_0_005, p_hat_t_average_std_GD_s_0_005, control_t_average_GD_s_0_005, control_t_average_std_GD_s_0_005, p_t_GD_s_0_005, p_hat_t_GD_s_0_005, control_t_GD_s_0_005, accumulated_cost_t_GD_s_0_005 = get_model_predictions_reaching_task_application_4D_state(num_trials_MC_runs, K_GD_s_0_005, L_GD_s_0_005, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η, random_seed_MC)

#Expected cost using moment's propagation
expected_cost_mom_prop_GD_s_0_005 = moments_propagation_reaching_task_application_4D_state(K_GD_s_0_005, L_GD_s_0_005, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η)

println("GD OPTIMIZATION\n")
println("- Expected cost using moment propagation (GD): $expected_cost_mom_prop_GD_s_0_005\n")
println("- Montecarlo - cost function (GD): $final_cost_average_GD_s_0_005  (error: $final_cost_average_std_GD_s_0_005)\n")

σ_η_tmp = σ_η_internal_noise_vec[2]
println("\n------- (2) σ_η = $σ_η_tmp --------------------------\n")

Ω_η = Diagonal([σ_η_internal_noise_vec[2]^2, 0, 0, 0]) #Internal noise is acting only on the osberved variables (position, velocity: visual feedbacks; force: proprioception)

#TODOROV'S APPROACH

#Montecarlo simulations
final_cost_average_TOD_s_0_05, final_cost_average_std_TOD_s_0_05, p_t_average_TOD_s_0_05, p_t_average_std_TOD_s_0_05, p_hat_t_average_TOD_s_0_05, p_hat_t_average_std_TOD_s_0_05, control_t_average_TOD_s_0_05, control_t_average_std_TOD_s_0_05, p_t_TOD_s_0_05, p_hat_t_TOD_s_0_05, control_t_TOD_s_0_05, accumulated_cost_t_TOD_s_0_05 = get_model_predictions_reaching_task_application_4D_state(num_trials_MC_runs, K_TOD_s_0_05, L_TOD_s_0_05, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η, random_seed_MC)

#Expected cost using moment's propagation
expected_cost_mom_prop_TOD_s_0_05 = moments_propagation_reaching_task_application_4D_state(K_TOD_s_0_05, L_TOD_s_0_05, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η)

println("TODOROV's OPTIMIZATION\n")
println("- Expected cost using moment propagation (Todorov): $expected_cost_mom_prop_TOD_s_0_05\n")
println("- Montecarlo - cost function (Todorov): $final_cost_average_TOD_s_0_05  (error: $final_cost_average_std_TOD_s_0_05)\n")

#GD APPROACH
#Montecarlo simulations
final_cost_average_GD_s_0_05, final_cost_average_std_GD_s_0_05, p_t_average_GD_s_0_05, p_t_average_std_GD_s_0_05, p_hat_t_average_GD_s_0_05, p_hat_t_average_std_GD_s_0_05, control_t_average_GD_s_0_05, control_t_average_std_GD_s_0_05, p_t_GD_s_0_05, p_hat_t_GD_s_0_05, control_t_GD_s_0_05, accumulated_cost_t_GD_s_0_05 = get_model_predictions_reaching_task_application_4D_state(num_trials_MC_runs, K_GD_s_0_05, L_GD_s_0_05, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η, random_seed_MC)

#Expected cost using moment's propagation
expected_cost_mom_prop_GD_s_0_05 = moments_propagation_reaching_task_application_4D_state(K_GD_s_0_05, L_GD_s_0_05, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η)

println("GD OPTIMIZATION\n")
println("- Expected cost using moment propagation (GD): $expected_cost_mom_prop_GD_s_0_05\n")
println("- Montecarlo - cost function (GD): $final_cost_average_GD_s_0_05  (error: $final_cost_average_std_GD_s_0_05)\n")

σ_η_tmp = σ_η_internal_noise_vec[3]
println("\n------- (3) σ_η = $σ_η_tmp --------------------------\n")

Ω_η = Diagonal([σ_η_internal_noise_vec[3]^2, 0, 0, 0]) #Internal noise is acting only on the osberved variables (position, velocity: visual feedbacks; force: proprioception)

#TODOROV'S APPROACH

#Montecarlo simulations
final_cost_average_TOD_s_0_5, final_cost_average_std_TOD_s_0_5, p_t_average_TOD_s_0_5, p_t_average_std_TOD_s_0_5, p_hat_t_average_TOD_s_0_5, p_hat_t_average_std_TOD_s_0_5, control_t_average_TOD_s_0_5, control_t_average_std_TOD_s_0_5, p_t_TOD_s_0_5, p_hat_t_TOD_s_0_5, control_t_TOD_s_0_5, accumulated_cost_t_TOD_s_0_5 = get_model_predictions_reaching_task_application_4D_state(num_trials_MC_runs, K_TOD_s_0_5, L_TOD_s_0_5, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η, random_seed_MC)

#Expected cost using moment's propagation
expected_cost_mom_prop_TOD_s_0_5 = moments_propagation_reaching_task_application_4D_state(K_TOD_s_0_5, L_TOD_s_0_5, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η)

println("TODOROV's OPTIMIZATION\n")
println("- Expected cost using moment propagation (Todorov): $expected_cost_mom_prop_TOD_s_0_5\n")
println("- Montecarlo - cost function (Todorov): $final_cost_average_TOD_s_0_5  (error: $final_cost_average_std_TOD_s_0_5)\n")

#GD APPROACH
#Montecarlo simulations
final_cost_average_GD_s_0_5, final_cost_average_std_GD_s_0_5, p_t_average_GD_s_0_5, p_t_average_std_GD_s_0_5, p_hat_t_average_GD_s_0_5, p_hat_t_average_std_GD_s_0_5, control_t_average_GD_s_0_5, control_t_average_std_GD_s_0_5, p_t_GD_s_0_5, p_hat_t_GD_s_0_5, control_t_GD_s_0_5, accumulated_cost_t_GD_s_0_5 = get_model_predictions_reaching_task_application_4D_state(num_trials_MC_runs, K_GD_s_0_5, L_GD_s_0_5, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η, random_seed_MC)

#Expected cost using moment's propagation
expected_cost_mom_prop_GD_s_0_5 = moments_propagation_reaching_task_application_4D_state(K_GD_s_0_5, L_GD_s_0_5, Δt, N_steps, m, tau_1, tau_2, x_state_mean_0, Σ_1_x, Σ_1_z, r, w_v, w_f, σ_ξ, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_ϵ_mult_control_noise, Ω_η)

println("GD OPTIMIZATION\n")
println("- Expected cost using moment propagation (GD): $expected_cost_mom_prop_GD_s_0_5\n")
println("- Montecarlo - cost function (GD): $final_cost_average_GD_s_0_5  (error: $final_cost_average_std_GD_s_0_5)\n")

#FOLDER TO SAVE PLOTS 

name_plots = ""

cd(@__DIR__)#to go back to the directory of the script
full_folder_save_plots = "../Multi_D_Plots/Figures_sensory_motor_task"

#PLOTS -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------

#norms of L and K vectors
norms_control_TOD_s_0 = zeros(N_steps-1)
norms_control_TOD_s_0_005 = zeros(N_steps-1)
norms_control_TOD_s_0_05 = zeros(N_steps-1)
norms_control_TOD_s_0_5 = zeros(N_steps-1)
norms_estimator_TOD_s_0 = zeros(N_steps-1)
norms_estimator_TOD_s_0_005 = zeros(N_steps-1)
norms_estimator_TOD_s_0_05 = zeros(N_steps-1)
norms_estimator_TOD_s_0_5 = zeros(N_steps-1)

norms_control_GD_s_0 = zeros(N_steps-1)
norms_control_GD_s_0_005 = zeros(N_steps-1)
norms_control_GD_s_0_05 = zeros(N_steps-1)
norms_control_GD_s_0_5 = zeros(N_steps-1)
norms_estimator_GD_s_0 = zeros(N_steps-1)
norms_estimator_GD_s_0_005 = zeros(N_steps-1)
norms_estimator_GD_s_0_05 = zeros(N_steps-1)
norms_estimator_GD_s_0_5 = zeros(N_steps-1)

for i in 1:N_steps-1

    #TOD
    norms_control_TOD_s_0[i] = norm(L_TOD_s_0[:, :, i])
    norms_estimator_TOD_s_0[i] = norm(K_TOD_s_0[:, :, i])
    norms_control_TOD_s_0_005[i] = norm(L_TOD_s_0_005[:, :, i])
    norms_estimator_TOD_s_0_005[i] = norm(K_TOD_s_0_005[:, :, i])
    norms_control_TOD_s_0_05[i] = norm(L_TOD_s_0_05[:, :, i])
    norms_estimator_TOD_s_0_05[i] = norm(K_TOD_s_0_05[:, :, i])
    norms_control_TOD_s_0_5[i] = norm(L_TOD_s_0_5[:, :, i])
    norms_estimator_TOD_s_0_5[i] = norm(K_TOD_s_0_5[:, :, i])

    #GD
    norms_control_GD_s_0[i] = norm(L_GD_s_0[:, :, i])
    norms_estimator_GD_s_0[i] = norm(K_GD_s_0[:, :, i])
    norms_control_GD_s_0_005[i] = norm(L_GD_s_0_005[:, :, i])
    norms_estimator_GD_s_0_005[i] = norm(K_GD_s_0_005[:, :, i])
    norms_control_GD_s_0_05[i] = norm(L_GD_s_0_05[:, :, i])
    norms_estimator_GD_s_0_05[i] = norm(K_GD_s_0_05[:, :, i])
    norms_control_GD_s_0_5[i] = norm(L_GD_s_0_5[:, :, i])
    norms_estimator_GD_s_0_5[i] = norm(K_GD_s_0_5[:, :, i])

end

#plot norms of L vectors for Iterative GD (with TOD estimator) and TOD

plt_L_vec_TOD_GD = plot(collect(1:N_steps-2), norms_control_TOD_s_0[1:end-1], title = "", titlefont = font(12,"Computer Modern"), xlabel = L"$t$", ylabel = L"$|L_t|$", lw = 3, grid = false, label = L"TOD - \sigma_{\eta} = 0.0", color = cgrad(:Blues_5, 21)[10], yscale =:log)
plot!(plt_L_vec_TOD_GD, collect(1:N_steps-2), norms_control_TOD_s_0_005[1:end-1], lw = 3, label = L"TOD - \sigma_{\eta} = 0.005", color = cgrad(:Blues_5, 21)[14])
plot!(plt_L_vec_TOD_GD, collect(1:N_steps-2), norms_control_TOD_s_0_05[1:end-1], lw = 3, label = L"TOD - \sigma_{\eta} = 0.05", color = cgrad(:Blues_5, 21)[17])
plot!(plt_L_vec_TOD_GD, collect(1:N_steps-2), norms_control_TOD_s_0_5[1:end-1], lw = 3, label = L"TOD - \sigma_{\eta} = 0.5", color = cgrad(:Blues_5, 21)[21])

plot!(plt_L_vec_TOD_GD, collect(1:N_steps-2), norms_control_GD_s_0_005[1:end-1], lw = 3, label = L"GD - \sigma_{\eta} = 0.005", color = cgrad(:BuGn_5, 21)[14])
plot!(plt_L_vec_TOD_GD, collect(1:N_steps-2), norms_control_GD_s_0_05[1:end-1], lw = 3, label = L"GD - \sigma_{\eta} = 0.05", color = cgrad(:BuGn_5, 21)[17])
plot!(plt_L_vec_TOD_GD, collect(1:N_steps-2), norms_control_GD_s_0_5[1:end-1], lw = 3, label = L"GD - \sigma_{\eta} = 0.5", color = cgrad(:BuGn_5, 21)[21])

plt_L_vec_TOD_GD = plot(plt_L_vec_TOD_GD, xtickfontsize=12, ytickfontsize=12, xguidefontsize=18, yguidefontsize=18, thickness_scaling=2.0, size = (1300, 600), grid = false, leg =:false)
name_and_path_plot = full_folder_save_plots * "/task_norm_L" * name_plots * ".pdf" 
savefig(plt_L_vec_TOD_GD, name_and_path_plot)

#plot of the control in one trial 
#single trial 
trial_num = 500

plt_control_one_trial = plot(collect(1:N_steps-1), control_t_TOD_s_0_05[trial_num,:], title = "", titlefont = font(12,"Computer Modern"), xlabel = L"$t$", ylabel = L"$u_t$", lw = 3, grid = false, label = "TOD", color = cgrad(:Blues_5, 21)[17])
plot!(plt_control_one_trial, collect(1:N_steps-1), control_t_GD_s_0_05[trial_num,:], ribbon = p_t_average_std_TOD_s_0_05[:], lw = 3, label = "GD", color = cgrad(:BuGn_5, 21)[17], leg = false, ylims = (-20,20))

plt_control_one_trial = plot(plt_control_one_trial, xtickfontsize=12, ytickfontsize=12, xguidefontsize=18, yguidefontsize=18, thickness_scaling=2.0, size = (1300, 600), grid = false, leg =:false)

name_and_path_plot = full_folder_save_plots * "/task_control_one_trial" * name_plots * ".pdf" 
savefig(plt_control_one_trial, name_and_path_plot)

layout_norm = @layout [a b]
plt_task_L_u = plot(plt_L_vec_TOD_GD, plt_control_one_trial, layout = layout_norm)
plt_task_L_u = plot(plt_task_L_u, xtickfontsize=12, ytickfontsize=12, xguidefontsize=18, yguidefontsize=18, thickness_scaling=2.0, size = (1300, 600), grid = false, leg =:false)

name_and_path_plot = full_folder_save_plots * "/task_norm_L_and_control_one_trial" * name_plots * ".pdf" 
savefig(plt_task_L_u, name_and_path_plot)

#plot of average position over time in the two algorithms 
ylims_p_t = (-0.05, 0.17)

σ_η_tmp = σ_η_internal_noise_vec[2]
plt_reaching_task_position_2 = plot(collect(1:N_steps), p_t_average_TOD_s_0_05[:], ribbon = p_t_average_std_TOD_s_0_05[:], xlabel = L"$t$", ylabel = L"$\mathbb{E}[p_t]$", lw = 3, grid = false, label = "TOD", color = cgrad(:Blues_5, 21)[17], ylims = ylims_p_t)
plot!(plt_reaching_task_position_2, collect(1:N_steps), p_t_average_GD_s_0_05[:], ribbon = p_t_average_std_GD_s_0_05[:], lw = 3, label = "GD", color = cgrad(:BuGn_5, 21)[17])

plt_reaching_task_position_2 = plot(plt_reaching_task_position_2, xtickfontsize=12, ytickfontsize=12, xguidefontsize=18, yguidefontsize=18, thickness_scaling=2.0, size = (1300, 600), grid = false)

name_and_path_plot = full_folder_save_plots * "/task_average_position_over_time" * name_plots * ".pdf" 
savefig(plt_reaching_task_position_2, name_and_path_plot)

#Plot of performance for the two algorithms [TOD and Numerical GD]
σ_η_internal_noise_position_vec = [0,0.005,0.05,0.5].^2

colors_TOD = [cgrad(:Blues_5, 21)[10], cgrad(:Blues_5, 21)[14], cgrad(:Blues_5, 21)[17], cgrad(:Blues_5, 21)[21]]
colors_GD = [cgrad(:BuGn_5, 21)[10], cgrad(:BuGn_5, 21)[14], cgrad(:BuGn_5, 21)[17], cgrad(:BuGn_5, 21)[21]]

markersize = 5
markerstyles = [:circle, :star5, :diamond]

MC_cost_TOD = [final_cost_average_TOD_s_0, final_cost_average_TOD_s_0_005, final_cost_average_TOD_s_0_05, final_cost_average_TOD_s_0_5]
MC_cost_GD = [final_cost_average_GD_s_0, final_cost_average_GD_s_0_005, final_cost_average_GD_s_0_05, final_cost_average_GD_s_0_5]

MC_std_cost_TOD = [final_cost_average_std_TOD_s_0, final_cost_average_std_TOD_s_0_005, final_cost_average_std_TOD_s_0_05, final_cost_average_std_TOD_s_0_5]
MC_std_cost_GD = [final_cost_average_std_GD_s_0, final_cost_average_std_GD_s_0_005, final_cost_average_std_GD_s_0_05, final_cost_average_std_GD_s_0_5]

plt_cost_comparison = scatter(σ_η_internal_noise_position_vec[2:end], MC_cost_TOD[2:end], yerr = MC_std_cost_TOD[2:end] , c = colors_TOD[2:end], ms=markersize, marker=markerstyles[1], title = "", titlefont = font(12,"Computer Modern"), xlabel = L"$\Omega_{\eta_{1,1}}$", ylabel = L"$\mathbb{E}[J(\Omega_{\eta_{1,1}})]$", lw = 3, grid = false, label = L"TOD", yscale =:log, xscale =:log, ylims = (0.00001, 1), yticks = [0.00001, 0.001, 0.1], legend=:topleft)

plot!(plt_cost_comparison, σ_η_internal_noise_position_vec[2:end], MC_cost_TOD[2:end], linestyle=:dash, color=cgrad(:Blues_5, 21)[10], label="")

scatter!(plt_cost_comparison, σ_η_internal_noise_position_vec[2:end], MC_cost_GD[2:end], yerr = MC_std_cost_GD[2:end] , c = colors_GD[2:end], ms=markersize, marker=markerstyles[1], lw = 3, label = L"GD")
plot!(plt_cost_comparison, σ_η_internal_noise_position_vec[2:end], MC_cost_GD[2:end], linestyle=:dash, color=cgrad(:BuGn_5, 21)[10], label="")

plt_cost_comparison = plot(plt_cost_comparison, xtickfontsize=12, ytickfontsize=12, xguidefontsize=18, yguidefontsize=18, thickness_scaling=2.0, size = (1300, 600), grid = false)

name_and_path_plot = full_folder_save_plots * "/cost_comparison_sensory_motor_task" * name_plots * ".pdf" 
savefig(plt_cost_comparison, name_and_path_plot)

#amount of control: average integral of |control| over time

function compute_mean_integral(matrix, dt)
    num_rows, _ = size(matrix)
    integral_matrix = zeros(Float64, num_rows)

    for i in 1:num_rows
        integral_matrix[i] = sum((matrix[i, 1:end-1] .+ matrix[i, 2:end]) / 2) * dt
    end

    return mean(integral_matrix), std(integral_matrix)

end

overall_control_TOD_s_0, std_overall_control_TOD_s_0 = compute_mean_integral(abs.(control_t_TOD_s_0), Δt)
overall_control_TOD_s_0_005, std_overall_control_TOD_s_0_005 = compute_mean_integral(abs.(control_t_TOD_s_0_005), Δt)
overall_control_TOD_s_0_05, std_overall_control_TOD_s_0_05 = compute_mean_integral(abs.(control_t_TOD_s_0_05), Δt)
overall_control_TOD_s_0_5, std_overall_control_TOD_s_0_5 = compute_mean_integral(abs.(control_t_TOD_s_0_5), Δt)

overall_control_GD_s_0, std_overall_control_GD_s_0 = compute_mean_integral(abs.(control_t_GD_s_0), Δt)
overall_control_GD_s_0_005, std_overall_control_GD_s_0_005 = compute_mean_integral(abs.(control_t_GD_s_0_005), Δt)
overall_control_GD_s_0_05, std_overall_control_GD_s_0_05 = compute_mean_integral(abs.(control_t_GD_s_0_05), Δt)
overall_control_GD_s_0_5, std_overall_control_GD_s_0_5 = compute_mean_integral(abs.(control_t_GD_s_0_5), Δt)

overall_control_vec_GD = [overall_control_GD_s_0, overall_control_GD_s_0_005, overall_control_GD_s_0_05, overall_control_GD_s_0_5]
overall_control_vec_TOD = [overall_control_TOD_s_0, overall_control_TOD_s_0_005, overall_control_TOD_s_0_05, overall_control_TOD_s_0_5]

std_overall_control_vec_GD = [std_overall_control_GD_s_0, std_overall_control_GD_s_0_005, std_overall_control_GD_s_0_05, std_overall_control_GD_s_0_5]
std_overall_control_vec_TOD = [std_overall_control_TOD_s_0, std_overall_control_TOD_s_0_005, std_overall_control_TOD_s_0_05, std_overall_control_TOD_s_0_5]

σ_η_internal_noise_vec_4_vals = [0.0000005,0.005,0.05,0.5].^2
plt_reaching_task_int_control = plot(σ_η_internal_noise_vec_4_vals, overall_control_vec_TOD[:], ribbon = std_overall_control_vec_TOD[:], title = "", titlefont = font(12,"Computer Modern"), xlabel = L"$\Omega_{\eta_{1,1}}$", ylabel = L"$\mathbb{E}[\int|u(t)|dt]$", lw = 3, grid = false, label = "TOD", color = cgrad(:Blues_5, 21)[21])
plot!(plt_reaching_task_int_control, σ_η_internal_noise_vec_4_vals, overall_control_vec_GD[:], ribbon = std_overall_control_vec_GD[:], lw = 3, grid = false, label = "GD", color = cgrad(:BuGn_5, 21)[21], yscale =:log, xscale =:log, legend=:topleft)

plt_reaching_task_int_control = plot(plt_reaching_task_int_control, xtickfontsize=12, ytickfontsize=12, xguidefontsize=18, yguidefontsize=18, thickness_scaling=2.0, size = (1300, 600), grid = false, leg =:false)

name_and_path_plot = full_folder_save_plots * "/average_integral_of_control" * name_plots * ".pdf" 
savefig(plt_reaching_task_int_control, name_and_path_plot)
