/*
 *  Server-side procedures for NFSv4.
 *
 *  Copyright (c) 2002 The Regents of the University of Michigan.
 *  All rights reserved.
 *
 *  Kendrick Smith <kmsmith@umich.edu>
 *  Andy Adamson   <andros@umich.edu>
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *  3. Neither the name of the University nor the names of its
 *     contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *  DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include <linux fs_struct.h="">
#include <linux file.h="">
#include <linux falloc.h="">
#include <linux slab.h="">
#include <linux kthread.h="">
#include <linux namei.h="">

#include <linux addr.h="" sunrpc="">
#include <linux nfs_ssc.h="">

#include "idmap.h"
#include "cache.h"
#include "xdr4.h"
#include "vfs.h"
#include "current_stateid.h"
#include "netns.h"
#include "acl.h"
#include "pnfs.h"
#include "trace.h"

static bool inter_copy_offload_enable;
module_param(inter_copy_offload_enable, bool, 0644);
MODULE_PARM_DESC(inter_copy_offload_enable,
		 "Enable inter server to server copy offload. Default: false");

#ifdef CONFIG_NFSD_V4_2_INTER_SSC
static int nfsd4_ssc_umount_timeout = 900000;		/* default to 15 mins */
module_param(nfsd4_ssc_umount_timeout, int, 0644);
MODULE_PARM_DESC(nfsd4_ssc_umount_timeout,
		"idle msecs before unmount export from source server");
#endif

#define NFSDDBG_FACILITY		NFSDDBG_PROC

static u32 nfsd_attrmask[] = {
	NFSD_WRITEABLE_ATTRS_WORD0,
	NFSD_WRITEABLE_ATTRS_WORD1,
	NFSD_WRITEABLE_ATTRS_WORD2
};

static u32 nfsd41_ex_attrmask[] = {
	NFSD_SUPPATTR_EXCLCREAT_WORD0,
	NFSD_SUPPATTR_EXCLCREAT_WORD1,
	NFSD_SUPPATTR_EXCLCREAT_WORD2
};

static __be32
check_attr_support(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
		   u32 *bmval, u32 *writable)
{
	struct dentry *dentry = cstate-&gt;current_fh.fh_dentry;
	struct svc_export *exp = cstate-&gt;current_fh.fh_export;

	if (!nfsd_attrs_supported(cstate-&gt;minorversion, bmval))
		return nfserr_attrnotsupp;
	if ((bmval[0] &amp; FATTR4_WORD0_ACL) &amp;&amp; !IS_POSIXACL(d_inode(dentry)))
		return nfserr_attrnotsupp;
	if ((bmval[2] &amp; FATTR4_WORD2_SECURITY_LABEL) &amp;&amp;
			!(exp-&gt;ex_flags &amp; NFSEXP_SECURITY_LABEL))
		return nfserr_attrnotsupp;
	if (writable &amp;&amp; !bmval_is_subset(bmval, writable))
		return nfserr_inval;
	if (writable &amp;&amp; (bmval[2] &amp; FATTR4_WORD2_MODE_UMASK) &amp;&amp;
			(bmval[1] &amp; FATTR4_WORD1_MODE))
		return nfserr_inval;
	return nfs_ok;
}

static __be32
nfsd4_check_open_attributes(struct svc_rqst *rqstp,
	struct nfsd4_compound_state *cstate, struct nfsd4_open *open)
{
	__be32 status = nfs_ok;

	if (open-&gt;op_create == NFS4_OPEN_CREATE) {
		if (open-&gt;op_createmode == NFS4_CREATE_UNCHECKED
		    || open-&gt;op_createmode == NFS4_CREATE_GUARDED)
			status = check_attr_support(rqstp, cstate,
					open-&gt;op_bmval, nfsd_attrmask);
		else if (open-&gt;op_createmode == NFS4_CREATE_EXCLUSIVE4_1)
			status = check_attr_support(rqstp, cstate,
					open-&gt;op_bmval, nfsd41_ex_attrmask);
	}

	return status;
}

static int
is_create_with_attrs(struct nfsd4_open *open)
{
	return open-&gt;op_create == NFS4_OPEN_CREATE
		&amp;&amp; (open-&gt;op_createmode == NFS4_CREATE_UNCHECKED
		    || open-&gt;op_createmode == NFS4_CREATE_GUARDED
		    || open-&gt;op_createmode == NFS4_CREATE_EXCLUSIVE4_1);
}

static inline void
fh_dup2(struct svc_fh *dst, struct svc_fh *src)
{
	fh_put(dst);
	dget(src-&gt;fh_dentry);
	if (src-&gt;fh_export)
		exp_get(src-&gt;fh_export);
	*dst = *src;
}

static __be32
do_open_permission(struct svc_rqst *rqstp, struct svc_fh *current_fh, struct nfsd4_open *open, int accmode)
{
	__be32 status;

	if (open-&gt;op_truncate &amp;&amp;
		!(open-&gt;op_share_access &amp; NFS4_SHARE_ACCESS_WRITE))
		return nfserr_inval;

	accmode |= NFSD_MAY_READ_IF_EXEC;

	if (open-&gt;op_share_access &amp; NFS4_SHARE_ACCESS_READ)
		accmode |= NFSD_MAY_READ;
	if (open-&gt;op_share_access &amp; NFS4_SHARE_ACCESS_WRITE)
		accmode |= (NFSD_MAY_WRITE | NFSD_MAY_TRUNC);
	if (open-&gt;op_share_deny &amp; NFS4_SHARE_DENY_READ)
		accmode |= NFSD_MAY_WRITE;

	status = fh_verify(rqstp, current_fh, S_IFREG, accmode);

	return status;
}

static __be32 nfsd_check_obj_isreg(struct svc_fh *fh)
{
	umode_t mode = d_inode(fh-&gt;fh_dentry)-&gt;i_mode;

	if (S_ISREG(mode))
		return nfs_ok;
	if (S_ISDIR(mode))
		return nfserr_isdir;
	/*
	 * Using err_symlink as our catch-all case may look odd; but
	 * there's no other obvious error for this case in 4.0, and we
	 * happen to know that it will cause the linux v4 client to do
	 * the right thing on attempts to open something other than a
	 * regular file.
	 */
	return nfserr_symlink;
}

static void nfsd4_set_open_owner_reply_cache(struct nfsd4_compound_state *cstate, struct nfsd4_open *open, struct svc_fh *resfh)
{
	if (nfsd4_has_session(cstate))
		return;
	fh_copy_shallow(&amp;open-&gt;op_openowner-&gt;oo_owner.so_replay.rp_openfh,
			&amp;resfh-&gt;fh_handle);
}

static inline bool nfsd4_create_is_exclusive(int createmode)
{
	return createmode == NFS4_CREATE_EXCLUSIVE ||
		createmode == NFS4_CREATE_EXCLUSIVE4_1;
}

static __be32
nfsd4_vfs_create(struct svc_fh *fhp, struct dentry *child,
		 struct nfsd4_open *open)
{
	struct file *filp;
	struct path path;
	int oflags;

	oflags = O_CREAT | O_LARGEFILE;
	switch (open-&gt;op_share_access &amp; NFS4_SHARE_ACCESS_BOTH) {
	case NFS4_SHARE_ACCESS_WRITE:
		oflags |= O_WRONLY;
		break;
	case NFS4_SHARE_ACCESS_BOTH:
		oflags |= O_RDWR;
		break;
	default:
		oflags |= O_RDONLY;
	}

	path.mnt = fhp-&gt;fh_export-&gt;ex_path.mnt;
	path.dentry = child;
	filp = dentry_create(&amp;path, oflags, open-&gt;op_iattr.ia_mode,
			     current_cred());
	if (IS_ERR(filp))
		return nfserrno(PTR_ERR(filp));

	open-&gt;op_filp = filp;
	return nfs_ok;
}

/*
 * Implement NFSv4's unchecked, guarded, and exclusive create
 * semantics for regular files. Open state for this new file is
 * subsequently fabricated in nfsd4_process_open2().
 *
 * Upon return, caller must release @fhp and @resfhp.
 */
static __be32
nfsd4_create_file(struct svc_rqst *rqstp, struct svc_fh *fhp,
		  struct svc_fh *resfhp, struct nfsd4_open *open)
{
	struct iattr *iap = &amp;open-&gt;op_iattr;
	struct nfsd_attrs attrs = {
		.na_iattr	= iap,
		.na_seclabel	= &amp;open-&gt;op_label,
	};
	struct dentry *parent, *child;
	__u32 v_mtime, v_atime;
	struct inode *inode;
	__be32 status;
	int host_err;

	if (isdotent(open-&gt;op_fname, open-&gt;op_fnamelen))
		return nfserr_exist;
	if (!(iap-&gt;ia_valid &amp; ATTR_MODE))
		iap-&gt;ia_mode = 0;

	status = fh_verify(rqstp, fhp, S_IFDIR, NFSD_MAY_EXEC);
	if (status != nfs_ok)
		return status;
	parent = fhp-&gt;fh_dentry;
	inode = d_inode(parent);

	host_err = fh_want_write(fhp);
	if (host_err)
		return nfserrno(host_err);

	if (is_create_with_attrs(open))
		nfsd4_acl_to_attr(NF4REG, open-&gt;op_acl, &amp;attrs);

	inode_lock_nested(inode, I_MUTEX_PARENT);

	child = lookup_one_len(open-&gt;op_fname, parent, open-&gt;op_fnamelen);
	if (IS_ERR(child)) {
		status = nfserrno(PTR_ERR(child));
		goto out;
	}

	if (d_really_is_negative(child)) {
		status = fh_verify(rqstp, fhp, S_IFDIR, NFSD_MAY_CREATE);
		if (status != nfs_ok)
			goto out;
	}

	status = fh_compose(resfhp, fhp-&gt;fh_export, child, fhp);
	if (status != nfs_ok)
		goto out;

	v_mtime = 0;
	v_atime = 0;
	if (nfsd4_create_is_exclusive(open-&gt;op_createmode)) {
		u32 *verifier = (u32 *)open-&gt;op_verf.data;

		/*
		 * Solaris 7 gets confused (bugid 4218508) if these have
		 * the high bit set, as do xfs filesystems without the
		 * "bigtime" feature. So just clear the high bits. If this
		 * is ever changed to use different attrs for storing the
		 * verifier, then do_open_lookup() will also need to be
		 * fixed accordingly.
		 */
		v_mtime = verifier[0] &amp; 0x7fffffff;
		v_atime = verifier[1] &amp; 0x7fffffff;
	}

	if (d_really_is_positive(child)) {
		status = nfs_ok;

		/* NFSv4 protocol requires change attributes even though
		 * no change happened.
		 */
		fh_fill_both_attrs(fhp);

		switch (open-&gt;op_createmode) {
		case NFS4_CREATE_UNCHECKED:
			if (!d_is_reg(child))
				break;

			/*
			 * In NFSv4, we don't want to truncate the file
			 * now. This would be wrong if the OPEN fails for
			 * some other reason. Furthermore, if the size is
			 * nonzero, we should ignore it according to spec!
			 */
			open-&gt;op_truncate = (iap-&gt;ia_valid &amp; ATTR_SIZE) &amp;&amp;
						!iap-&gt;ia_size;
			break;
		case NFS4_CREATE_GUARDED:
			status = nfserr_exist;
			break;
		case NFS4_CREATE_EXCLUSIVE:
			if (d_inode(child)-&gt;i_mtime.tv_sec == v_mtime &amp;&amp;
			    d_inode(child)-&gt;i_atime.tv_sec == v_atime &amp;&amp;
			    d_inode(child)-&gt;i_size == 0) {
				open-&gt;op_created = true;
				break;		/* subtle */
			}
			status = nfserr_exist;
			break;
		case NFS4_CREATE_EXCLUSIVE4_1:
			if (d_inode(child)-&gt;i_mtime.tv_sec == v_mtime &amp;&amp;
			    d_inode(child)-&gt;i_atime.tv_sec == v_atime &amp;&amp;
			    d_inode(child)-&gt;i_size == 0) {
				open-&gt;op_created = true;
				goto set_attr;	/* subtle */
			}
			status = nfserr_exist;
		}
		goto out;
	}

	if (!IS_POSIXACL(inode))
		iap-&gt;ia_mode &amp;= ~current_umask();

	fh_fill_pre_attrs(fhp);
	status = nfsd4_vfs_create(fhp, child, open);
	if (status != nfs_ok)
		goto out;
	open-&gt;op_created = true;
	fh_fill_post_attrs(fhp);

	/* A newly created file already has a file size of zero. */
	if ((iap-&gt;ia_valid &amp; ATTR_SIZE) &amp;&amp; (iap-&gt;ia_size == 0))
		iap-&gt;ia_valid &amp;= ~ATTR_SIZE;
	if (nfsd4_create_is_exclusive(open-&gt;op_createmode)) {
		iap-&gt;ia_valid = ATTR_MTIME | ATTR_ATIME |
				ATTR_MTIME_SET|ATTR_ATIME_SET;
		iap-&gt;ia_mtime.tv_sec = v_mtime;
		iap-&gt;ia_atime.tv_sec = v_atime;
		iap-&gt;ia_mtime.tv_nsec = 0;
		iap-&gt;ia_atime.tv_nsec = 0;
	}

set_attr:
	status = nfsd_create_setattr(rqstp, fhp, resfhp, &amp;attrs);

	if (attrs.na_labelerr)
		open-&gt;op_bmval[2] &amp;= ~FATTR4_WORD2_SECURITY_LABEL;
	if (attrs.na_aclerr)
		open-&gt;op_bmval[0] &amp;= ~FATTR4_WORD0_ACL;
out:
	inode_unlock(inode);
	nfsd_attrs_free(&amp;attrs);
	if (child &amp;&amp; !IS_ERR(child))
		dput(child);
	fh_drop_write(fhp);
	return status;
}

static __be32
do_open_lookup(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate, struct nfsd4_open *open, struct svc_fh **resfh)
{
	struct svc_fh *current_fh = &amp;cstate-&gt;current_fh;
	int accmode;
	__be32 status;

	*resfh = kmalloc(sizeof(struct svc_fh), GFP_KERNEL);
	if (!*resfh)
		return nfserr_jukebox;
	fh_init(*resfh, NFS4_FHSIZE);
	open-&gt;op_truncate = false;

	if (open-&gt;op_create) {
		/* FIXME: check session persistence and pnfs flags.
		 * The nfsv4.1 spec requires the following semantics:
		 *
		 * Persistent   | pNFS   | Server REQUIRED | Client Allowed
		 * Reply Cache  | server |                 |
		 * -------------+--------+-----------------+--------------------
		 * no           | no     | EXCLUSIVE4_1    | EXCLUSIVE4_1
		 *              |        |                 | (SHOULD)
		 *              |        | and EXCLUSIVE4  | or EXCLUSIVE4
		 *              |        |                 | (SHOULD NOT)
		 * no           | yes    | EXCLUSIVE4_1    | EXCLUSIVE4_1
		 * yes          | no     | GUARDED4        | GUARDED4
		 * yes          | yes    | GUARDED4        | GUARDED4
		 */

		current-&gt;fs-&gt;umask = open-&gt;op_umask;
		status = nfsd4_create_file(rqstp, current_fh, *resfh, open);
		current-&gt;fs-&gt;umask = 0;

		/*
		 * Following rfc 3530 14.2.16, and rfc 5661 18.16.4
		 * use the returned bitmask to indicate which attributes
		 * we used to store the verifier:
		 */
		if (nfsd4_create_is_exclusive(open-&gt;op_createmode) &amp;&amp; status == 0)
			open-&gt;op_bmval[1] |= (FATTR4_WORD1_TIME_ACCESS |
						FATTR4_WORD1_TIME_MODIFY);
	} else {
		status = nfsd_lookup(rqstp, current_fh,
				     open-&gt;op_fname, open-&gt;op_fnamelen, *resfh);
		if (!status)
			/* NFSv4 protocol requires change attributes even though
			 * no change happened.
			 */
			fh_fill_both_attrs(current_fh);
	}
	if (status)
		goto out;
	status = nfsd_check_obj_isreg(*resfh);
	if (status)
		goto out;

	nfsd4_set_open_owner_reply_cache(cstate, open, *resfh);
	accmode = NFSD_MAY_NOP;
	if (open-&gt;op_created ||
			open-&gt;op_claim_type == NFS4_OPEN_CLAIM_DELEGATE_CUR)
		accmode |= NFSD_MAY_OWNER_OVERRIDE;
	status = do_open_permission(rqstp, *resfh, open, accmode);
	set_change_info(&amp;open-&gt;op_cinfo, current_fh);
out:
	return status;
}

static __be32
do_open_fhandle(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate, struct nfsd4_open *open)
{
	struct svc_fh *current_fh = &amp;cstate-&gt;current_fh;
	__be32 status;
	int accmode = 0;

	/* We don't know the target directory, and therefore can not
	* set the change info
	*/

	memset(&amp;open-&gt;op_cinfo, 0, sizeof(struct nfsd4_change_info));

	nfsd4_set_open_owner_reply_cache(cstate, open, current_fh);

	open-&gt;op_truncate = (open-&gt;op_iattr.ia_valid &amp; ATTR_SIZE) &amp;&amp;
		(open-&gt;op_iattr.ia_size == 0);
	/*
	 * In the delegation case, the client is telling us about an
	 * open that it *already* performed locally, some time ago.  We
	 * should let it succeed now if possible.
	 *
	 * In the case of a CLAIM_FH open, on the other hand, the client
	 * may be counting on us to enforce permissions (the Linux 4.1
	 * client uses this for normal opens, for example).
	 */
	if (open-&gt;op_claim_type == NFS4_OPEN_CLAIM_DELEG_CUR_FH)
		accmode = NFSD_MAY_OWNER_OVERRIDE;

	status = do_open_permission(rqstp, current_fh, open, accmode);

	return status;
}

static void
copy_clientid(clientid_t *clid, struct nfsd4_session *session)
{
	struct nfsd4_sessionid *sid =
			(struct nfsd4_sessionid *)session-&gt;se_sessionid.data;

	clid-&gt;cl_boot = sid-&gt;clientid.cl_boot;
	clid-&gt;cl_id = sid-&gt;clientid.cl_id;
}

static __be32
nfsd4_open(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
	   union nfsd4_op_u *u)
{
	struct nfsd4_open *open = &amp;u-&gt;open;
	__be32 status;
	struct svc_fh *resfh = NULL;
	struct net *net = SVC_NET(rqstp);
	struct nfsd_net *nn = net_generic(net, nfsd_net_id);
	bool reclaim = false;

	dprintk("NFSD: nfsd4_open filename %.*s op_openowner %p\n",
		(int)open-&gt;op_fnamelen, open-&gt;op_fname,
		open-&gt;op_openowner);

	open-&gt;op_filp = NULL;
	open-&gt;op_rqstp = rqstp;

	/* This check required by spec. */
	if (open-&gt;op_create &amp;&amp; open-&gt;op_claim_type != NFS4_OPEN_CLAIM_NULL)
		return nfserr_inval;

	open-&gt;op_created = false;
	/*
	 * RFC5661 18.51.3
	 * Before RECLAIM_COMPLETE done, server should deny new lock
	 */
	if (nfsd4_has_session(cstate) &amp;&amp;
	    !test_bit(NFSD4_CLIENT_RECLAIM_COMPLETE, &amp;cstate-&gt;clp-&gt;cl_flags) &amp;&amp;
	    open-&gt;op_claim_type != NFS4_OPEN_CLAIM_PREVIOUS)
		return nfserr_grace;

	if (nfsd4_has_session(cstate))
		copy_clientid(&amp;open-&gt;op_clientid, cstate-&gt;session);

	/* check seqid for replay. set nfs4_owner */
	status = nfsd4_process_open1(cstate, open, nn);
	if (status == nfserr_replay_me) {
		struct nfs4_replay *rp = &amp;open-&gt;op_openowner-&gt;oo_owner.so_replay;
		fh_put(&amp;cstate-&gt;current_fh);
		fh_copy_shallow(&amp;cstate-&gt;current_fh.fh_handle,
				&amp;rp-&gt;rp_openfh);
		status = fh_verify(rqstp, &amp;cstate-&gt;current_fh, 0, NFSD_MAY_NOP);
		if (status)
			dprintk("nfsd4_open: replay failed"
				" restoring previous filehandle\n");
		else
			status = nfserr_replay_me;
	}
	if (status)
		goto out;
	if (open-&gt;op_xdr_error) {
		status = open-&gt;op_xdr_error;
		goto out;
	}

	status = nfsd4_check_open_attributes(rqstp, cstate, open);
	if (status)
		goto out;

	/* Openowner is now set, so sequence id will get bumped.  Now we need
	 * these checks before we do any creates: */
	status = nfserr_grace;
	if (opens_in_grace(net) &amp;&amp; open-&gt;op_claim_type != NFS4_OPEN_CLAIM_PREVIOUS)
		goto out;
	status = nfserr_no_grace;
	if (!opens_in_grace(net) &amp;&amp; open-&gt;op_claim_type == NFS4_OPEN_CLAIM_PREVIOUS)
		goto out;

	switch (open-&gt;op_claim_type) {
	case NFS4_OPEN_CLAIM_DELEGATE_CUR:
	case NFS4_OPEN_CLAIM_NULL:
		status = do_open_lookup(rqstp, cstate, open, &amp;resfh);
		if (status)
			goto out;
		break;
	case NFS4_OPEN_CLAIM_PREVIOUS:
		status = nfs4_check_open_reclaim(cstate-&gt;clp);
		if (status)
			goto out;
		open-&gt;op_openowner-&gt;oo_flags |= NFS4_OO_CONFIRMED;
		reclaim = true;
		fallthrough;
	case NFS4_OPEN_CLAIM_FH:
	case NFS4_OPEN_CLAIM_DELEG_CUR_FH:
		status = do_open_fhandle(rqstp, cstate, open);
		if (status)
			goto out;
		resfh = &amp;cstate-&gt;current_fh;
		break;
	case NFS4_OPEN_CLAIM_DELEG_PREV_FH:
	case NFS4_OPEN_CLAIM_DELEGATE_PREV:
		status = nfserr_notsupp;
		goto out;
	default:
		status = nfserr_inval;
		goto out;
	}

	status = nfsd4_process_open2(rqstp, resfh, open);
	if (status &amp;&amp; open-&gt;op_created)
		pr_warn("nfsd4_process_open2 failed to open newly-created file: status=%u\n",
			be32_to_cpu(status));
	if (reclaim &amp;&amp; !status)
		nn-&gt;somebody_reclaimed = true;
out:
	if (open-&gt;op_filp) {
		fput(open-&gt;op_filp);
		open-&gt;op_filp = NULL;
	}
	if (resfh &amp;&amp; resfh != &amp;cstate-&gt;current_fh) {
		fh_dup2(&amp;cstate-&gt;current_fh, resfh);
		fh_put(resfh);
		kfree(resfh);
	}
	nfsd4_cleanup_open_state(cstate, open);
	nfsd4_bump_seqid(cstate, status);
	return status;
}

/*
 * OPEN is the only seqid-mutating operation whose decoding can fail
 * with a seqid-mutating error (specifically, decoding of user names in
 * the attributes).  Therefore we have to do some processing to look up
 * the stateowner so that we can bump the seqid.
 */
static __be32 nfsd4_open_omfg(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate, struct nfsd4_op *op)
{
	struct nfsd4_open *open = &amp;op-&gt;u.open;

	if (!seqid_mutating_err(ntohl(op-&gt;status)))
		return op-&gt;status;
	if (nfsd4_has_session(cstate))
		return op-&gt;status;
	open-&gt;op_xdr_error = op-&gt;status;
	return nfsd4_open(rqstp, cstate, &amp;op-&gt;u);
}

/*
 * filehandle-manipulating ops.
 */
static __be32
nfsd4_getfh(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
	    union nfsd4_op_u *u)
{
	u-&gt;getfh = &amp;cstate-&gt;current_fh;
	return nfs_ok;
}

static __be32
nfsd4_putfh(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
	    union nfsd4_op_u *u)
{
	struct nfsd4_putfh *putfh = &amp;u-&gt;putfh;
	__be32 ret;

	fh_put(&amp;cstate-&gt;current_fh);
	cstate-&gt;current_fh.fh_handle.fh_size = putfh-&gt;pf_fhlen;
	memcpy(&amp;cstate-&gt;current_fh.fh_handle.fh_raw, putfh-&gt;pf_fhval,
	       putfh-&gt;pf_fhlen);
	ret = fh_verify(rqstp, &amp;cstate-&gt;current_fh, 0, NFSD_MAY_BYPASS_GSS);
#ifdef CONFIG_NFSD_V4_2_INTER_SSC
	if (ret == nfserr_stale &amp;&amp; putfh-&gt;no_verify) {
		SET_FH_FLAG(&amp;cstate-&gt;current_fh, NFSD4_FH_FOREIGN);
		ret = 0;
	}
#endif
	return ret;
}

static __be32
nfsd4_putrootfh(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
		union nfsd4_op_u *u)
{
	__be32 status;

	fh_put(&amp;cstate-&gt;current_fh);
	status = exp_pseudoroot(rqstp, &amp;cstate-&gt;current_fh);
	return status;
}

static __be32
nfsd4_restorefh(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
		union nfsd4_op_u *u)
{
	if (!cstate-&gt;save_fh.fh_dentry)
		return nfserr_restorefh;

	fh_dup2(&amp;cstate-&gt;current_fh, &amp;cstate-&gt;save_fh);
	if (HAS_CSTATE_FLAG(cstate, SAVED_STATE_ID_FLAG)) {
		memcpy(&amp;cstate-&gt;current_stateid, &amp;cstate-&gt;save_stateid, sizeof(stateid_t));
		SET_CSTATE_FLAG(cstate, CURRENT_STATE_ID_FLAG);
	}
	return nfs_ok;
}

static __be32
nfsd4_savefh(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
	     union nfsd4_op_u *u)
{
	fh_dup2(&amp;cstate-&gt;save_fh, &amp;cstate-&gt;current_fh);
	if (HAS_CSTATE_FLAG(cstate, CURRENT_STATE_ID_FLAG)) {
		memcpy(&amp;cstate-&gt;save_stateid, &amp;cstate-&gt;current_stateid, sizeof(stateid_t));
		SET_CSTATE_FLAG(cstate, SAVED_STATE_ID_FLAG);
	}
	return nfs_ok;
}

/*
 * misc nfsv4 ops
 */
static __be32
nfsd4_access(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
	     union nfsd4_op_u *u)
{
	struct nfsd4_access *access = &amp;u-&gt;access;
	u32 access_full;

	access_full = NFS3_ACCESS_FULL;
	if (cstate-&gt;minorversion &gt;= 2)
		access_full |= NFS4_ACCESS_XALIST | NFS4_ACCESS_XAREAD |
			       NFS4_ACCESS_XAWRITE;

	if (access-&gt;ac_req_access &amp; ~access_full)
		return nfserr_inval;

	access-&gt;ac_resp_access = access-&gt;ac_req_access;
	return nfsd_access(rqstp, &amp;cstate-&gt;current_fh, &amp;access-&gt;ac_resp_access,
			   &amp;access-&gt;ac_supported);
}

static void gen_boot_verifier(nfs4_verifier *verifier, struct net *net)
{
	__be32 *verf = (__be32 *)verifier-&gt;data;

	BUILD_BUG_ON(2*sizeof(*verf) != sizeof(verifier-&gt;data));

	nfsd_copy_write_verifier(verf, net_generic(net, nfsd_net_id));
}

static __be32
nfsd4_commit(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
	     union nfsd4_op_u *u)
{
	struct nfsd4_commit *commit = &amp;u-&gt;commit;

	return nfsd_commit(rqstp, &amp;cstate-&gt;current_fh, commit-&gt;co_offset,
			     commit-&gt;co_count,
			     (__be32 *)commit-&gt;co_verf.data);
}

static __be32
nfsd4_create(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
	     union nfsd4_op_u *u)
{
	struct nfsd4_create *create = &amp;u-&gt;create;
	struct nfsd_attrs attrs = {
		.na_iattr	= &amp;create-&gt;cr_iattr,
		.na_seclabel	= &amp;create-&gt;cr_label,
	};
	struct svc_fh resfh;
	__be32 status;
	dev_t rdev;

	fh_init(&amp;resfh, NFS4_FHSIZE);

	status = fh_verify(rqstp, &amp;cstate-&gt;current_fh, S_IFDIR, NFSD_MAY_NOP);
	if (status)
		return status;

	status = check_attr_support(rqstp, cstate, create-&gt;cr_bmval,
				    nfsd_attrmask);
	if (status)
		return status;

	status = nfsd4_acl_to_attr(create-&gt;cr_type, create-&gt;cr_acl, &amp;attrs);
	current-&gt;fs-&gt;umask = create-&gt;cr_umask;
	switch (create-&gt;cr_type) {
	case NF4LNK:
		status = nfsd_symlink(rqstp, &amp;cstate-&gt;current_fh,
				      create-&gt;cr_name, create-&gt;cr_namelen,
				      create-&gt;cr_data, &amp;attrs, &amp;resfh);
		break;

	case NF4BLK:
		status = nfserr_inval;
		rdev = MKDEV(create-&gt;cr_specdata1, create-&gt;cr_specdata2);
		if (MAJOR(rdev) != create-&gt;cr_specdata1 ||
		    MINOR(rdev) != create-&gt;cr_specdata2)
			goto out_umask;
		status = nfsd_create(rqstp, &amp;cstate-&gt;current_fh,
				     create-&gt;cr_name, create-&gt;cr_namelen,
				     &amp;attrs, S_IFBLK, rdev, &amp;resfh);
		break;

	case NF4CHR:
		status = nfserr_inval;
		rdev = MKDEV(create-&gt;cr_specdata1, create-&gt;cr_specdata2);
		if (MAJOR(rdev) != create-&gt;cr_specdata1 ||
		    MINOR(rdev) != create-&gt;cr_specdata2)
			goto out_umask;
		status = nfsd_create(rqstp, &amp;cstate-&gt;current_fh,
				     create-&gt;cr_name, create-&gt;cr_namelen,
				     &amp;attrs, S_IFCHR, rdev, &amp;resfh);
		break;

	case NF4SOCK:
		status = nfsd_create(rqstp, &amp;cstate-&gt;current_fh,
				     create-&gt;cr_name, create-&gt;cr_namelen,
				     &amp;attrs, S_IFSOCK, 0, &amp;resfh);
		break;

	case NF4FIFO:
		status = nfsd_create(rqstp, &amp;cstate-&gt;current_fh,
				     create-&gt;cr_name, create-&gt;cr_namelen,
				     &amp;attrs, S_IFIFO, 0, &amp;resfh);
		break;

	case NF4DIR:
		create-&gt;cr_iattr.ia_valid &amp;= ~ATTR_SIZE;
		status = nfsd_create(rqstp, &amp;cstate-&gt;current_fh,
				     create-&gt;cr_name, create-&gt;cr_namelen,
				     &amp;attrs, S_IFDIR, 0, &amp;resfh);
		break;

	default:
		status = nfserr_badtype;
	}

	if (status)
		goto out;

	if (attrs.na_labelerr)
		create-&gt;cr_bmval[2] &amp;= ~FATTR4_WORD2_SECURITY_LABEL;
	if (attrs.na_aclerr)
		create-&gt;cr_bmval[0] &amp;= ~FATTR4_WORD0_ACL;
	set_change_info(&amp;create-&gt;cr_cinfo, &amp;cstate-&gt;current_fh);
	fh_dup2(&amp;cstate-&gt;current_fh, &amp;resfh);
out:
	fh_put(&amp;resfh);
out_umask:
	current-&gt;fs-&gt;umask = 0;
	nfsd_attrs_free(&amp;attrs);
	return status;
}

static __be32
nfsd4_getattr(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
	      union nfsd4_op_u *u)
{
	struct nfsd4_getattr *getattr = &amp;u-&gt;getattr;
	__be32 status;

	status = fh_verify(rqstp, &amp;cstate-&gt;current_fh, 0, NFSD_MAY_NOP);
	if (status)
		return status;

	if (getattr-&gt;ga_bmval[1] &amp; NFSD_WRITEONLY_ATTRS_WORD1)
		return nfserr_inval;

	getattr-&gt;ga_bmval[0] &amp;= nfsd_suppattrs[cstate-&gt;minorversion][0];
	getattr-&gt;ga_bmval[1] &amp;= nfsd_suppattrs[cstate-&gt;minorversion][1];
	getattr-&gt;ga_bmval[2] &amp;= nfsd_suppattrs[cstate-&gt;minorversion][2];

	getattr-&gt;ga_fhp = &amp;cstate-&gt;current_fh;
	return nfs_ok;
}

static __be32
nfsd4_link(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
	   union nfsd4_op_u *u)
{
	struct nfsd4_link *link = &amp;u-&gt;link;
	__be32 status;

	status = nfsd_link(rqstp, &amp;cstate-&gt;current_fh,
			   link-&gt;li_name, link-&gt;li_namelen, &amp;cstate-&gt;save_fh);
	if (!status)
		set_change_info(&amp;link-&gt;li_cinfo, &amp;cstate-&gt;current_fh);
	return status;
}

static __be32 nfsd4_do_lookupp(struct svc_rqst *rqstp, struct svc_fh *fh)
{
	struct svc_fh tmp_fh;
	__be32 ret;

	fh_init(&amp;tmp_fh, NFS4_FHSIZE);
	ret = exp_pseudoroot(rqstp, &amp;tmp_fh);
	if (ret)
		return ret;
	if (tmp_fh.fh_dentry == fh-&gt;fh_dentry) {
		fh_put(&amp;tmp_fh);
		return nfserr_noent;
	}
	fh_put(&amp;tmp_fh);
	return nfsd_lookup(rqstp, fh, "..", 2, fh);
}

static __be32
nfsd4_lookupp(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
	      union nfsd4_op_u *u)
{
	return nfsd4_do_lookupp(rqstp, &amp;cstate-&gt;current_fh);
}

static __be32
nfsd4_lookup(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
	     union nfsd4_op_u *u)
{
	return nfsd_lookup(rqstp, &amp;cstate-&gt;current_fh,
			   u-&gt;lookup.lo_name, u-&gt;lookup.lo_len,
			   &amp;cstate-&gt;current_fh);
}

static __be32
nfsd4_read(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
	   union nfsd4_op_u *u)
{
	struct nfsd4_read *read = &amp;u-&gt;read;
	__be32 status;

	read-&gt;rd_nf = NULL;

	trace_nfsd_read_start(rqstp, &amp;cstate-&gt;current_fh,
			      read-&gt;rd_offset, read-&gt;rd_length);

	read-&gt;rd_length = min_t(u32, read-&gt;rd_length, svc_max_payload(rqstp));
	if (read-&gt;rd_offset &gt; (u64)OFFSET_MAX)
		read-&gt;rd_offset = (u64)OFFSET_MAX;
	if (read-&gt;rd_offset + read-&gt;rd_length &gt; (u64)OFFSET_MAX)
		read-&gt;rd_length = (u64)OFFSET_MAX - read-&gt;rd_offset;

	/*
	 * If we do a zero copy read, then a client will see read data
	 * that reflects the state of the file *after* performing the
	 * following compound.
	 *
	 * To ensure proper ordering, we therefore turn off zero copy if
	 * the client wants us to do more in this compound:
	 */
	if (!nfsd4_last_compound_op(rqstp))
		__clear_bit(RQ_SPLICE_OK, &amp;rqstp-&gt;rq_flags);

	/* check stateid */
	status = nfs4_preprocess_stateid_op(rqstp, cstate, &amp;cstate-&gt;current_fh,
					&amp;read-&gt;rd_stateid, RD_STATE,
					&amp;read-&gt;rd_nf, NULL);
	if (status) {
		dprintk("NFSD: nfsd4_read: couldn't process stateid!\n");
		goto out;
	}
	status = nfs_ok;
out:
	read-&gt;rd_rqstp = rqstp;
	read-&gt;rd_fhp = &amp;cstate-&gt;current_fh;
	return status;
}


static void
nfsd4_read_release(union nfsd4_op_u *u)
{
	if (u-&gt;read.rd_nf)
		nfsd_file_put(u-&gt;read.rd_nf);
	trace_nfsd_read_done(u-&gt;read.rd_rqstp, u-&gt;read.rd_fhp,
			     u-&gt;read.rd_offset, u-&gt;read.rd_length);
}

static __be32
nfsd4_readdir(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
	      union nfsd4_op_u *u)
{
	struct nfsd4_readdir *readdir = &amp;u-&gt;readdir;
	u64 cookie = readdir-&gt;rd_cookie;
	static const nfs4_verifier zeroverf;

	/* no need to check permission - this will be done in nfsd_readdir() */

	if (readdir-&gt;rd_bmval[1] &amp; NFSD_WRITEONLY_ATTRS_WORD1)
		return nfserr_inval;

	readdir-&gt;rd_bmval[0] &amp;= nfsd_suppattrs[cstate-&gt;minorversion][0];
	readdir-&gt;rd_bmval[1] &amp;= nfsd_suppattrs[cstate-&gt;minorversion][1];
	readdir-&gt;rd_bmval[2] &amp;= nfsd_suppattrs[cstate-&gt;minorversion][2];

	if ((cookie == 1) || (cookie == 2) ||
	    (cookie == 0 &amp;&amp; memcmp(readdir-&gt;rd_verf.data, zeroverf.data, NFS4_VERIFIER_SIZE)))
		return nfserr_bad_cookie;

	readdir-&gt;rd_rqstp = rqstp;
	readdir-&gt;rd_fhp = &amp;cstate-&gt;current_fh;
	return nfs_ok;
}

static __be32
nfsd4_readlink(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
	       union nfsd4_op_u *u)
{
	u-&gt;readlink.rl_rqstp = rqstp;
	u-&gt;readlink.rl_fhp = &amp;cstate-&gt;current_fh;
	return nfs_ok;
}

static __be32
nfsd4_remove(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
	     union nfsd4_op_u *u)
{
	struct nfsd4_remove *remove = &amp;u-&gt;remove;
	__be32 status;

	if (opens_in_grace(SVC_NET(rqstp)))
		return nfserr_grace;
	status = nfsd_unlink(rqstp, &amp;cstate-&gt;current_fh, 0,
			     remove-&gt;rm_name, remove-&gt;rm_namelen);
	if (!status)
		set_change_info(&amp;remove-&gt;rm_cinfo, &amp;cstate-&gt;current_fh);
	return status;
}

static __be32
nfsd4_rename(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
	     union nfsd4_op_u *u)
{
	struct nfsd4_rename *rename = &amp;u-&gt;rename;
	__be32 status;

	if (opens_in_grace(SVC_NET(rqstp)))
		return nfserr_grace;
	status = nfsd_rename(rqstp, &amp;cstate-&gt;save_fh, rename-&gt;rn_sname,
			     rename-&gt;rn_snamelen, &amp;cstate-&gt;current_fh,
			     rename-&gt;rn_tname, rename-&gt;rn_tnamelen);
	if (status)
		return status;
	set_change_info(&amp;rename-&gt;rn_sinfo, &amp;cstate-&gt;current_fh);
	set_change_info(&amp;rename-&gt;rn_tinfo, &amp;cstate-&gt;save_fh);
	return nfs_ok;
}

static __be32
nfsd4_secinfo(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
	      union nfsd4_op_u *u)
{
	struct nfsd4_secinfo *secinfo = &amp;u-&gt;secinfo;
	struct svc_export *exp;
	struct dentry *dentry;
	__be32 err;

	err = fh_verify(rqstp, &amp;cstate-&gt;current_fh, S_IFDIR, NFSD_MAY_EXEC);
	if (err)
		return err;
	err = nfsd_lookup_dentry(rqstp, &amp;cstate-&gt;current_fh,
				    secinfo-&gt;si_name, secinfo-&gt;si_namelen,
				    &amp;exp, &amp;dentry);
	if (err)
		return err;
	if (d_really_is_negative(dentry)) {
		exp_put(exp);
		err = nfserr_noent;
	} else
		secinfo-&gt;si_exp = exp;
	dput(dentry);
	if (cstate-&gt;minorversion)
		/* See rfc 5661 section 2.6.3.1.1.8 */
		fh_put(&amp;cstate-&gt;current_fh);
	return err;
}

static __be32
nfsd4_secinfo_no_name(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
		union nfsd4_op_u *u)
{
	__be32 err;

	switch (u-&gt;secinfo_no_name.sin_style) {
	case NFS4_SECINFO_STYLE4_CURRENT_FH:
		break;
	case NFS4_SECINFO_STYLE4_PARENT:
		err = nfsd4_do_lookupp(rqstp, &amp;cstate-&gt;current_fh);
		if (err)
			return err;
		break;
	default:
		return nfserr_inval;
	}

	u-&gt;secinfo_no_name.sin_exp = exp_get(cstate-&gt;current_fh.fh_export);
	fh_put(&amp;cstate-&gt;current_fh);
	return nfs_ok;
}

static void
nfsd4_secinfo_release(union nfsd4_op_u *u)
{
	if (u-&gt;secinfo.si_exp)
		exp_put(u-&gt;secinfo.si_exp);
}

static void
nfsd4_secinfo_no_name_release(union nfsd4_op_u *u)
{
	if (u-&gt;secinfo_no_name.sin_exp)
		exp_put(u-&gt;secinfo_no_name.sin_exp);
}

static __be32
nfsd4_setattr(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
	      union nfsd4_op_u *u)
{
	struct nfsd4_setattr *setattr = &amp;u-&gt;setattr;
	struct nfsd_attrs attrs = {
		.na_iattr	= &amp;setattr-&gt;sa_iattr,
		.na_seclabel	= &amp;setattr-&gt;sa_label,
	};
	struct inode *inode;
	__be32 status = nfs_ok;
	int err;

	if (setattr-&gt;sa_iattr.ia_valid &amp; ATTR_SIZE) {
		status = nfs4_preprocess_stateid_op(rqstp, cstate,
				&amp;cstate-&gt;current_fh, &amp;setattr-&gt;sa_stateid,
				WR_STATE, NULL, NULL);
		if (status) {
			dprintk("NFSD: nfsd4_setattr: couldn't process stateid!\n");
			return status;
		}
	}
	err = fh_want_write(&amp;cstate-&gt;current_fh);
	if (err)
		return nfserrno(err);
	status = nfs_ok;

	status = check_attr_support(rqstp, cstate, setattr-&gt;sa_bmval,
				    nfsd_attrmask);
	if (status)
		goto out;

	inode = cstate-&gt;current_fh.fh_dentry-&gt;d_inode;
	status = nfsd4_acl_to_attr(S_ISDIR(inode-&gt;i_mode) ? NF4DIR : NF4REG,
				   setattr-&gt;sa_acl, &amp;attrs);

	if (status)
		goto out;
	status = nfsd_setattr(rqstp, &amp;cstate-&gt;current_fh, &amp;attrs,
				0, (time64_t)0);
	if (!status)
		status = nfserrno(attrs.na_labelerr);
out:
	nfsd_attrs_free(&amp;attrs);
	fh_drop_write(&amp;cstate-&gt;current_fh);
	return status;
}

static __be32
nfsd4_write(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
	    union nfsd4_op_u *u)
{
	struct nfsd4_write *write = &amp;u-&gt;write;
	stateid_t *stateid = &amp;write-&gt;wr_stateid;
	struct nfsd_file *nf = NULL;
	__be32 status = nfs_ok;
	unsigned long cnt;
	int nvecs;

	if (write-&gt;wr_offset &gt; (u64)OFFSET_MAX ||
	    write-&gt;wr_offset + write-&gt;wr_buflen &gt; (u64)OFFSET_MAX)
		return nfserr_fbig;

	cnt = write-&gt;wr_buflen;
	trace_nfsd_write_start(rqstp, &amp;cstate-&gt;current_fh,
			       write-&gt;wr_offset, cnt);
	status = nfs4_preprocess_stateid_op(rqstp, cstate, &amp;cstate-&gt;current_fh,
						stateid, WR_STATE, &amp;nf, NULL);
	if (status) {
		dprintk("NFSD: nfsd4_write: couldn't process stateid!\n");
		return status;
	}

	write-&gt;wr_how_written = write-&gt;wr_stable_how;

	nvecs = svc_fill_write_vector(rqstp, &amp;write-&gt;wr_payload);
	WARN_ON_ONCE(nvecs &gt; ARRAY_SIZE(rqstp-&gt;rq_vec));

	status = nfsd_vfs_write(rqstp, &amp;cstate-&gt;current_fh, nf,
				write-&gt;wr_offset, rqstp-&gt;rq_vec, nvecs, &amp;cnt,
				write-&gt;wr_how_written,
				(__be32 *)write-&gt;wr_verifier.data);
	nfsd_file_put(nf);

	write-&gt;wr_bytes_written = cnt;
	trace_nfsd_write_done(rqstp, &amp;cstate-&gt;current_fh,
			      write-&gt;wr_offset, cnt);
	return status;
}

static __be32
nfsd4_verify_copy(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
		  stateid_t *src_stateid, struct nfsd_file **src,
		  stateid_t *dst_stateid, struct nfsd_file **dst)
{
	__be32 status;

	if (!cstate-&gt;save_fh.fh_dentry)
		return nfserr_nofilehandle;

	status = nfs4_preprocess_stateid_op(rqstp, cstate, &amp;cstate-&gt;save_fh,
					    src_stateid, RD_STATE, src, NULL);
	if (status) {
		dprintk("NFSD: %s: couldn't process src stateid!\n", __func__);
		goto out;
	}

	status = nfs4_preprocess_stateid_op(rqstp, cstate, &amp;cstate-&gt;current_fh,
					    dst_stateid, WR_STATE, dst, NULL);
	if (status) {
		dprintk("NFSD: %s: couldn't process dst stateid!\n", __func__);
		goto out_put_src;
	}

	/* fix up for NFS-specific error code */
	if (!S_ISREG(file_inode((*src)-&gt;nf_file)-&gt;i_mode) ||
	    !S_ISREG(file_inode((*dst)-&gt;nf_file)-&gt;i_mode)) {
		status = nfserr_wrong_type;
		goto out_put_dst;
	}

out:
	return status;
out_put_dst:
	nfsd_file_put(*dst);
out_put_src:
	nfsd_file_put(*src);
	goto out;
}

static __be32
nfsd4_clone(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
		union nfsd4_op_u *u)
{
	struct nfsd4_clone *clone = &amp;u-&gt;clone;
	struct nfsd_file *src, *dst;
	__be32 status;

	status = nfsd4_verify_copy(rqstp, cstate, &amp;clone-&gt;cl_src_stateid, &amp;src,
				   &amp;clone-&gt;cl_dst_stateid, &amp;dst);
	if (status)
		goto out;

	status = nfsd4_clone_file_range(rqstp, src, clone-&gt;cl_src_pos,
			dst, clone-&gt;cl_dst_pos, clone-&gt;cl_count,
			EX_ISSYNC(cstate-&gt;current_fh.fh_export));

	nfsd_file_put(dst);
	nfsd_file_put(src);
out:
	return status;
}

static void nfs4_put_copy(struct nfsd4_copy *copy)
{
	if (!refcount_dec_and_test(&amp;copy-&gt;refcount))
		return;
	kfree(copy-&gt;cp_src);
	kfree(copy);
}

static void nfsd4_stop_copy(struct nfsd4_copy *copy)
{
	if (!test_and_set_bit(NFSD4_COPY_F_STOPPED, &amp;copy-&gt;cp_flags))
		kthread_stop(copy-&gt;copy_task);
	nfs4_put_copy(copy);
}

static struct nfsd4_copy *nfsd4_get_copy(struct nfs4_client *clp)
{
	struct nfsd4_copy *copy = NULL;

	spin_lock(&amp;clp-&gt;async_lock);
	if (!list_empty(&amp;clp-&gt;async_copies)) {
		copy = list_first_entry(&amp;clp-&gt;async_copies, struct nfsd4_copy,
					copies);
		refcount_inc(&amp;copy-&gt;refcount);
	}
	spin_unlock(&amp;clp-&gt;async_lock);
	return copy;
}

void nfsd4_shutdown_copy(struct nfs4_client *clp)
{
	struct nfsd4_copy *copy;

	while ((copy = nfsd4_get_copy(clp)) != NULL)
		nfsd4_stop_copy(copy);
}
#ifdef CONFIG_NFSD_V4_2_INTER_SSC

extern struct file *nfs42_ssc_open(struct vfsmount *ss_mnt,
				   struct nfs_fh *src_fh,
				   nfs4_stateid *stateid);
extern void nfs42_ssc_close(struct file *filep);

extern void nfs_sb_deactive(struct super_block *sb);

#define NFSD42_INTERSSC_MOUNTOPS "vers=4.2,addr=%s,sec=sys"

/*
 * setup a work entry in the ssc delayed unmount list.
 */
static __be32 nfsd4_ssc_setup_dul(struct nfsd_net *nn, char *ipaddr,
		struct nfsd4_ssc_umount_item **retwork, struct vfsmount **ss_mnt)
{
	struct nfsd4_ssc_umount_item *ni = NULL;
	struct nfsd4_ssc_umount_item *work = NULL;
	struct nfsd4_ssc_umount_item *tmp;
	DEFINE_WAIT(wait);

	*ss_mnt = NULL;
	*retwork = NULL;
	work = kzalloc(sizeof(*work), GFP_KERNEL);
try_again:
	spin_lock(&amp;nn-&gt;nfsd_ssc_lock);
	list_for_each_entry_safe(ni, tmp, &amp;nn-&gt;nfsd_ssc_mount_list, nsui_list) {
		if (strncmp(ni-&gt;nsui_ipaddr, ipaddr, sizeof(ni-&gt;nsui_ipaddr)))
			continue;
		/* found a match */
		if (ni-&gt;nsui_busy) {
			/*  wait - and try again */
			prepare_to_wait(&amp;nn-&gt;nfsd_ssc_waitq, &amp;wait,
				TASK_INTERRUPTIBLE);
			spin_unlock(&amp;nn-&gt;nfsd_ssc_lock);

			/* allow 20secs for mount/unmount for now - revisit */
			if (signal_pending(current) ||
					(schedule_timeout(20*HZ) == 0)) {
				kfree(work);
				return nfserr_eagain;
			}
			finish_wait(&amp;nn-&gt;nfsd_ssc_waitq, &amp;wait);
			goto try_again;
		}
		*ss_mnt = ni-&gt;nsui_vfsmount;
		refcount_inc(&amp;ni-&gt;nsui_refcnt);
		spin_unlock(&amp;nn-&gt;nfsd_ssc_lock);
		kfree(work);

		/* return vfsmount in ss_mnt */
		return 0;
	}
	if (work) {
		strlcpy(work-&gt;nsui_ipaddr, ipaddr, sizeof(work-&gt;nsui_ipaddr) - 1);
		refcount_set(&amp;work-&gt;nsui_refcnt, 2);
		work-&gt;nsui_busy = true;
		list_add_tail(&amp;work-&gt;nsui_list, &amp;nn-&gt;nfsd_ssc_mount_list);
		*retwork = work;
	}
	spin_unlock(&amp;nn-&gt;nfsd_ssc_lock);
	return 0;
}

static void nfsd4_ssc_update_dul_work(struct nfsd_net *nn,
		struct nfsd4_ssc_umount_item *work, struct vfsmount *ss_mnt)
{
	/* set nsui_vfsmount, clear busy flag and wakeup waiters */
	spin_lock(&amp;nn-&gt;nfsd_ssc_lock);
	work-&gt;nsui_vfsmount = ss_mnt;
	work-&gt;nsui_busy = false;
	wake_up_all(&amp;nn-&gt;nfsd_ssc_waitq);
	spin_unlock(&amp;nn-&gt;nfsd_ssc_lock);
}

static void nfsd4_ssc_cancel_dul_work(struct nfsd_net *nn,
		struct nfsd4_ssc_umount_item *work)
{
	spin_lock(&amp;nn-&gt;nfsd_ssc_lock);
	list_del(&amp;work-&gt;nsui_list);
	wake_up_all(&amp;nn-&gt;nfsd_ssc_waitq);
	spin_unlock(&amp;nn-&gt;nfsd_ssc_lock);
	kfree(work);
}

/*
 * Support one copy source server for now.
 */
static __be32
nfsd4_interssc_connect(struct nl4_server *nss, struct svc_rqst *rqstp,
		       struct vfsmount **mount)
{
	struct file_system_type *type;
	struct vfsmount *ss_mnt;
	struct nfs42_netaddr *naddr;
	struct sockaddr_storage tmp_addr;
	size_t tmp_addrlen, match_netid_len = 3;
	char *startsep = "", *endsep = "", *match_netid = "tcp";
	char *ipaddr, *dev_name, *raw_data;
	int len, raw_len;
	__be32 status = nfserr_inval;
	struct nfsd4_ssc_umount_item *work = NULL;
	struct nfsd_net *nn = net_generic(SVC_NET(rqstp), nfsd_net_id);

	naddr = &amp;nss-&gt;u.nl4_addr;
	tmp_addrlen = rpc_uaddr2sockaddr(SVC_NET(rqstp), naddr-&gt;addr,
					 naddr-&gt;addr_len,
					 (struct sockaddr *)&amp;tmp_addr,
					 sizeof(tmp_addr));
	if (tmp_addrlen == 0)
		goto out_err;

	if (tmp_addr.ss_family == AF_INET6) {
		startsep = "[";
		endsep = "]";
		match_netid = "tcp6";
		match_netid_len = 4;
	}

	if (naddr-&gt;netid_len != match_netid_len ||
		strncmp(naddr-&gt;netid, match_netid, naddr-&gt;netid_len))
		goto out_err;

	/* Construct the raw data for the vfs_kern_mount call */
	len = RPC_MAX_ADDRBUFLEN + 1;
	ipaddr = kzalloc(len, GFP_KERNEL);
	if (!ipaddr)
		goto out_err;

	rpc_ntop((struct sockaddr *)&amp;tmp_addr, ipaddr, len);

	/* 2 for ipv6 endsep and startsep. 3 for ":/" and trailing '/0'*/

	raw_len = strlen(NFSD42_INTERSSC_MOUNTOPS) + strlen(ipaddr);
	raw_data = kzalloc(raw_len, GFP_KERNEL);
	if (!raw_data)
		goto out_free_ipaddr;

	snprintf(raw_data, raw_len, NFSD42_INTERSSC_MOUNTOPS, ipaddr);

	status = nfserr_nodev;
	type = get_fs_type("nfs");
	if (!type)
		goto out_free_rawdata;

	/* Set the server:<export> for the vfs_kern_mount call */
	dev_name = kzalloc(len + 5, GFP_KERNEL);
	if (!dev_name)
		goto out_free_rawdata;
	snprintf(dev_name, len + 5, "%s%s%s:/", startsep, ipaddr, endsep);

	status = nfsd4_ssc_setup_dul(nn, ipaddr, &amp;work, &amp;ss_mnt);
	if (status)
		goto out_free_devname;
	if (ss_mnt)
		goto out_done;

	/* Use an 'internal' mount: SB_KERNMOUNT -&gt; MNT_INTERNAL */
	ss_mnt = vfs_kern_mount(type, SB_KERNMOUNT, dev_name, raw_data);
	module_put(type-&gt;owner);
	if (IS_ERR(ss_mnt)) {
		status = nfserr_nodev;
		if (work)
			nfsd4_ssc_cancel_dul_work(nn, work);
		goto out_free_devname;
	}
	if (work)
		nfsd4_ssc_update_dul_work(nn, work, ss_mnt);
out_done:
	status = 0;
	*mount = ss_mnt;

out_free_devname:
	kfree(dev_name);
out_free_rawdata:
	kfree(raw_data);
out_free_ipaddr:
	kfree(ipaddr);
out_err:
	return status;
}

static void
nfsd4_interssc_disconnect(struct vfsmount *ss_mnt)
{
	nfs_do_sb_deactive(ss_mnt-&gt;mnt_sb);
	mntput(ss_mnt);
}

/*
 * Verify COPY destination stateid.
 *
 * Connect to the source server with NFSv4.1.
 * Create the source struct file for nfsd_copy_range.
 * Called with COPY cstate:
 *    SAVED_FH: source filehandle
 *    CURRENT_FH: destination filehandle
 */
static __be32
nfsd4_setup_inter_ssc(struct svc_rqst *rqstp,
		      struct nfsd4_compound_state *cstate,
		      struct nfsd4_copy *copy, struct vfsmount **mount)
{
	struct svc_fh *s_fh = NULL;
	stateid_t *s_stid = &amp;copy-&gt;cp_src_stateid;
	__be32 status = nfserr_inval;

	/* Verify the destination stateid and set dst struct file*/
	status = nfs4_preprocess_stateid_op(rqstp, cstate, &amp;cstate-&gt;current_fh,
					    &amp;copy-&gt;cp_dst_stateid,
					    WR_STATE, &amp;copy-&gt;nf_dst, NULL);
	if (status)
		goto out;

	status = nfsd4_interssc_connect(copy-&gt;cp_src, rqstp, mount);
	if (status)
		goto out;

	s_fh = &amp;cstate-&gt;save_fh;

	copy-&gt;c_fh.size = s_fh-&gt;fh_handle.fh_size;
	memcpy(copy-&gt;c_fh.data, &amp;s_fh-&gt;fh_handle.fh_raw, copy-&gt;c_fh.size);
	copy-&gt;stateid.seqid = cpu_to_be32(s_stid-&gt;si_generation);
	memcpy(copy-&gt;stateid.other, (void *)&amp;s_stid-&gt;si_opaque,
	       sizeof(stateid_opaque_t));

	status = 0;
out:
	return status;
}

static void
nfsd4_cleanup_inter_ssc(struct vfsmount *ss_mnt, struct file *filp,
			struct nfsd_file *dst)
{
	bool found = false;
	long timeout;
	struct nfsd4_ssc_umount_item *tmp;
	struct nfsd4_ssc_umount_item *ni = NULL;
	struct nfsd_net *nn = net_generic(dst-&gt;nf_net, nfsd_net_id);

	nfs42_ssc_close(filp);
	nfsd_file_put(dst);
	fput(filp);

	if (!nn) {
		mntput(ss_mnt);
		return;
	}
	spin_lock(&amp;nn-&gt;nfsd_ssc_lock);
	timeout = msecs_to_jiffies(nfsd4_ssc_umount_timeout);
	list_for_each_entry_safe(ni, tmp, &amp;nn-&gt;nfsd_ssc_mount_list, nsui_list) {
		if (ni-&gt;nsui_vfsmount-&gt;mnt_sb == ss_mnt-&gt;mnt_sb) {
			list_del(&amp;ni-&gt;nsui_list);
			/*
			 * vfsmount can be shared by multiple exports,
			 * decrement refcnt. If the count drops to 1 it
			 * will be unmounted when nsui_expire expires.
			 */
			refcount_dec(&amp;ni-&gt;nsui_refcnt);
			ni-&gt;nsui_expire = jiffies + timeout;
			list_add_tail(&amp;ni-&gt;nsui_list, &amp;nn-&gt;nfsd_ssc_mount_list);
			found = true;
			break;
		}
	}
	spin_unlock(&amp;nn-&gt;nfsd_ssc_lock);
	if (!found) {
		mntput(ss_mnt);
		return;
	}
}

#else /* CONFIG_NFSD_V4_2_INTER_SSC */

static __be32
nfsd4_setup_inter_ssc(struct svc_rqst *rqstp,
		      struct nfsd4_compound_state *cstate,
		      struct nfsd4_copy *copy,
		      struct vfsmount **mount)
{
	*mount = NULL;
	return nfserr_inval;
}

static void
nfsd4_cleanup_inter_ssc(struct vfsmount *ss_mnt, struct file *filp,
			struct nfsd_file *dst)
{
}

static void
nfsd4_interssc_disconnect(struct vfsmount *ss_mnt)
{
}

static struct file *nfs42_ssc_open(struct vfsmount *ss_mnt,
				   struct nfs_fh *src_fh,
				   nfs4_stateid *stateid)
{
	return NULL;
}
#endif /* CONFIG_NFSD_V4_2_INTER_SSC */

static __be32
nfsd4_setup_intra_ssc(struct svc_rqst *rqstp,
		      struct nfsd4_compound_state *cstate,
		      struct nfsd4_copy *copy)
{
	return nfsd4_verify_copy(rqstp, cstate, &amp;copy-&gt;cp_src_stateid,
				 &amp;copy-&gt;nf_src, &amp;copy-&gt;cp_dst_stateid,
				 &amp;copy-&gt;nf_dst);
}

static void
nfsd4_cleanup_intra_ssc(struct nfsd_file *src, struct nfsd_file *dst)
{
	nfsd_file_put(src);
	nfsd_file_put(dst);
}

static void nfsd4_cb_offload_release(struct nfsd4_callback *cb)
{
	struct nfsd4_cb_offload *cbo =
		container_of(cb, struct nfsd4_cb_offload, co_cb);

	kfree(cbo);
}

static int nfsd4_cb_offload_done(struct nfsd4_callback *cb,
				 struct rpc_task *task)
{
	return 1;
}

static const struct nfsd4_callback_ops nfsd4_cb_offload_ops = {
	.release = nfsd4_cb_offload_release,
	.done = nfsd4_cb_offload_done
};

static void nfsd4_init_copy_res(struct nfsd4_copy *copy, bool sync)
{
	copy-&gt;cp_res.wr_stable_how =
		test_bit(NFSD4_COPY_F_COMMITTED, &amp;copy-&gt;cp_flags) ?
			NFS_FILE_SYNC : NFS_UNSTABLE;
	nfsd4_copy_set_sync(copy, sync);
	gen_boot_verifier(&amp;copy-&gt;cp_res.wr_verifier, copy-&gt;cp_clp-&gt;net);
}

static ssize_t _nfsd_copy_file_range(struct nfsd4_copy *copy,
				     struct file *dst,
				     struct file *src)
{
	errseq_t since;
	ssize_t bytes_copied = 0;
	u64 bytes_total = copy-&gt;cp_count;
	u64 src_pos = copy-&gt;cp_src_pos;
	u64 dst_pos = copy-&gt;cp_dst_pos;
	int status;

	/* See RFC 7862 p.67: */
	if (bytes_total == 0)
		bytes_total = ULLONG_MAX;
	do {
		if (kthread_should_stop())
			break;
		bytes_copied = nfsd_copy_file_range(src, src_pos, dst, dst_pos,
						    bytes_total);
		if (bytes_copied &lt;= 0)
			break;
		bytes_total -= bytes_copied;
		copy-&gt;cp_res.wr_bytes_written += bytes_copied;
		src_pos += bytes_copied;
		dst_pos += bytes_copied;
	} while (bytes_total &gt; 0 &amp;&amp; nfsd4_copy_is_async(copy));
	/* for a non-zero asynchronous copy do a commit of data */
	if (nfsd4_copy_is_async(copy) &amp;&amp; copy-&gt;cp_res.wr_bytes_written &gt; 0) {
		since = READ_ONCE(dst-&gt;f_wb_err);
		status = vfs_fsync_range(dst, copy-&gt;cp_dst_pos,
					 copy-&gt;cp_res.wr_bytes_written, 0);
		if (!status)
			status = filemap_check_wb_err(dst-&gt;f_mapping, since);
		if (!status)
			set_bit(NFSD4_COPY_F_COMMITTED, &amp;copy-&gt;cp_flags);
	}
	return bytes_copied;
}

static __be32 nfsd4_do_copy(struct nfsd4_copy *copy,
			    struct file *src, struct file *dst,
			    bool sync)
{
	__be32 status;
	ssize_t bytes;

	bytes = _nfsd_copy_file_range(copy, dst, src);

	/* for async copy, we ignore the error, client can always retry
	 * to get the error
	 */
	if (bytes &lt; 0 &amp;&amp; !copy-&gt;cp_res.wr_bytes_written)
		status = nfserrno(bytes);
	else {
		nfsd4_init_copy_res(copy, sync);
		status = nfs_ok;
	}
	return status;
}

static void dup_copy_fields(struct nfsd4_copy *src, struct nfsd4_copy *dst)
{
	dst-&gt;cp_src_pos = src-&gt;cp_src_pos;
	dst-&gt;cp_dst_pos = src-&gt;cp_dst_pos;
	dst-&gt;cp_count = src-&gt;cp_count;
	dst-&gt;cp_flags = src-&gt;cp_flags;
	memcpy(&amp;dst-&gt;cp_res, &amp;src-&gt;cp_res, sizeof(src-&gt;cp_res));
	memcpy(&amp;dst-&gt;fh, &amp;src-&gt;fh, sizeof(src-&gt;fh));
	dst-&gt;cp_clp = src-&gt;cp_clp;
	dst-&gt;nf_dst = nfsd_file_get(src-&gt;nf_dst);
	/* for inter, nf_src doesn't exist yet */
	if (!nfsd4_ssc_is_inter(src))
		dst-&gt;nf_src = nfsd_file_get(src-&gt;nf_src);

	memcpy(&amp;dst-&gt;cp_stateid, &amp;src-&gt;cp_stateid, sizeof(src-&gt;cp_stateid));
	memcpy(dst-&gt;cp_src, src-&gt;cp_src, sizeof(struct nl4_server));
	memcpy(&amp;dst-&gt;stateid, &amp;src-&gt;stateid, sizeof(src-&gt;stateid));
	memcpy(&amp;dst-&gt;c_fh, &amp;src-&gt;c_fh, sizeof(src-&gt;c_fh));
	dst-&gt;ss_mnt = src-&gt;ss_mnt;
}

static void cleanup_async_copy(struct nfsd4_copy *copy)
{
	nfs4_free_copy_state(copy);
	nfsd_file_put(copy-&gt;nf_dst);
	if (!nfsd4_ssc_is_inter(copy))
		nfsd_file_put(copy-&gt;nf_src);
	spin_lock(&amp;copy-&gt;cp_clp-&gt;async_lock);
	list_del(&amp;copy-&gt;copies);
	spin_unlock(&amp;copy-&gt;cp_clp-&gt;async_lock);
	nfs4_put_copy(copy);
}

static void nfsd4_send_cb_offload(struct nfsd4_copy *copy, __be32 nfserr)
{
	struct nfsd4_cb_offload *cbo;

	cbo = kzalloc(sizeof(*cbo), GFP_KERNEL);
	if (!cbo)
		return;

	memcpy(&amp;cbo-&gt;co_res, &amp;copy-&gt;cp_res, sizeof(copy-&gt;cp_res));
	memcpy(&amp;cbo-&gt;co_fh, &amp;copy-&gt;fh, sizeof(copy-&gt;fh));
	cbo-&gt;co_nfserr = nfserr;

	nfsd4_init_cb(&amp;cbo-&gt;co_cb, copy-&gt;cp_clp, &amp;nfsd4_cb_offload_ops,
		      NFSPROC4_CLNT_CB_OFFLOAD);
	trace_nfsd_cb_offload(copy-&gt;cp_clp, &amp;cbo-&gt;co_res.cb_stateid,
			      &amp;cbo-&gt;co_fh, copy-&gt;cp_count, nfserr);
	nfsd4_run_cb(&amp;cbo-&gt;co_cb);
}

/**
 * nfsd4_do_async_copy - kthread function for background server-side COPY
 * @data: arguments for COPY operation
 *
 * Return values:
 *   %0: Copy operation is done.
 */
static int nfsd4_do_async_copy(void *data)
{
	struct nfsd4_copy *copy = (struct nfsd4_copy *)data;
	__be32 nfserr;

	if (nfsd4_ssc_is_inter(copy)) {
		struct file *filp;

		filp = nfs42_ssc_open(copy-&gt;ss_mnt, &amp;copy-&gt;c_fh,
				      &amp;copy-&gt;stateid);
		if (IS_ERR(filp)) {
			nfserr = nfserr_offload_denied;
			nfsd4_interssc_disconnect(copy-&gt;ss_mnt);
			goto do_callback;
		}
		nfserr = nfsd4_do_copy(copy, filp, copy-&gt;nf_dst-&gt;nf_file,
				       false);
		nfsd4_cleanup_inter_ssc(copy-&gt;ss_mnt, filp, copy-&gt;nf_dst);
	} else {
		nfserr = nfsd4_do_copy(copy, copy-&gt;nf_src-&gt;nf_file,
				       copy-&gt;nf_dst-&gt;nf_file, false);
		nfsd4_cleanup_intra_ssc(copy-&gt;nf_src, copy-&gt;nf_dst);
	}

do_callback:
	nfsd4_send_cb_offload(copy, nfserr);
	cleanup_async_copy(copy);
	return 0;
}

static __be32
nfsd4_copy(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
		union nfsd4_op_u *u)
{
	struct nfsd4_copy *copy = &amp;u-&gt;copy;
	__be32 status;
	struct nfsd4_copy *async_copy = NULL;

	if (nfsd4_ssc_is_inter(copy)) {
		if (!inter_copy_offload_enable || nfsd4_copy_is_sync(copy)) {
			status = nfserr_notsupp;
			goto out;
		}
		status = nfsd4_setup_inter_ssc(rqstp, cstate, copy,
				&amp;copy-&gt;ss_mnt);
		if (status)
			return nfserr_offload_denied;
	} else {
		status = nfsd4_setup_intra_ssc(rqstp, cstate, copy);
		if (status)
			return status;
	}

	copy-&gt;cp_clp = cstate-&gt;clp;
	memcpy(&amp;copy-&gt;fh, &amp;cstate-&gt;current_fh.fh_handle,
		sizeof(struct knfsd_fh));
	if (nfsd4_copy_is_async(copy)) {
		struct nfsd_net *nn = net_generic(SVC_NET(rqstp), nfsd_net_id);

		status = nfserrno(-ENOMEM);
		async_copy = kzalloc(sizeof(struct nfsd4_copy), GFP_KERNEL);
		if (!async_copy)
			goto out_err;
		async_copy-&gt;cp_src = kmalloc(sizeof(*async_copy-&gt;cp_src), GFP_KERNEL);
		if (!async_copy-&gt;cp_src)
			goto out_err;
		if (!nfs4_init_copy_state(nn, copy))
			goto out_err;
		refcount_set(&amp;async_copy-&gt;refcount, 1);
		memcpy(&amp;copy-&gt;cp_res.cb_stateid, &amp;copy-&gt;cp_stateid.stid,
			sizeof(copy-&gt;cp_res.cb_stateid));
		dup_copy_fields(copy, async_copy);
		async_copy-&gt;copy_task = kthread_create(nfsd4_do_async_copy,
				async_copy, "%s", "copy thread");
		if (IS_ERR(async_copy-&gt;copy_task))
			goto out_err;
		spin_lock(&amp;async_copy-&gt;cp_clp-&gt;async_lock);
		list_add(&amp;async_copy-&gt;copies,
				&amp;async_copy-&gt;cp_clp-&gt;async_copies);
		spin_unlock(&amp;async_copy-&gt;cp_clp-&gt;async_lock);
		wake_up_process(async_copy-&gt;copy_task);
		status = nfs_ok;
	} else {
		status = nfsd4_do_copy(copy, copy-&gt;nf_src-&gt;nf_file,
				       copy-&gt;nf_dst-&gt;nf_file, true);
		nfsd4_cleanup_intra_ssc(copy-&gt;nf_src, copy-&gt;nf_dst);
	}
out:
	return status;
out_err:
	if (async_copy)
		cleanup_async_copy(async_copy);
	status = nfserrno(-ENOMEM);
	if (nfsd4_ssc_is_inter(copy))
		nfsd4_interssc_disconnect(copy-&gt;ss_mnt);
	goto out;
}

struct nfsd4_copy *
find_async_copy(struct nfs4_client *clp, stateid_t *stateid)
{
	struct nfsd4_copy *copy;

	spin_lock(&amp;clp-&gt;async_lock);
	list_for_each_entry(copy, &amp;clp-&gt;async_copies, copies) {
		if (memcmp(&amp;copy-&gt;cp_stateid.stid, stateid, NFS4_STATEID_SIZE))
			continue;
		refcount_inc(&amp;copy-&gt;refcount);
		spin_unlock(&amp;clp-&gt;async_lock);
		return copy;
	}
	spin_unlock(&amp;clp-&gt;async_lock);
	return NULL;
}

static __be32
nfsd4_offload_cancel(struct svc_rqst *rqstp,
		     struct nfsd4_compound_state *cstate,
		     union nfsd4_op_u *u)
{
	struct nfsd4_offload_status *os = &amp;u-&gt;offload_status;
	struct nfsd4_copy *copy;
	struct nfs4_client *clp = cstate-&gt;clp;

	copy = find_async_copy(clp, &amp;os-&gt;stateid);
	if (!copy) {
		struct nfsd_net *nn = net_generic(SVC_NET(rqstp), nfsd_net_id);

		return manage_cpntf_state(nn, &amp;os-&gt;stateid, clp, NULL);
	} else
		nfsd4_stop_copy(copy);

	return nfs_ok;
}

static __be32
nfsd4_copy_notify(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
		  union nfsd4_op_u *u)
{
	struct nfsd4_copy_notify *cn = &amp;u-&gt;copy_notify;
	__be32 status;
	struct nfsd_net *nn = net_generic(SVC_NET(rqstp), nfsd_net_id);
	struct nfs4_stid *stid;
	struct nfs4_cpntf_state *cps;
	struct nfs4_client *clp = cstate-&gt;clp;

	status = nfs4_preprocess_stateid_op(rqstp, cstate, &amp;cstate-&gt;current_fh,
					&amp;cn-&gt;cpn_src_stateid, RD_STATE, NULL,
					&amp;stid);
	if (status)
		return status;

	cn-&gt;cpn_sec = nn-&gt;nfsd4_lease;
	cn-&gt;cpn_nsec = 0;

	status = nfserrno(-ENOMEM);
	cps = nfs4_alloc_init_cpntf_state(nn, stid);
	if (!cps)
		goto out;
	memcpy(&amp;cn-&gt;cpn_cnr_stateid, &amp;cps-&gt;cp_stateid.stid, sizeof(stateid_t));
	memcpy(&amp;cps-&gt;cp_p_stateid, &amp;stid-&gt;sc_stateid, sizeof(stateid_t));
	memcpy(&amp;cps-&gt;cp_p_clid, &amp;clp-&gt;cl_clientid, sizeof(clientid_t));

	/* For now, only return one server address in cpn_src, the
	 * address used by the client to connect to this server.
	 */
	cn-&gt;cpn_src-&gt;nl4_type = NL4_NETADDR;
	status = nfsd4_set_netaddr((struct sockaddr *)&amp;rqstp-&gt;rq_daddr,
				 &amp;cn-&gt;cpn_src-&gt;u.nl4_addr);
	WARN_ON_ONCE(status);
	if (status) {
		nfs4_put_cpntf_state(nn, cps);
		goto out;
	}
out:
	nfs4_put_stid(stid);
	return status;
}

static __be32
nfsd4_fallocate(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
		struct nfsd4_fallocate *fallocate, int flags)
{
	__be32 status;
	struct nfsd_file *nf;

	status = nfs4_preprocess_stateid_op(rqstp, cstate, &amp;cstate-&gt;current_fh,
					    &amp;fallocate-&gt;falloc_stateid,
					    WR_STATE, &amp;nf, NULL);
	if (status != nfs_ok) {
		dprintk("NFSD: nfsd4_fallocate: couldn't process stateid!\n");
		return status;
	}

	status = nfsd4_vfs_fallocate(rqstp, &amp;cstate-&gt;current_fh, nf-&gt;nf_file,
				     fallocate-&gt;falloc_offset,
				     fallocate-&gt;falloc_length,
				     flags);
	nfsd_file_put(nf);
	return status;
}
static __be32
nfsd4_offload_status(struct svc_rqst *rqstp,
		     struct nfsd4_compound_state *cstate,
		     union nfsd4_op_u *u)
{
	struct nfsd4_offload_status *os = &amp;u-&gt;offload_status;
	__be32 status = 0;
	struct nfsd4_copy *copy;
	struct nfs4_client *clp = cstate-&gt;clp;

	copy = find_async_copy(clp, &amp;os-&gt;stateid);
	if (copy) {
		os-&gt;count = copy-&gt;cp_res.wr_bytes_written;
		nfs4_put_copy(copy);
	} else
		status = nfserr_bad_stateid;

	return status;
}

static __be32
nfsd4_allocate(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
	       union nfsd4_op_u *u)
{
	return nfsd4_fallocate(rqstp, cstate, &amp;u-&gt;allocate, 0);
}

static __be32
nfsd4_deallocate(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
		 union nfsd4_op_u *u)
{
	return nfsd4_fallocate(rqstp, cstate, &amp;u-&gt;deallocate,
			       FALLOC_FL_PUNCH_HOLE | FALLOC_FL_KEEP_SIZE);
}

static __be32
nfsd4_seek(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
	   union nfsd4_op_u *u)
{
	struct nfsd4_seek *seek = &amp;u-&gt;seek;
	int whence;
	__be32 status;
	struct nfsd_file *nf;

	status = nfs4_preprocess_stateid_op(rqstp, cstate, &amp;cstate-&gt;current_fh,
					    &amp;seek-&gt;seek_stateid,
					    RD_STATE, &amp;nf, NULL);
	if (status) {
		dprintk("NFSD: nfsd4_seek: couldn't process stateid!\n");
		return status;
	}

	switch (seek-&gt;seek_whence) {
	case NFS4_CONTENT_DATA:
		whence = SEEK_DATA;
		break;
	case NFS4_CONTENT_HOLE:
		whence = SEEK_HOLE;
		break;
	default:
		status = nfserr_union_notsupp;
		goto out;
	}

	/*
	 * Note:  This call does change file-&gt;f_pos, but nothing in NFSD
	 *        should ever file-&gt;f_pos.
	 */
	seek-&gt;seek_pos = vfs_llseek(nf-&gt;nf_file, seek-&gt;seek_offset, whence);
	if (seek-&gt;seek_pos &lt; 0)
		status = nfserrno(seek-&gt;seek_pos);
	else if (seek-&gt;seek_pos &gt;= i_size_read(file_inode(nf-&gt;nf_file)))
		seek-&gt;seek_eof = true;

out:
	nfsd_file_put(nf);
	return status;
}

/* This routine never returns NFS_OK!  If there are no other errors, it
 * will return NFSERR_SAME or NFSERR_NOT_SAME depending on whether the
 * attributes matched.  VERIFY is implemented by mapping NFSERR_SAME
 * to NFS_OK after the call; NVERIFY by mapping NFSERR_NOT_SAME to NFS_OK.
 */
static __be32
_nfsd4_verify(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
	     struct nfsd4_verify *verify)
{
	__be32 *buf, *p;
	int count;
	__be32 status;

	status = fh_verify(rqstp, &amp;cstate-&gt;current_fh, 0, NFSD_MAY_NOP);
	if (status)
		return status;

	status = check_attr_support(rqstp, cstate, verify-&gt;ve_bmval, NULL);
	if (status)
		return status;

	if ((verify-&gt;ve_bmval[0] &amp; FATTR4_WORD0_RDATTR_ERROR)
	    || (verify-&gt;ve_bmval[1] &amp; NFSD_WRITEONLY_ATTRS_WORD1))
		return nfserr_inval;
	if (verify-&gt;ve_attrlen &amp; 3)
		return nfserr_inval;

	/* count in words:
	 *   bitmap_len(1) + bitmap(2) + attr_len(1) = 4
	 */
	count = 4 + (verify-&gt;ve_attrlen &gt;&gt; 2);
	buf = kmalloc(count &lt;&lt; 2, GFP_KERNEL);
	if (!buf)
		return nfserr_jukebox;

	p = buf;
	status = nfsd4_encode_fattr_to_buf(&amp;p, count, &amp;cstate-&gt;current_fh,
				    cstate-&gt;current_fh.fh_export,
				    cstate-&gt;current_fh.fh_dentry,
				    verify-&gt;ve_bmval,
				    rqstp, 0);
	/*
	 * If nfsd4_encode_fattr() ran out of space, assume that's because
	 * the attributes are longer (hence different) than those given:
	 */
	if (status == nfserr_resource)
		status = nfserr_not_same;
	if (status)
		goto out_kfree;

	/* skip bitmap */
	p = buf + 1 + ntohl(buf[0]);
	status = nfserr_not_same;
	if (ntohl(*p++) != verify-&gt;ve_attrlen)
		goto out_kfree;
	if (!memcmp(p, verify-&gt;ve_attrval, verify-&gt;ve_attrlen))
		status = nfserr_same;

out_kfree:
	kfree(buf);
	return status;
}

static __be32
nfsd4_nverify(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
	      union nfsd4_op_u *u)
{
	__be32 status;

	status = _nfsd4_verify(rqstp, cstate, &amp;u-&gt;verify);
	return status == nfserr_not_same ? nfs_ok : status;
}

static __be32
nfsd4_verify(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
	     union nfsd4_op_u *u)
{
	__be32 status;

	status = _nfsd4_verify(rqstp, cstate, &amp;u-&gt;nverify);
	return status == nfserr_same ? nfs_ok : status;
}

#ifdef CONFIG_NFSD_PNFS
static const struct nfsd4_layout_ops *
nfsd4_layout_verify(struct svc_export *exp, unsigned int layout_type)
{
	if (!exp-&gt;ex_layout_types) {
		dprintk("%s: export does not support pNFS\n", __func__);
		return NULL;
	}

	if (layout_type &gt;= LAYOUT_TYPE_MAX ||
	    !(exp-&gt;ex_layout_types &amp; (1 &lt;&lt; layout_type))) {
		dprintk("%s: layout type %d not supported\n",
			__func__, layout_type);
		return NULL;
	}

	return nfsd4_layout_ops[layout_type];
}

static __be32
nfsd4_getdeviceinfo(struct svc_rqst *rqstp,
		struct nfsd4_compound_state *cstate, union nfsd4_op_u *u)
{
	struct nfsd4_getdeviceinfo *gdp = &amp;u-&gt;getdeviceinfo;
	const struct nfsd4_layout_ops *ops;
	struct nfsd4_deviceid_map *map;
	struct svc_export *exp;
	__be32 nfserr;

	dprintk("%s: layout_type %u dev_id [0x%llx:0x%x] maxcnt %u\n",
	       __func__,
	       gdp-&gt;gd_layout_type,
	       gdp-&gt;gd_devid.fsid_idx, gdp-&gt;gd_devid.generation,
	       gdp-&gt;gd_maxcount);

	map = nfsd4_find_devid_map(gdp-&gt;gd_devid.fsid_idx);
	if (!map) {
		dprintk("%s: couldn't find device ID to export mapping!\n",
			__func__);
		return nfserr_noent;
	}

	exp = rqst_exp_find(rqstp, map-&gt;fsid_type, map-&gt;fsid);
	if (IS_ERR(exp)) {
		dprintk("%s: could not find device id\n", __func__);
		return nfserr_noent;
	}

	nfserr = nfserr_layoutunavailable;
	ops = nfsd4_layout_verify(exp, gdp-&gt;gd_layout_type);
	if (!ops)
		goto out;

	nfserr = nfs_ok;
	if (gdp-&gt;gd_maxcount != 0) {
		nfserr = ops-&gt;proc_getdeviceinfo(exp-&gt;ex_path.mnt-&gt;mnt_sb,
				rqstp, cstate-&gt;clp, gdp);
	}

	gdp-&gt;gd_notify_types &amp;= ops-&gt;notify_types;
out:
	exp_put(exp);
	return nfserr;
}

static void
nfsd4_getdeviceinfo_release(union nfsd4_op_u *u)
{
	kfree(u-&gt;getdeviceinfo.gd_device);
}

static __be32
nfsd4_layoutget(struct svc_rqst *rqstp,
		struct nfsd4_compound_state *cstate, union nfsd4_op_u *u)
{
	struct nfsd4_layoutget *lgp = &amp;u-&gt;layoutget;
	struct svc_fh *current_fh = &amp;cstate-&gt;current_fh;
	const struct nfsd4_layout_ops *ops;
	struct nfs4_layout_stateid *ls;
	__be32 nfserr;
	int accmode = NFSD_MAY_READ_IF_EXEC;

	switch (lgp-&gt;lg_seg.iomode) {
	case IOMODE_READ:
		accmode |= NFSD_MAY_READ;
		break;
	case IOMODE_RW:
		accmode |= NFSD_MAY_READ | NFSD_MAY_WRITE;
		break;
	default:
		dprintk("%s: invalid iomode %d\n",
			__func__, lgp-&gt;lg_seg.iomode);
		nfserr = nfserr_badiomode;
		goto out;
	}

	nfserr = fh_verify(rqstp, current_fh, 0, accmode);
	if (nfserr)
		goto out;

	nfserr = nfserr_layoutunavailable;
	ops = nfsd4_layout_verify(current_fh-&gt;fh_export, lgp-&gt;lg_layout_type);
	if (!ops)
		goto out;

	/*
	 * Verify minlength and range as per RFC5661:
	 *  o  If loga_length is less than loga_minlength,
	 *     the metadata server MUST return NFS4ERR_INVAL.
	 *  o  If the sum of loga_offset and loga_minlength exceeds
	 *     NFS4_UINT64_MAX, and loga_minlength is not
	 *     NFS4_UINT64_MAX, the error NFS4ERR_INVAL MUST result.
	 *  o  If the sum of loga_offset and loga_length exceeds
	 *     NFS4_UINT64_MAX, and loga_length is not NFS4_UINT64_MAX,
	 *     the error NFS4ERR_INVAL MUST result.
	 */
	nfserr = nfserr_inval;
	if (lgp-&gt;lg_seg.length &lt; lgp-&gt;lg_minlength ||
	    (lgp-&gt;lg_minlength != NFS4_MAX_UINT64 &amp;&amp;
	     lgp-&gt;lg_minlength &gt; NFS4_MAX_UINT64 - lgp-&gt;lg_seg.offset) ||
	    (lgp-&gt;lg_seg.length != NFS4_MAX_UINT64 &amp;&amp;
	     lgp-&gt;lg_seg.length &gt; NFS4_MAX_UINT64 - lgp-&gt;lg_seg.offset))
		goto out;
	if (lgp-&gt;lg_seg.length == 0)
		goto out;

	nfserr = nfsd4_preprocess_layout_stateid(rqstp, cstate, &amp;lgp-&gt;lg_sid,
						true, lgp-&gt;lg_layout_type, &amp;ls);
	if (nfserr) {
		trace_nfsd_layout_get_lookup_fail(&amp;lgp-&gt;lg_sid);
		goto out;
	}

	nfserr = nfserr_recallconflict;
	if (atomic_read(&amp;ls-&gt;ls_stid.sc_file-&gt;fi_lo_recalls))
		goto out_put_stid;

	nfserr = ops-&gt;proc_layoutget(d_inode(current_fh-&gt;fh_dentry),
				     current_fh, lgp);
	if (nfserr)
		goto out_put_stid;

	nfserr = nfsd4_insert_layout(lgp, ls);

out_put_stid:
	mutex_unlock(&amp;ls-&gt;ls_mutex);
	nfs4_put_stid(&amp;ls-&gt;ls_stid);
out:
	return nfserr;
}

static void
nfsd4_layoutget_release(union nfsd4_op_u *u)
{
	kfree(u-&gt;layoutget.lg_content);
}

static __be32
nfsd4_layoutcommit(struct svc_rqst *rqstp,
		struct nfsd4_compound_state *cstate, union nfsd4_op_u *u)
{
	struct nfsd4_layoutcommit *lcp = &amp;u-&gt;layoutcommit;
	const struct nfsd4_layout_seg *seg = &amp;lcp-&gt;lc_seg;
	struct svc_fh *current_fh = &amp;cstate-&gt;current_fh;
	const struct nfsd4_layout_ops *ops;
	loff_t new_size = lcp-&gt;lc_last_wr + 1;
	struct inode *inode;
	struct nfs4_layout_stateid *ls;
	__be32 nfserr;

	nfserr = fh_verify(rqstp, current_fh, 0, NFSD_MAY_WRITE);
	if (nfserr)
		goto out;

	nfserr = nfserr_layoutunavailable;
	ops = nfsd4_layout_verify(current_fh-&gt;fh_export, lcp-&gt;lc_layout_type);
	if (!ops)
		goto out;
	inode = d_inode(current_fh-&gt;fh_dentry);

	nfserr = nfserr_inval;
	if (new_size &lt;= seg-&gt;offset) {
		dprintk("pnfsd: last write before layout segment\n");
		goto out;
	}
	if (new_size &gt; seg-&gt;offset + seg-&gt;length) {
		dprintk("pnfsd: last write beyond layout segment\n");
		goto out;
	}
	if (!lcp-&gt;lc_newoffset &amp;&amp; new_size &gt; i_size_read(inode)) {
		dprintk("pnfsd: layoutcommit beyond EOF\n");
		goto out;
	}

	nfserr = nfsd4_preprocess_layout_stateid(rqstp, cstate, &amp;lcp-&gt;lc_sid,
						false, lcp-&gt;lc_layout_type,
						&amp;ls);
	if (nfserr) {
		trace_nfsd_layout_commit_lookup_fail(&amp;lcp-&gt;lc_sid);
		/* fixup error code as per RFC5661 */
		if (nfserr == nfserr_bad_stateid)
			nfserr = nfserr_badlayout;
		goto out;
	}

	/* LAYOUTCOMMIT does not require any serialization */
	mutex_unlock(&amp;ls-&gt;ls_mutex);

	if (new_size &gt; i_size_read(inode)) {
		lcp-&gt;lc_size_chg = 1;
		lcp-&gt;lc_newsize = new_size;
	} else {
		lcp-&gt;lc_size_chg = 0;
	}

	nfserr = ops-&gt;proc_layoutcommit(inode, lcp);
	nfs4_put_stid(&amp;ls-&gt;ls_stid);
out:
	return nfserr;
}

static __be32
nfsd4_layoutreturn(struct svc_rqst *rqstp,
		struct nfsd4_compound_state *cstate, union nfsd4_op_u *u)
{
	struct nfsd4_layoutreturn *lrp = &amp;u-&gt;layoutreturn;
	struct svc_fh *current_fh = &amp;cstate-&gt;current_fh;
	__be32 nfserr;

	nfserr = fh_verify(rqstp, current_fh, 0, NFSD_MAY_NOP);
	if (nfserr)
		goto out;

	nfserr = nfserr_layoutunavailable;
	if (!nfsd4_layout_verify(current_fh-&gt;fh_export, lrp-&gt;lr_layout_type))
		goto out;

	switch (lrp-&gt;lr_seg.iomode) {
	case IOMODE_READ:
	case IOMODE_RW:
	case IOMODE_ANY:
		break;
	default:
		dprintk("%s: invalid iomode %d\n", __func__,
			lrp-&gt;lr_seg.iomode);
		nfserr = nfserr_inval;
		goto out;
	}

	switch (lrp-&gt;lr_return_type) {
	case RETURN_FILE:
		nfserr = nfsd4_return_file_layouts(rqstp, cstate, lrp);
		break;
	case RETURN_FSID:
	case RETURN_ALL:
		nfserr = nfsd4_return_client_layouts(rqstp, cstate, lrp);
		break;
	default:
		dprintk("%s: invalid return_type %d\n", __func__,
			lrp-&gt;lr_return_type);
		nfserr = nfserr_inval;
		break;
	}
out:
	return nfserr;
}
#endif /* CONFIG_NFSD_PNFS */

static __be32
nfsd4_getxattr(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
	       union nfsd4_op_u *u)
{
	struct nfsd4_getxattr *getxattr = &amp;u-&gt;getxattr;

	return nfsd_getxattr(rqstp, &amp;cstate-&gt;current_fh,
			     getxattr-&gt;getxa_name, &amp;getxattr-&gt;getxa_buf,
			     &amp;getxattr-&gt;getxa_len);
}

static __be32
nfsd4_setxattr(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
	   union nfsd4_op_u *u)
{
	struct nfsd4_setxattr *setxattr = &amp;u-&gt;setxattr;
	__be32 ret;

	if (opens_in_grace(SVC_NET(rqstp)))
		return nfserr_grace;

	ret = nfsd_setxattr(rqstp, &amp;cstate-&gt;current_fh, setxattr-&gt;setxa_name,
			    setxattr-&gt;setxa_buf, setxattr-&gt;setxa_len,
			    setxattr-&gt;setxa_flags);

	if (!ret)
		set_change_info(&amp;setxattr-&gt;setxa_cinfo, &amp;cstate-&gt;current_fh);

	return ret;
}

static __be32
nfsd4_listxattrs(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
	   union nfsd4_op_u *u)
{
	/*
	 * Get the entire list, then copy out only the user attributes
	 * in the encode function.
	 */
	return nfsd_listxattr(rqstp, &amp;cstate-&gt;current_fh,
			     &amp;u-&gt;listxattrs.lsxa_buf, &amp;u-&gt;listxattrs.lsxa_len);
}

static __be32
nfsd4_removexattr(struct svc_rqst *rqstp, struct nfsd4_compound_state *cstate,
	   union nfsd4_op_u *u)
{
	struct nfsd4_removexattr *removexattr = &amp;u-&gt;removexattr;
	__be32 ret;

	if (opens_in_grace(SVC_NET(rqstp)))
		return nfserr_grace;

	ret = nfsd_removexattr(rqstp, &amp;cstate-&gt;current_fh,
	    removexattr-&gt;rmxa_name);

	if (!ret)
		set_change_info(&amp;removexattr-&gt;rmxa_cinfo, &amp;cstate-&gt;current_fh);

	return ret;
}

/*
 * NULL call.
 */
static __be32
nfsd4_proc_null(struct svc_rqst *rqstp)
{
	return rpc_success;
}

static inline void nfsd4_increment_op_stats(u32 opnum)
{
	if (opnum &gt;= FIRST_NFS4_OP &amp;&amp; opnum &lt;= LAST_NFS4_OP)
		percpu_counter_inc(&amp;nfsdstats.counter[NFSD_STATS_NFS4_OP(opnum)]);
}

static const struct nfsd4_operation nfsd4_ops[];

static const char *nfsd4_op_name(unsigned opnum);

/*
 * Enforce NFSv4.1 COMPOUND ordering rules:
 *
 * Also note, enforced elsewhere:
 *	- SEQUENCE other than as first op results in
 *	  NFS4ERR_SEQUENCE_POS. (Enforced in nfsd4_sequence().)
 *	- BIND_CONN_TO_SESSION must be the only op in its compound.
 *	  (Enforced in nfsd4_bind_conn_to_session().)
 *	- DESTROY_SESSION must be the final operation in a compound, if
 *	  sessionid's in SEQUENCE and DESTROY_SESSION are the same.
 *	  (Enforced in nfsd4_destroy_session().)
 */
static __be32 nfs41_check_op_ordering(struct nfsd4_compoundargs *args)
{
	struct nfsd4_op *first_op = &amp;args-&gt;ops[0];

	/* These ordering requirements don't apply to NFSv4.0: */
	if (args-&gt;minorversion == 0)
		return nfs_ok;
	/* This is weird, but OK, not our problem: */
	if (args-&gt;opcnt == 0)
		return nfs_ok;
	if (first_op-&gt;status == nfserr_op_illegal)
		return nfs_ok;
	if (!(nfsd4_ops[first_op-&gt;opnum].op_flags &amp; ALLOWED_AS_FIRST_OP))
		return nfserr_op_not_in_session;
	if (first_op-&gt;opnum == OP_SEQUENCE)
		return nfs_ok;
	/*
	 * So first_op is something allowed outside a session, like
	 * EXCHANGE_ID; but then it has to be the only op in the
	 * compound:
	 */
	if (args-&gt;opcnt != 1)
		return nfserr_not_only_op;
	return nfs_ok;
}

const struct nfsd4_operation *OPDESC(struct nfsd4_op *op)
{
	return &amp;nfsd4_ops[op-&gt;opnum];
}

bool nfsd4_cache_this_op(struct nfsd4_op *op)
{
	if (op-&gt;opnum == OP_ILLEGAL)
		return false;
	return OPDESC(op)-&gt;op_flags &amp; OP_CACHEME;
}

static bool need_wrongsec_check(struct svc_rqst *rqstp)
{
	struct nfsd4_compoundres *resp = rqstp-&gt;rq_resp;
	struct nfsd4_compoundargs *argp = rqstp-&gt;rq_argp;
	struct nfsd4_op *this = &amp;argp-&gt;ops[resp-&gt;opcnt - 1];
	struct nfsd4_op *next = &amp;argp-&gt;ops[resp-&gt;opcnt];
	const struct nfsd4_operation *thisd = OPDESC(this);
	const struct nfsd4_operation *nextd;

	/*
	 * Most ops check wronsec on our own; only the putfh-like ops
	 * have special rules.
	 */
	if (!(thisd-&gt;op_flags &amp; OP_IS_PUTFH_LIKE))
		return false;
	/*
	 * rfc 5661 2.6.3.1.1.6: don't bother erroring out a
	 * put-filehandle operation if we're not going to use the
	 * result:
	 */
	if (argp-&gt;opcnt == resp-&gt;opcnt)
		return false;
	if (next-&gt;opnum == OP_ILLEGAL)
		return false;
	nextd = OPDESC(next);
	/*
	 * Rest of 2.6.3.1.1: certain operations will return WRONGSEC
	 * errors themselves as necessary; others should check for them
	 * now:
	 */
	return !(nextd-&gt;op_flags &amp; OP_HANDLES_WRONGSEC);
}

#ifdef CONFIG_NFSD_V4_2_INTER_SSC
static void
check_if_stalefh_allowed(struct nfsd4_compoundargs *args)
{
	struct nfsd4_op	*op, *current_op = NULL, *saved_op = NULL;
	struct nfsd4_copy *copy;
	struct nfsd4_putfh *putfh;
	int i;

	/* traverse all operation and if it's a COPY compound, mark the
	 * source filehandle to skip verification
	 */
	for (i = 0; i &lt; args-&gt;opcnt; i++) {
		op = &amp;args-&gt;ops[i];
		if (op-&gt;opnum == OP_PUTFH)
			current_op = op;
		else if (op-&gt;opnum == OP_SAVEFH)
			saved_op = current_op;
		else if (op-&gt;opnum == OP_RESTOREFH)
			current_op = saved_op;
		else if (op-&gt;opnum == OP_COPY) {
			copy = (struct nfsd4_copy *)&amp;op-&gt;u;
			if (!saved_op) {
				op-&gt;status = nfserr_nofilehandle;
				return;
			}
			putfh = (struct nfsd4_putfh *)&amp;saved_op-&gt;u;
			if (nfsd4_ssc_is_inter(copy))
				putfh-&gt;no_verify = true;
		}
	}
}
#else
static void
check_if_stalefh_allowed(struct nfsd4_compoundargs *args)
{
}
#endif

/*
 * COMPOUND call.
 */
static __be32
nfsd4_proc_compound(struct svc_rqst *rqstp)
{
	struct nfsd4_compoundargs *args = rqstp-&gt;rq_argp;
	struct nfsd4_compoundres *resp = rqstp-&gt;rq_resp;
	struct nfsd4_op	*op;
	struct nfsd4_compound_state *cstate = &amp;resp-&gt;cstate;
	struct svc_fh *current_fh = &amp;cstate-&gt;current_fh;
	struct svc_fh *save_fh = &amp;cstate-&gt;save_fh;
	struct nfsd_net *nn = net_generic(SVC_NET(rqstp), nfsd_net_id);
	__be32		status;

	resp-&gt;xdr = &amp;rqstp-&gt;rq_res_stream;
	resp-&gt;statusp = resp-&gt;xdr-&gt;p;

	/* reserve space for: NFS status code */
	xdr_reserve_space(resp-&gt;xdr, XDR_UNIT);

	/* reserve space for: taglen, tag, and opcnt */
	xdr_reserve_space(resp-&gt;xdr, XDR_UNIT * 2 + args-&gt;taglen);
	resp-&gt;taglen = args-&gt;taglen;
	resp-&gt;tag = args-&gt;tag;
	resp-&gt;rqstp = rqstp;
	cstate-&gt;minorversion = args-&gt;minorversion;
	fh_init(current_fh, NFS4_FHSIZE);
	fh_init(save_fh, NFS4_FHSIZE);

	/*
	 * Don't use the deferral mechanism for NFSv4; compounds make it
	 * too hard to avoid non-idempotency problems.
	 */
	__clear_bit(RQ_USEDEFERRAL, &amp;rqstp-&gt;rq_flags);

	/*
	 * According to RFC3010, this takes precedence over all other errors.
	 */
	status = nfserr_minor_vers_mismatch;
	if (nfsd_minorversion(nn, args-&gt;minorversion, NFSD_TEST) &lt;= 0)
		goto out;
	status = nfserr_resource;
	if (args-&gt;opcnt &gt; NFSD_MAX_OPS_PER_COMPOUND)
		goto out;

	status = nfs41_check_op_ordering(args);
	if (status) {
		op = &amp;args-&gt;ops[0];
		op-&gt;status = status;
		resp-&gt;opcnt = 1;
		goto encode_op;
	}
	check_if_stalefh_allowed(args);

	rqstp-&gt;rq_lease_breaker = (void **)&amp;cstate-&gt;clp;

	trace_nfsd_compound(rqstp, args-&gt;opcnt);
	while (!status &amp;&amp; resp-&gt;opcnt &lt; args-&gt;opcnt) {
		op = &amp;args-&gt;ops[resp-&gt;opcnt++];

		/*
		 * The XDR decode routines may have pre-set op-&gt;status;
		 * for example, if there is a miscellaneous XDR error
		 * it will be set to nfserr_bad_xdr.
		 */
		if (op-&gt;status) {
			if (op-&gt;opnum == OP_OPEN)
				op-&gt;status = nfsd4_open_omfg(rqstp, cstate, op);
			goto encode_op;
		}
		if (!current_fh-&gt;fh_dentry &amp;&amp;
				!HAS_FH_FLAG(current_fh, NFSD4_FH_FOREIGN)) {
			if (!(op-&gt;opdesc-&gt;op_flags &amp; ALLOWED_WITHOUT_FH)) {
				op-&gt;status = nfserr_nofilehandle;
				goto encode_op;
			}
		} else if (current_fh-&gt;fh_export &amp;&amp;
			   current_fh-&gt;fh_export-&gt;ex_fslocs.migrated &amp;&amp;
			  !(op-&gt;opdesc-&gt;op_flags &amp; ALLOWED_ON_ABSENT_FS)) {
			op-&gt;status = nfserr_moved;
			goto encode_op;
		}

		fh_clear_pre_post_attrs(current_fh);

		/* If op is non-idempotent */
		if (op-&gt;opdesc-&gt;op_flags &amp; OP_MODIFIES_SOMETHING) {
			/*
			 * Don't execute this op if we couldn't encode a
			 * successful reply:
			 */
			u32 plen = op-&gt;opdesc-&gt;op_rsize_bop(rqstp, op);
			/*
			 * Plus if there's another operation, make sure
			 * we'll have space to at least encode an error:
			 */
			if (resp-&gt;opcnt &lt; args-&gt;opcnt)
				plen += COMPOUND_ERR_SLACK_SPACE;
			op-&gt;status = nfsd4_check_resp_size(resp, plen);
		}

		if (op-&gt;status)
			goto encode_op;

		if (op-&gt;opdesc-&gt;op_get_currentstateid)
			op-&gt;opdesc-&gt;op_get_currentstateid(cstate, &amp;op-&gt;u);
		op-&gt;status = op-&gt;opdesc-&gt;op_func(rqstp, cstate, &amp;op-&gt;u);

		/* Only from SEQUENCE */
		if (cstate-&gt;status == nfserr_replay_cache) {
			dprintk("%s NFS4.1 replay from cache\n", __func__);
			status = op-&gt;status;
			goto out;
		}
		if (!op-&gt;status) {
			if (op-&gt;opdesc-&gt;op_set_currentstateid)
				op-&gt;opdesc-&gt;op_set_currentstateid(cstate, &amp;op-&gt;u);

			if (op-&gt;opdesc-&gt;op_flags &amp; OP_CLEAR_STATEID)
				clear_current_stateid(cstate);

			if (current_fh-&gt;fh_export &amp;&amp;
					need_wrongsec_check(rqstp))
				op-&gt;status = check_nfsd_access(current_fh-&gt;fh_export, rqstp);
		}
encode_op:
		if (op-&gt;status == nfserr_replay_me) {
			op-&gt;replay = &amp;cstate-&gt;replay_owner-&gt;so_replay;
			nfsd4_encode_replay(resp-&gt;xdr, op);
			status = op-&gt;status = op-&gt;replay-&gt;rp_status;
		} else {
			nfsd4_encode_operation(resp, op);
			status = op-&gt;status;
		}

		trace_nfsd_compound_status(args-&gt;opcnt, resp-&gt;opcnt, status,
					   nfsd4_op_name(op-&gt;opnum));

		nfsd4_cstate_clear_replay(cstate);
		nfsd4_increment_op_stats(op-&gt;opnum);
	}

	fh_put(current_fh);
	fh_put(save_fh);
	BUG_ON(cstate-&gt;replay_owner);
out:
	cstate-&gt;status = status;
	/* Reset deferral mechanism for RPC deferrals */
	__set_bit(RQ_USEDEFERRAL, &amp;rqstp-&gt;rq_flags);
	return rpc_success;
}

#define op_encode_hdr_size		(2)
#define op_encode_stateid_maxsz		(XDR_QUADLEN(NFS4_STATEID_SIZE))
#define op_encode_verifier_maxsz	(XDR_QUADLEN(NFS4_VERIFIER_SIZE))
#define op_encode_change_info_maxsz	(5)
#define nfs4_fattr_bitmap_maxsz		(4)

/* We'll fall back on returning no lockowner if run out of space: */
#define op_encode_lockowner_maxsz	(0)
#define op_encode_lock_denied_maxsz	(8 + op_encode_lockowner_maxsz)

#define nfs4_owner_maxsz		(1 + XDR_QUADLEN(IDMAP_NAMESZ))

#define op_encode_ace_maxsz		(3 + nfs4_owner_maxsz)
#define op_encode_delegation_maxsz	(1 + op_encode_stateid_maxsz + 1 + \
					 op_encode_ace_maxsz)

#define op_encode_channel_attrs_maxsz	(6 + 1 + 1)

static inline u32 nfsd4_only_status_rsize(struct svc_rqst *rqstp, struct nfsd4_op *op)
{
	return (op_encode_hdr_size) * sizeof(__be32);
}

static inline u32 nfsd4_status_stateid_rsize(struct svc_rqst *rqstp, struct nfsd4_op *op)
{
	return (op_encode_hdr_size + op_encode_stateid_maxsz)* sizeof(__be32);
}

static inline u32 nfsd4_access_rsize(struct svc_rqst *rqstp, struct nfsd4_op *op)
{
	/* ac_supported, ac_resp_access */
	return (op_encode_hdr_size + 2)* sizeof(__be32);
}

static inline u32 nfsd4_commit_rsize(struct svc_rqst *rqstp, struct nfsd4_op *op)
{
	return (op_encode_hdr_size + op_encode_verifier_maxsz) * sizeof(__be32);
}

static inline u32 nfsd4_create_rsize(struct svc_rqst *rqstp, struct nfsd4_op *op)
{
	return (op_encode_hdr_size + op_encode_change_info_maxsz
		+ nfs4_fattr_bitmap_maxsz) * sizeof(__be32);
}

/*
 * Note since this is an idempotent operation we won't insist on failing
 * the op prematurely if the estimate is too large.  We may turn off splice
 * reads unnecessarily.
 */
static inline u32 nfsd4_getattr_rsize(struct svc_rqst *rqstp,
				      struct nfsd4_op *op)
{
	u32 *bmap = op-&gt;u.getattr.ga_bmval;
	u32 bmap0 = bmap[0], bmap1 = bmap[1], bmap2 = bmap[2];
	u32 ret = 0;

	if (bmap0 &amp; FATTR4_WORD0_ACL)
		return svc_max_payload(rqstp);
	if (bmap0 &amp; FATTR4_WORD0_FS_LOCATIONS)
		return svc_max_payload(rqstp);

	if (bmap1 &amp; FATTR4_WORD1_OWNER) {
		ret += IDMAP_NAMESZ + 4;
		bmap1 &amp;= ~FATTR4_WORD1_OWNER;
	}
	if (bmap1 &amp; FATTR4_WORD1_OWNER_GROUP) {
		ret += IDMAP_NAMESZ + 4;
		bmap1 &amp;= ~FATTR4_WORD1_OWNER_GROUP;
	}
	if (bmap0 &amp; FATTR4_WORD0_FILEHANDLE) {
		ret += NFS4_FHSIZE + 4;
		bmap0 &amp;= ~FATTR4_WORD0_FILEHANDLE;
	}
	if (bmap2 &amp; FATTR4_WORD2_SECURITY_LABEL) {
		ret += NFS4_MAXLABELLEN + 12;
		bmap2 &amp;= ~FATTR4_WORD2_SECURITY_LABEL;
	}
	/*
	 * Largest of remaining attributes are 16 bytes (e.g.,
	 * supported_attributes)
	 */
	ret += 16 * (hweight32(bmap0) + hweight32(bmap1) + hweight32(bmap2));
	/* bitmask, length */
	ret += 20;
	return ret;
}

static inline u32 nfsd4_getfh_rsize(struct svc_rqst *rqstp, struct nfsd4_op *op)
{
	return (op_encode_hdr_size + 1) * sizeof(__be32) + NFS4_FHSIZE;
}

static inline u32 nfsd4_link_rsize(struct svc_rqst *rqstp, struct nfsd4_op *op)
{
	return (op_encode_hdr_size + op_encode_change_info_maxsz)
		* sizeof(__be32);
}

static inline u32 nfsd4_lock_rsize(struct svc_rqst *rqstp, struct nfsd4_op *op)
{
	return (op_encode_hdr_size + op_encode_lock_denied_maxsz)
		* sizeof(__be32);
}

static inline u32 nfsd4_open_rsize(struct svc_rqst *rqstp, struct nfsd4_op *op)
{
	return (op_encode_hdr_size + op_encode_stateid_maxsz
		+ op_encode_change_info_maxsz + 1
		+ nfs4_fattr_bitmap_maxsz
		+ op_encode_delegation_maxsz) * sizeof(__be32);
}

static inline u32 nfsd4_read_rsize(struct svc_rqst *rqstp, struct nfsd4_op *op)
{
	u32 maxcount = 0, rlen = 0;

	maxcount = svc_max_payload(rqstp);
	rlen = min(op-&gt;u.read.rd_length, maxcount);

	return (op_encode_hdr_size + 2 + XDR_QUADLEN(rlen)) * sizeof(__be32);
}

static inline u32 nfsd4_read_plus_rsize(struct svc_rqst *rqstp, struct nfsd4_op *op)
{
	u32 maxcount = svc_max_payload(rqstp);
	u32 rlen = min(op-&gt;u.read.rd_length, maxcount);
	/*
	 * If we detect that the file changed during hole encoding, then we
	 * recover by encoding the remaining reply as data. This means we need
	 * to set aside enough room to encode two data segments.
	 */
	u32 seg_len = 2 * (1 + 2 + 1);

	return (op_encode_hdr_size + 2 + seg_len + XDR_QUADLEN(rlen)) * sizeof(__be32);
}

static inline u32 nfsd4_readdir_rsize(struct svc_rqst *rqstp, struct nfsd4_op *op)
{
	u32 maxcount = 0, rlen = 0;

	maxcount = svc_max_payload(rqstp);
	rlen = min(op-&gt;u.readdir.rd_maxcount, maxcount);

	return (op_encode_hdr_size + op_encode_verifier_maxsz +
		XDR_QUADLEN(rlen)) * sizeof(__be32);
}

static inline u32 nfsd4_readlink_rsize(struct svc_rqst *rqstp, struct nfsd4_op *op)
{
	return (op_encode_hdr_size + 1) * sizeof(__be32) + PAGE_SIZE;
}

static inline u32 nfsd4_remove_rsize(struct svc_rqst *rqstp, struct nfsd4_op *op)
{
	return (op_encode_hdr_size + op_encode_change_info_maxsz)
		* sizeof(__be32);
}

static inline u32 nfsd4_rename_rsize(struct svc_rqst *rqstp, struct nfsd4_op *op)
{
	return (op_encode_hdr_size + op_encode_change_info_maxsz
		+ op_encode_change_info_maxsz) * sizeof(__be32);
}

static inline u32 nfsd4_sequence_rsize(struct svc_rqst *rqstp,
				       struct nfsd4_op *op)
{
	return (op_encode_hdr_size
		+ XDR_QUADLEN(NFS4_MAX_SESSIONID_LEN) + 5) * sizeof(__be32);
}

static inline u32 nfsd4_test_stateid_rsize(struct svc_rqst *rqstp, struct nfsd4_op *op)
{
	return (op_encode_hdr_size + 1 + op-&gt;u.test_stateid.ts_num_ids)
		* sizeof(__be32);
}

static inline u32 nfsd4_setattr_rsize(struct svc_rqst *rqstp, struct nfsd4_op *op)
{
	return (op_encode_hdr_size + nfs4_fattr_bitmap_maxsz) * sizeof(__be32);
}

static inline u32 nfsd4_secinfo_rsize(struct svc_rqst *rqstp, struct nfsd4_op *op)
{
	return (op_encode_hdr_size + RPC_AUTH_MAXFLAVOR *
		(4 + XDR_QUADLEN(GSS_OID_MAX_LEN))) * sizeof(__be32);
}

static inline u32 nfsd4_setclientid_rsize(struct svc_rqst *rqstp, struct nfsd4_op *op)
{
	return (op_encode_hdr_size + 2 + XDR_QUADLEN(NFS4_VERIFIER_SIZE)) *
								sizeof(__be32);
}

static inline u32 nfsd4_write_rsize(struct svc_rqst *rqstp, struct nfsd4_op *op)
{
	return (op_encode_hdr_size + 2 + op_encode_verifier_maxsz) * sizeof(__be32);
}

static inline u32 nfsd4_exchange_id_rsize(struct svc_rqst *rqstp, struct nfsd4_op *op)
{
	return (op_encode_hdr_size + 2 + 1 + /* eir_clientid, eir_sequenceid */\
		1 + 1 + /* eir_flags, spr_how */\
		4 + /* spo_must_enforce &amp; _allow with bitmap */\
		2 + /*eir_server_owner.so_minor_id */\
		/* eir_server_owner.so_major_id&lt;&gt; */\
		XDR_QUADLEN(NFS4_OPAQUE_LIMIT) + 1 +\
		/* eir_server_scope&lt;&gt; */\
		XDR_QUADLEN(NFS4_OPAQUE_LIMIT) + 1 +\
		1 + /* eir_server_impl_id array length */\
		0 /* ignored eir_server_impl_id contents */) * sizeof(__be32);
}

static inline u32 nfsd4_bind_conn_to_session_rsize(struct svc_rqst *rqstp, struct nfsd4_op *op)
{
	return (op_encode_hdr_size + \
		XDR_QUADLEN(NFS4_MAX_SESSIONID_LEN) + /* bctsr_sessid */\
		2 /* bctsr_dir, use_conn_in_rdma_mode */) * sizeof(__be32);
}

static inline u32 nfsd4_create_session_rsize(struct svc_rqst *rqstp, struct nfsd4_op *op)
{
	return (op_encode_hdr_size + \
		XDR_QUADLEN(NFS4_MAX_SESSIONID_LEN) + /* sessionid */\
		2 + /* csr_sequence, csr_flags */\
		op_encode_channel_attrs_maxsz + \
		op_encode_channel_attrs_maxsz) * sizeof(__be32);
}

static inline u32 nfsd4_copy_rsize(struct svc_rqst *rqstp, struct nfsd4_op *op)
{
	return (op_encode_hdr_size +
		1 /* wr_callback */ +
		op_encode_stateid_maxsz /* wr_callback */ +
		2 /* wr_count */ +
		1 /* wr_committed */ +
		op_encode_verifier_maxsz +
		1 /* cr_consecutive */ +
		1 /* cr_synchronous */) * sizeof(__be32);
}

static inline u32 nfsd4_offload_status_rsize(struct svc_rqst *rqstp,
					     struct nfsd4_op *op)
{
	return (op_encode_hdr_size +
		2 /* osr_count */ +
		1 /* osr_complete&lt;1&gt; optional 0 for now */) * sizeof(__be32);
}

static inline u32 nfsd4_copy_notify_rsize(struct svc_rqst *rqstp,
					struct nfsd4_op *op)
{
	return (op_encode_hdr_size +
		3 /* cnr_lease_time */ +
		1 /* We support one cnr_source_server */ +
		1 /* cnr_stateid seq */ +
		op_encode_stateid_maxsz /* cnr_stateid */ +
		1 /* num cnr_source_server*/ +
		1 /* nl4_type */ +
		1 /* nl4 size */ +
		XDR_QUADLEN(NFS4_OPAQUE_LIMIT) /*nl4_loc + nl4_loc_sz */)
		* sizeof(__be32);
}

#ifdef CONFIG_NFSD_PNFS
static inline u32 nfsd4_getdeviceinfo_rsize(struct svc_rqst *rqstp, struct nfsd4_op *op)
{
	u32 maxcount = 0, rlen = 0;

	maxcount = svc_max_payload(rqstp);
	rlen = min(op-&gt;u.getdeviceinfo.gd_maxcount, maxcount);

	return (op_encode_hdr_size +
		1 /* gd_layout_type*/ +
		XDR_QUADLEN(rlen) +
		2 /* gd_notify_types */) * sizeof(__be32);
}

/*
 * At this stage we don't really know what layout driver will handle the request,
 * so we need to define an arbitrary upper bound here.
 */
#define MAX_LAYOUT_SIZE		128
static inline u32 nfsd4_layoutget_rsize(struct svc_rqst *rqstp, struct nfsd4_op *op)
{
	return (op_encode_hdr_size +
		1 /* logr_return_on_close */ +
		op_encode_stateid_maxsz +
		1 /* nr of layouts */ +
		MAX_LAYOUT_SIZE) * sizeof(__be32);
}

static inline u32 nfsd4_layoutcommit_rsize(struct svc_rqst *rqstp, struct nfsd4_op *op)
{
	return (op_encode_hdr_size +
		1 /* locr_newsize */ +
		2 /* ns_size */) * sizeof(__be32);
}

static inline u32 nfsd4_layoutreturn_rsize(struct svc_rqst *rqstp, struct nfsd4_op *op)
{
	return (op_encode_hdr_size +
		1 /* lrs_stateid */ +
		op_encode_stateid_maxsz) * sizeof(__be32);
}
#endif /* CONFIG_NFSD_PNFS */


static inline u32 nfsd4_seek_rsize(struct svc_rqst *rqstp, struct nfsd4_op *op)
{
	return (op_encode_hdr_size + 3) * sizeof(__be32);
}

static inline u32 nfsd4_getxattr_rsize(struct svc_rqst *rqstp,
				       struct nfsd4_op *op)
{
	u32 maxcount, rlen;

	maxcount = svc_max_payload(rqstp);
	rlen = min_t(u32, XATTR_SIZE_MAX, maxcount);

	return (op_encode_hdr_size + 1 + XDR_QUADLEN(rlen)) * sizeof(__be32);
}

static inline u32 nfsd4_setxattr_rsize(struct svc_rqst *rqstp,
				       struct nfsd4_op *op)
{
	return (op_encode_hdr_size + op_encode_change_info_maxsz)
		* sizeof(__be32);
}
static inline u32 nfsd4_listxattrs_rsize(struct svc_rqst *rqstp,
					 struct nfsd4_op *op)
{
	u32 maxcount, rlen;

	maxcount = svc_max_payload(rqstp);
	rlen = min(op-&gt;u.listxattrs.lsxa_maxcount, maxcount);

	return (op_encode_hdr_size + 4 + XDR_QUADLEN(rlen)) * sizeof(__be32);
}

static inline u32 nfsd4_removexattr_rsize(struct svc_rqst *rqstp,
					  struct nfsd4_op *op)
{
	return (op_encode_hdr_size + op_encode_change_info_maxsz)
		* sizeof(__be32);
}


static const struct nfsd4_operation nfsd4_ops[] = {
	[OP_ACCESS] = {
		.op_func = nfsd4_access,
		.op_name = "OP_ACCESS",
		.op_rsize_bop = nfsd4_access_rsize,
	},
	[OP_CLOSE] = {
		.op_func = nfsd4_close,
		.op_flags = OP_MODIFIES_SOMETHING,
		.op_name = "OP_CLOSE",
		.op_rsize_bop = nfsd4_status_stateid_rsize,
		.op_get_currentstateid = nfsd4_get_closestateid,
		.op_set_currentstateid = nfsd4_set_closestateid,
	},
	[OP_COMMIT] = {
		.op_func = nfsd4_commit,
		.op_flags = OP_MODIFIES_SOMETHING,
		.op_name = "OP_COMMIT",
		.op_rsize_bop = nfsd4_commit_rsize,
	},
	[OP_CREATE] = {
		.op_func = nfsd4_create,
		.op_flags = OP_MODIFIES_SOMETHING | OP_CACHEME | OP_CLEAR_STATEID,
		.op_name = "OP_CREATE",
		.op_rsize_bop = nfsd4_create_rsize,
	},
	[OP_DELEGRETURN] = {
		.op_func = nfsd4_delegreturn,
		.op_flags = OP_MODIFIES_SOMETHING,
		.op_name = "OP_DELEGRETURN",
		.op_rsize_bop = nfsd4_only_status_rsize,
		.op_get_currentstateid = nfsd4_get_delegreturnstateid,
	},
	[OP_GETATTR] = {
		.op_func = nfsd4_getattr,
		.op_flags = ALLOWED_ON_ABSENT_FS,
		.op_rsize_bop = nfsd4_getattr_rsize,
		.op_name = "OP_GETATTR",
	},
	[OP_GETFH] = {
		.op_func = nfsd4_getfh,
		.op_name = "OP_GETFH",
		.op_rsize_bop = nfsd4_getfh_rsize,
	},
	[OP_LINK] = {
		.op_func = nfsd4_link,
		.op_flags = ALLOWED_ON_ABSENT_FS | OP_MODIFIES_SOMETHING
				| OP_CACHEME,
		.op_name = "OP_LINK",
		.op_rsize_bop = nfsd4_link_rsize,
	},
	[OP_LOCK] = {
		.op_func = nfsd4_lock,
		.op_flags = OP_MODIFIES_SOMETHING |
				OP_NONTRIVIAL_ERROR_ENCODE,
		.op_name = "OP_LOCK",
		.op_rsize_bop = nfsd4_lock_rsize,
		.op_set_currentstateid = nfsd4_set_lockstateid,
	},
	[OP_LOCKT] = {
		.op_func = nfsd4_lockt,
		.op_flags = OP_NONTRIVIAL_ERROR_ENCODE,
		.op_name = "OP_LOCKT",
		.op_rsize_bop = nfsd4_lock_rsize,
	},
	[OP_LOCKU] = {
		.op_func = nfsd4_locku,
		.op_flags = OP_MODIFIES_SOMETHING,
		.op_name = "OP_LOCKU",
		.op_rsize_bop = nfsd4_status_stateid_rsize,
		.op_get_currentstateid = nfsd4_get_lockustateid,
	},
	[OP_LOOKUP] = {
		.op_func = nfsd4_lookup,
		.op_flags = OP_HANDLES_WRONGSEC | OP_CLEAR_STATEID,
		.op_name = "OP_LOOKUP",
		.op_rsize_bop = nfsd4_only_status_rsize,
	},
	[OP_LOOKUPP] = {
		.op_func = nfsd4_lookupp,
		.op_flags = OP_HANDLES_WRONGSEC | OP_CLEAR_STATEID,
		.op_name = "OP_LOOKUPP",
		.op_rsize_bop = nfsd4_only_status_rsize,
	},
	[OP_NVERIFY] = {
		.op_func = nfsd4_nverify,
		.op_name = "OP_NVERIFY",
		.op_rsize_bop = nfsd4_only_status_rsize,
	},
	[OP_OPEN] = {
		.op_func = nfsd4_open,
		.op_flags = OP_HANDLES_WRONGSEC | OP_MODIFIES_SOMETHING,
		.op_name = "OP_OPEN",
		.op_rsize_bop = nfsd4_open_rsize,
		.op_set_currentstateid = nfsd4_set_openstateid,
	},
	[OP_OPEN_CONFIRM] = {
		.op_func = nfsd4_open_confirm,
		.op_flags = OP_MODIFIES_SOMETHING,
		.op_name = "OP_OPEN_CONFIRM",
		.op_rsize_bop = nfsd4_status_stateid_rsize,
	},
	[OP_OPEN_DOWNGRADE] = {
		.op_func = nfsd4_open_downgrade,
		.op_flags = OP_MODIFIES_SOMETHING,
		.op_name = "OP_OPEN_DOWNGRADE",
		.op_rsize_bop = nfsd4_status_stateid_rsize,
		.op_get_currentstateid = nfsd4_get_opendowngradestateid,
		.op_set_currentstateid = nfsd4_set_opendowngradestateid,
	},
	[OP_PUTFH] = {
		.op_func = nfsd4_putfh,
		.op_flags = ALLOWED_WITHOUT_FH | ALLOWED_ON_ABSENT_FS
				| OP_IS_PUTFH_LIKE | OP_CLEAR_STATEID,
		.op_name = "OP_PUTFH",
		.op_rsize_bop = nfsd4_only_status_rsize,
	},
	[OP_PUTPUBFH] = {
		.op_func = nfsd4_putrootfh,
		.op_flags = ALLOWED_WITHOUT_FH | ALLOWED_ON_ABSENT_FS
				| OP_IS_PUTFH_LIKE | OP_CLEAR_STATEID,
		.op_name = "OP_PUTPUBFH",
		.op_rsize_bop = nfsd4_only_status_rsize,
	},
	[OP_PUTROOTFH] = {
		.op_func = nfsd4_putrootfh,
		.op_flags = ALLOWED_WITHOUT_FH | ALLOWED_ON_ABSENT_FS
				| OP_IS_PUTFH_LIKE | OP_CLEAR_STATEID,
		.op_name = "OP_PUTROOTFH",
		.op_rsize_bop = nfsd4_only_status_rsize,
	},
	[OP_READ] = {
		.op_func = nfsd4_read,
		.op_release = nfsd4_read_release,
		.op_name = "OP_READ",
		.op_rsize_bop = nfsd4_read_rsize,
		.op_get_currentstateid = nfsd4_get_readstateid,
	},
	[OP_READDIR] = {
		.op_func = nfsd4_readdir,
		.op_name = "OP_READDIR",
		.op_rsize_bop = nfsd4_readdir_rsize,
	},
	[OP_READLINK] = {
		.op_func = nfsd4_readlink,
		.op_name = "OP_READLINK",
		.op_rsize_bop = nfsd4_readlink_rsize,
	},
	[OP_REMOVE] = {
		.op_func = nfsd4_remove,
		.op_flags = OP_MODIFIES_SOMETHING | OP_CACHEME,
		.op_name = "OP_REMOVE",
		.op_rsize_bop = nfsd4_remove_rsize,
	},
	[OP_RENAME] = {
		.op_func = nfsd4_rename,
		.op_flags = OP_MODIFIES_SOMETHING | OP_CACHEME,
		.op_name = "OP_RENAME",
		.op_rsize_bop = nfsd4_rename_rsize,
	},
	[OP_RENEW] = {
		.op_func = nfsd4_renew,
		.op_flags = ALLOWED_WITHOUT_FH | ALLOWED_ON_ABSENT_FS
				| OP_MODIFIES_SOMETHING,
		.op_name = "OP_RENEW",
		.op_rsize_bop = nfsd4_only_status_rsize,

	},
	[OP_RESTOREFH] = {
		.op_func = nfsd4_restorefh,
		.op_flags = ALLOWED_WITHOUT_FH | ALLOWED_ON_ABSENT_FS
				| OP_IS_PUTFH_LIKE | OP_MODIFIES_SOMETHING,
		.op_name = "OP_RESTOREFH",
		.op_rsize_bop = nfsd4_only_status_rsize,
	},
	[OP_SAVEFH] = {
		.op_func = nfsd4_savefh,
		.op_flags = OP_HANDLES_WRONGSEC | OP_MODIFIES_SOMETHING,
		.op_name = "OP_SAVEFH",
		.op_rsize_bop = nfsd4_only_status_rsize,
	},
	[OP_SECINFO] = {
		.op_func = nfsd4_secinfo,
		.op_release = nfsd4_secinfo_release,
		.op_flags = OP_HANDLES_WRONGSEC,
		.op_name = "OP_SECINFO",
		.op_rsize_bop = nfsd4_secinfo_rsize,
	},
	[OP_SETATTR] = {
		.op_func = nfsd4_setattr,
		.op_name = "OP_SETATTR",
		.op_flags = OP_MODIFIES_SOMETHING | OP_CACHEME
				| OP_NONTRIVIAL_ERROR_ENCODE,
		.op_rsize_bop = nfsd4_setattr_rsize,
		.op_get_currentstateid = nfsd4_get_setattrstateid,
	},
	[OP_SETCLIENTID] = {
		.op_func = nfsd4_setclientid,
		.op_flags = ALLOWED_WITHOUT_FH | ALLOWED_ON_ABSENT_FS
				| OP_MODIFIES_SOMETHING | OP_CACHEME
				| OP_NONTRIVIAL_ERROR_ENCODE,
		.op_name = "OP_SETCLIENTID",
		.op_rsize_bop = nfsd4_setclientid_rsize,
	},
	[OP_SETCLIENTID_CONFIRM] = {
		.op_func = nfsd4_setclientid_confirm,
		.op_flags = ALLOWED_WITHOUT_FH | ALLOWED_ON_ABSENT_FS
				| OP_MODIFIES_SOMETHING | OP_CACHEME,
		.op_name = "OP_SETCLIENTID_CONFIRM",
		.op_rsize_bop = nfsd4_only_status_rsize,
	},
	[OP_VERIFY] = {
		.op_func = nfsd4_verify,
		.op_name = "OP_VERIFY",
		.op_rsize_bop = nfsd4_only_status_rsize,
	},
	[OP_WRITE] = {
		.op_func = nfsd4_write,
		.op_flags = OP_MODIFIES_SOMETHING | OP_CACHEME,
		.op_name = "OP_WRITE",
		.op_rsize_bop = nfsd4_write_rsize,
		.op_get_currentstateid = nfsd4_get_writestateid,
	},
	[OP_RELEASE_LOCKOWNER] = {
		.op_func = nfsd4_release_lockowner,
		.op_flags = ALLOWED_WITHOUT_FH | ALLOWED_ON_ABSENT_FS
				| OP_MODIFIES_SOMETHING,
		.op_name = "OP_RELEASE_LOCKOWNER",
		.op_rsize_bop = nfsd4_only_status_rsize,
	},

	/* NFSv4.1 operations */
	[OP_EXCHANGE_ID] = {
		.op_func = nfsd4_exchange_id,
		.op_flags = ALLOWED_WITHOUT_FH | ALLOWED_AS_FIRST_OP
				| OP_MODIFIES_SOMETHING,
		.op_name = "OP_EXCHANGE_ID",
		.op_rsize_bop = nfsd4_exchange_id_rsize,
	},
	[OP_BACKCHANNEL_CTL] = {
		.op_func = nfsd4_backchannel_ctl,
		.op_flags = ALLOWED_WITHOUT_FH | OP_MODIFIES_SOMETHING,
		.op_name = "OP_BACKCHANNEL_CTL",
		.op_rsize_bop = nfsd4_only_status_rsize,
	},
	[OP_BIND_CONN_TO_SESSION] = {
		.op_func = nfsd4_bind_conn_to_session,
		.op_flags = ALLOWED_WITHOUT_FH | ALLOWED_AS_FIRST_OP
				| OP_MODIFIES_SOMETHING,
		.op_name = "OP_BIND_CONN_TO_SESSION",
		.op_rsize_bop = nfsd4_bind_conn_to_session_rsize,
	},
	[OP_CREATE_SESSION] = {
		.op_func = nfsd4_create_session,
		.op_flags = ALLOWED_WITHOUT_FH | ALLOWED_AS_FIRST_OP
				| OP_MODIFIES_SOMETHING,
		.op_name = "OP_CREATE_SESSION",
		.op_rsize_bop = nfsd4_create_session_rsize,
	},
	[OP_DESTROY_SESSION] = {
		.op_func = nfsd4_destroy_session,
		.op_flags = ALLOWED_WITHOUT_FH | ALLOWED_AS_FIRST_OP
				| OP_MODIFIES_SOMETHING,
		.op_name = "OP_DESTROY_SESSION",
		.op_rsize_bop = nfsd4_only_status_rsize,
	},
	[OP_SEQUENCE] = {
		.op_func = nfsd4_sequence,
		.op_flags = ALLOWED_WITHOUT_FH | ALLOWED_AS_FIRST_OP,
		.op_name = "OP_SEQUENCE",
		.op_rsize_bop = nfsd4_sequence_rsize,
	},
	[OP_DESTROY_CLIENTID] = {
		.op_func = nfsd4_destroy_clientid,
		.op_flags = ALLOWED_WITHOUT_FH | ALLOWED_AS_FIRST_OP
				| OP_MODIFIES_SOMETHING,
		.op_name = "OP_DESTROY_CLIENTID",
		.op_rsize_bop = nfsd4_only_status_rsize,
	},
	[OP_RECLAIM_COMPLETE] = {
		.op_func = nfsd4_reclaim_complete,
		.op_flags = ALLOWED_WITHOUT_FH | OP_MODIFIES_SOMETHING,
		.op_name = "OP_RECLAIM_COMPLETE",
		.op_rsize_bop = nfsd4_only_status_rsize,
	},
	[OP_SECINFO_NO_NAME] = {
		.op_func = nfsd4_secinfo_no_name,
		.op_release = nfsd4_secinfo_no_name_release,
		.op_flags = OP_HANDLES_WRONGSEC,
		.op_name = "OP_SECINFO_NO_NAME",
		.op_rsize_bop = nfsd4_secinfo_rsize,
	},
	[OP_TEST_STATEID] = {
		.op_func = nfsd4_test_stateid,
		.op_flags = ALLOWED_WITHOUT_FH,
		.op_name = "OP_TEST_STATEID",
		.op_rsize_bop = nfsd4_test_stateid_rsize,
	},
	[OP_FREE_STATEID] = {
		.op_func = nfsd4_free_stateid,
		.op_flags = ALLOWED_WITHOUT_FH | OP_MODIFIES_SOMETHING,
		.op_name = "OP_FREE_STATEID",
		.op_get_currentstateid = nfsd4_get_freestateid,
		.op_rsize_bop = nfsd4_only_status_rsize,
	},
#ifdef CONFIG_NFSD_PNFS
	[OP_GETDEVICEINFO] = {
		.op_func = nfsd4_getdeviceinfo,
		.op_release = nfsd4_getdeviceinfo_release,
		.op_flags = ALLOWED_WITHOUT_FH,
		.op_name = "OP_GETDEVICEINFO",
		.op_rsize_bop = nfsd4_getdeviceinfo_rsize,
	},
	[OP_LAYOUTGET] = {
		.op_func = nfsd4_layoutget,
		.op_release = nfsd4_layoutget_release,
		.op_flags = OP_MODIFIES_SOMETHING,
		.op_name = "OP_LAYOUTGET",
		.op_rsize_bop = nfsd4_layoutget_rsize,
	},
	[OP_LAYOUTCOMMIT] = {
		.op_func = nfsd4_layoutcommit,
		.op_flags = OP_MODIFIES_SOMETHING,
		.op_name = "OP_LAYOUTCOMMIT",
		.op_rsize_bop = nfsd4_layoutcommit_rsize,
	},
	[OP_LAYOUTRETURN] = {
		.op_func = nfsd4_layoutreturn,
		.op_flags = OP_MODIFIES_SOMETHING,
		.op_name = "OP_LAYOUTRETURN",
		.op_rsize_bop = nfsd4_layoutreturn_rsize,
	},
#endif /* CONFIG_NFSD_PNFS */

	/* NFSv4.2 operations */
	[OP_ALLOCATE] = {
		.op_func = nfsd4_allocate,
		.op_flags = OP_MODIFIES_SOMETHING,
		.op_name = "OP_ALLOCATE",
		.op_rsize_bop = nfsd4_only_status_rsize,
	},
	[OP_DEALLOCATE] = {
		.op_func = nfsd4_deallocate,
		.op_flags = OP_MODIFIES_SOMETHING,
		.op_name = "OP_DEALLOCATE",
		.op_rsize_bop = nfsd4_only_status_rsize,
	},
	[OP_CLONE] = {
		.op_func = nfsd4_clone,
		.op_flags = OP_MODIFIES_SOMETHING,
		.op_name = "OP_CLONE",
		.op_rsize_bop = nfsd4_only_status_rsize,
	},
	[OP_COPY] = {
		.op_func = nfsd4_copy,
		.op_flags = OP_MODIFIES_SOMETHING,
		.op_name = "OP_COPY",
		.op_rsize_bop = nfsd4_copy_rsize,
	},
	[OP_READ_PLUS] = {
		.op_func = nfsd4_read,
		.op_release = nfsd4_read_release,
		.op_name = "OP_READ_PLUS",
		.op_rsize_bop = nfsd4_read_plus_rsize,
		.op_get_currentstateid = nfsd4_get_readstateid,
	},
	[OP_SEEK] = {
		.op_func = nfsd4_seek,
		.op_name = "OP_SEEK",
		.op_rsize_bop = nfsd4_seek_rsize,
	},
	[OP_OFFLOAD_STATUS] = {
		.op_func = nfsd4_offload_status,
		.op_name = "OP_OFFLOAD_STATUS",
		.op_rsize_bop = nfsd4_offload_status_rsize,
	},
	[OP_OFFLOAD_CANCEL] = {
		.op_func = nfsd4_offload_cancel,
		.op_flags = OP_MODIFIES_SOMETHING,
		.op_name = "OP_OFFLOAD_CANCEL",
		.op_rsize_bop = nfsd4_only_status_rsize,
	},
	[OP_COPY_NOTIFY] = {
		.op_func = nfsd4_copy_notify,
		.op_flags = OP_MODIFIES_SOMETHING,
		.op_name = "OP_COPY_NOTIFY",
		.op_rsize_bop = nfsd4_copy_notify_rsize,
	},
	[OP_GETXATTR] = {
		.op_func = nfsd4_getxattr,
		.op_name = "OP_GETXATTR",
		.op_rsize_bop = nfsd4_getxattr_rsize,
	},
	[OP_SETXATTR] = {
		.op_func = nfsd4_setxattr,
		.op_flags = OP_MODIFIES_SOMETHING | OP_CACHEME,
		.op_name = "OP_SETXATTR",
		.op_rsize_bop = nfsd4_setxattr_rsize,
	},
	[OP_LISTXATTRS] = {
		.op_func = nfsd4_listxattrs,
		.op_name = "OP_LISTXATTRS",
		.op_rsize_bop = nfsd4_listxattrs_rsize,
	},
	[OP_REMOVEXATTR] = {
		.op_func = nfsd4_removexattr,
		.op_flags = OP_MODIFIES_SOMETHING | OP_CACHEME,
		.op_name = "OP_REMOVEXATTR",
		.op_rsize_bop = nfsd4_removexattr_rsize,
	},
};

/**
 * nfsd4_spo_must_allow - Determine if the compound op contains an
 * operation that is allowed to be sent with machine credentials
 *
 * @rqstp: a pointer to the struct svc_rqst
 *
 * Checks to see if the compound contains a spo_must_allow op
 * and confirms that it was sent with the proper machine creds.
 */

bool nfsd4_spo_must_allow(struct svc_rqst *rqstp)
{
	struct nfsd4_compoundres *resp = rqstp-&gt;rq_resp;
	struct nfsd4_compoundargs *argp = rqstp-&gt;rq_argp;
	struct nfsd4_op *this;
	struct nfsd4_compound_state *cstate = &amp;resp-&gt;cstate;
	struct nfs4_op_map *allow = &amp;cstate-&gt;clp-&gt;cl_spo_must_allow;
	u32 opiter;

	if (!cstate-&gt;minorversion)
		return false;

	if (cstate-&gt;spo_must_allowed)
		return true;

	opiter = resp-&gt;opcnt;
	while (opiter &lt; argp-&gt;opcnt) {
		this = &amp;argp-&gt;ops[opiter++];
		if (test_bit(this-&gt;opnum, allow-&gt;u.longs) &amp;&amp;
			cstate-&gt;clp-&gt;cl_mach_cred &amp;&amp;
			nfsd4_mach_creds_match(cstate-&gt;clp, rqstp)) {
			cstate-&gt;spo_must_allowed = true;
			return true;
		}
	}
	cstate-&gt;spo_must_allowed = false;
	return false;
}

int nfsd4_max_reply(struct svc_rqst *rqstp, struct nfsd4_op *op)
{
	if (op-&gt;opnum == OP_ILLEGAL || op-&gt;status == nfserr_notsupp)
		return op_encode_hdr_size * sizeof(__be32);

	BUG_ON(OPDESC(op)-&gt;op_rsize_bop == NULL);
	return OPDESC(op)-&gt;op_rsize_bop(rqstp, op);
}

void warn_on_nonidempotent_op(struct nfsd4_op *op)
{
	if (OPDESC(op)-&gt;op_flags &amp; OP_MODIFIES_SOMETHING) {
		pr_err("unable to encode reply to nonidempotent op %u (%s)\n",
			op-&gt;opnum, nfsd4_op_name(op-&gt;opnum));
		WARN_ON_ONCE(1);
	}
}

static const char *nfsd4_op_name(unsigned opnum)
{
	if (opnum &lt; ARRAY_SIZE(nfsd4_ops))
		return nfsd4_ops[opnum].op_name;
	return "unknown_operation";
}

static const struct svc_procedure nfsd_procedures4[2] = {
	[NFSPROC4_NULL] = {
		.pc_func = nfsd4_proc_null,
		.pc_decode = nfssvc_decode_voidarg,
		.pc_encode = nfssvc_encode_voidres,
		.pc_argsize = sizeof(struct nfsd_voidargs),
		.pc_ressize = sizeof(struct nfsd_voidres),
		.pc_cachetype = RC_NOCACHE,
		.pc_xdrressize = 1,
		.pc_name = "NULL",
	},
	[NFSPROC4_COMPOUND] = {
		.pc_func = nfsd4_proc_compound,
		.pc_decode = nfs4svc_decode_compoundargs,
		.pc_encode = nfs4svc_encode_compoundres,
		.pc_argsize = sizeof(struct nfsd4_compoundargs),
		.pc_ressize = sizeof(struct nfsd4_compoundres),
		.pc_release = nfsd4_release_compoundargs,
		.pc_cachetype = RC_NOCACHE,
		.pc_xdrressize = NFSD_BUFSIZE/4,
		.pc_name = "COMPOUND",
	},
};

static unsigned int nfsd_count3[ARRAY_SIZE(nfsd_procedures4)];
const struct svc_version nfsd_version4 = {
	.vs_vers		= 4,
	.vs_nproc		= 2,
	.vs_proc		= nfsd_procedures4,
	.vs_count		= nfsd_count3,
	.vs_dispatch		= nfsd_dispatch,
	.vs_xdrsize		= NFS4_SVC_XDRSIZE,
	.vs_rpcb_optnl		= true,
	.vs_need_cong_ctrl	= true,
};
</export></linux></linux></linux></linux></linux></linux></linux></linux></andros@umich.edu></kmsmith@umich.edu>