import torch
from torch import nn

class RevIN(nn.Module):
    def __init__(self, num_features: int, eps=1e-16, affine=True):
        """
        :param num_features: the number of features or channels
        :param eps: a value added for numerical stability
        :param affine: if True, RevIN has learnable affine parameters
        """
        super(RevIN, self).__init__()
        self.num_features = num_features
        self.eps = eps
        self.val_mask = None
        self.types = None
        self.affine = affine
        if self.affine:
            self._init_params()

    def forward(self, x, mode:str, val_mask=None, types=None):
        self.mask = val_mask
        self.types = types
        if mode == 'norm':
            self._get_statistics(x)
            x = self._normalize(x)
        elif mode == 'denorm':
            x = self._denormalize(x)
        else: raise NotImplementedError
        return x

    def _init_params(self):
        # initialize RevIN params: (C,)
        self.affine_weight = nn.Parameter(torch.ones(self.num_features))
        self.affine_bias = nn.Parameter(torch.zeros(self.num_features))

    def _get_statistics(self, x):
        dim2reduce = tuple(range(1, x.ndim-1))
        if self.val_mask is not None:
            bs = x.shape[0]
            x = x[~self.val_mask].reshape(bs, -1, 1)
        self.mean = torch.mean(x, dim=dim2reduce, keepdim=True).detach()
        self.stdev = torch.sqrt(torch.var(x, dim=dim2reduce, keepdim=True, unbiased=False) + self.eps).detach()
        self.val_mask = None


        # print('ok')
        # print(self.types.shape)
        # print(self.mean.shape)
        # print(self.stdev.shape)

    def _normalize(self, x):
        x = x - self.mean
        x = x / self.stdev
        if self.affine:
            x = x * self.affine_weight
            x = x + self.affine_bias
        return x

    def _denormalize(self, x):
        if self.affine:
            x = x - self.affine_bias
            x = x / (self.affine_weight + self.eps)
        x = x * self.stdev.cuda()
        x = x + self.mean.cuda()
        return x
