'''
Author: 
Email: 
Date: 2021-09-07 00:04:00
LastEditTime: 2022-03-26 00:12:34
Description: From https://github.com/MankaranSingh/car-models
'''
import numpy as np
import math


class Bicycle(object):
    def __init__(self, start_point=[0, 0, 0], v=0, shape=[10, 10]):
        self.xc = start_point[0]
        self.yc = start_point[1]
        self.theta = start_point[2]
        self.v = v
        self.shape = shape
        self.delta = 0
        self.beta = 0
        
        self.L = 2.56
        self.lr = self.L / 2
        self.max_v = 100

    def get_info(self):
        return [self.xc, self.yc, self.v, self.theta] + self.shape

    def get_speed(self):
        return self.v
    
    def reset(self, start_point, v):
        self.xc = start_point[0]
        self.yc = start_point[1]
        self.theta = start_point[2]
        self.v = v
        self.delta = 0
        self.beta = 0

    # step with Kinematics
    def step(self, a, d_delta, dt=0.2, const_speed=False):
        if not const_speed:
            dx = self.v*np.cos(self.theta) * dt 
            dy = self.v*np.sin(self.theta) * dt 
            self.xc = self.xc + dx
            self.yc = self.yc + dy
            self.theta = self.theta + self.v*np.cos(self.beta)*np.tan(self.delta)/self.L * dt
            self.beta = np.arctan(self.lr*self.delta/self.L)
            self.v += a*dt
            self.delta += d_delta
            self.delta = np.clip(self.delta, -0.5, 0.5)
            self.v = np.clip(self.v, 0, self.max_v)
        else:
            dx = self.v*np.cos(self.theta) * dt 
            dy = self.v*np.sin(self.theta) * dt 
            self.xc = self.xc + dx
            self.yc = self.yc + dy
            self.theta = self.theta + self.v*np.cos(self.beta)*np.tan(self.delta)/self.L * dt
            self.beta = np.arctan(self.lr*self.delta/self.L)
            self.delta += d_delta
            self.delta = np.clip(self.delta, -0.5, 0.5)
    
    def change_lane(self, change, lane_width):
        if change == 'right':
            self.xc = self.xc + lane_width
        elif change == 'left':
            self.xc = self.xc - lane_width


class DynamicBicycle(object):
    def __init__(self, start_point=[0, 0, 0], v=0, shape=[10, 10]):
        self.xc = start_point[0]
        self.yc = start_point[1]
        self.theta = start_point[2]
        self.shape = shape
        self.vx = v
        self.vy = 0.0
        self.max_v = 100
        self.omega = 0.0
        self.delta = 0.0 
        self.L = 2.56  # [m]
        self.Lr = self.L / 2.0  # [m]
        self.Lf = self.L - self.Lr
        self.Cf = 1600.0 * 2.0  # N/rad
        self.Cr = 1700.0 * 2.0  # N/rad
        self.Iz = 2250.0  # kg/m2
        self.m = 1500.0  # kg
        self.magic_number = 25

    def get_speed(self):
        return self.vx

    def reset(self, start_point, v):
        self.xc = start_point[0]
        self.yc = start_point[1]
        self.theta = start_point[2]
        self.vx = v
        self.vy = 0.0
        self.delta = 0
        self.omega = 0.0

    def get_info(self):
        return [self.xc, self.yc, self.v, self.theta] + self.shape

    # step with Dynamics
    def step(self, a, d_delta, dt=0.2, const_speed=False):
        if not const_speed:
            self.xc = self.xc + self.vx * math.cos(self.theta) * dt - self.vy * math.sin(self.theta) * dt
            self.yc = self.yc + self.vx * math.sin(self.theta) * dt + self.vy * math.cos(self.theta) * dt
            self.theta = self.theta + self.omega * dt
            Ffy = -self.Cf * math.atan2(((self.vy + self.Lf * self.omega) / self.vx - self.delta), 1.0)
            Fry = -self.Cr * math.atan2((self.vy - self.Lr * self.omega) / self.vx, 1.0)
            self.vx = self.vx + (a - Ffy * math.sin(self.delta) / self.m + self.vy * self.omega) * dt - self.vx / self.magic_number
            self.vy = self.vy + (Fry / self.m + Ffy * math.cos(self.delta) / self.m - self.vx * self.omega) * dt
            self.omega = self.omega + (Ffy * self.Lf * math.cos(self.delta) - Fry * self.Lr) / self.Iz * dt
            self.delta += d_delta
            self.vx = np.clip(self.vx, 0, self.max_v)        
            self.delta = np.clip(self.delta, -0.5, 0.5)
        else:
            self.xc = self.xc + self.vx * math.cos(self.theta) * dt - self.vy * math.sin(self.theta) * dt
            self.yc = self.yc + self.vx * math.sin(self.theta) * dt + self.vy * math.cos(self.theta) * dt
            self.theta = self.theta + self.omega * dt


class ConstSpeed(object):
    """
    This model will not be influenced by any actions
    """
    def __init__(self, start_point=[0, 0, 0], v=0, shape=[10, 10], magic_number=25):
        self.xc = start_point[0]
        self.yc = start_point[1]
        self.theta = start_point[2]
        self.v = v
        self.shape = shape
        self.delta = 0
        self.beta = 0
        
        self.L = 2.56
        self.lr = self.L / 2
        self.magic_number = magic_number
        self.max_v = 100

    def get_info(self):
        return [self.xc, self.yc, self.v, self.theta] + self.shape

    def get_speed(self):
        return self.v
    
    def reset(self, start_point, v):
        self.xc = start_point[0]
        self.yc = start_point[1]
        self.theta = start_point[2]
        self.v = v
        self.delta = 0
        self.beta = 0

    # step with Kinematics
    def step(self, a, d_delta, dt=0.2):
        dx = self.v*np.cos(self.theta) * dt 
        dy = self.v*np.sin(self.theta) * dt 
        self.xc = self.xc + dx
        self.yc = self.yc + dy
        self.delta = np.clip(self.delta, -0.5, 0.5)
        self.v = np.clip(self.v, 0, self.max_v)
