import argparse
import numpy as np

from improved_diffusion import dist_util, logger
from improved_diffusion.image_datasets import load_data
from improved_diffusion.resample import create_named_schedule_sampler
from improved_diffusion.script_util import (
    model_and_diffusion_defaults,
    create_model_and_diffusion,
    args_to_dict,
    add_dict_to_argparser,
)
from improved_diffusion.train_util import TrainLoop

import torch as th

def main():
    args = create_argparser().parse_args()

    if th.cuda.is_available():
        print('cuda here!')
    else:
        print('no cuda')

    dist_util.setup_dist()
    logger.configure()

    # Initialize beta_values as None
    beta_values = None
    # Load beta values if the file is provided and not empty
    if args.beta_values_file:
        try:
            loaded_beta_values = np.load(args.beta_values_file)
            if loaded_beta_values.size > 0:  # Ensure loaded beta values are not empty
                beta_values = loaded_beta_values
                logger.log(f"Loaded beta values from {args.beta_values_file}")
            else:
                logger.log("Beta values file is empty. Proceeding with default beta values.")
        except Exception as e:
            logger.log(f"Error loading beta values: {e}. Proceeding with default beta values.")

    logger.log("creating model and diffusion...")
    # Create model and diffusion with or without custom beta values
    model_kwargs = args_to_dict(args, model_and_diffusion_defaults().keys())
    # if beta_values is not None:
    #     model_kwargs['beta_values'] = beta_values  # Add beta_values to kwargs if they are loaded

    model, diffusion = create_model_and_diffusion(**model_kwargs,in_channels = 1) #MNIST 1
    model.to(dist_util.dev())
    schedule_sampler = create_named_schedule_sampler(args.schedule_sampler, diffusion)

    if beta_values is not None:
        diffusion.betas = beta_values
        diffusion.update_alpha()
        diffusion.reset_counters()


    logger.log("creating data loader...")
    data = load_data(
        data_dir=args.data_dir,
        batch_size=args.batch_size,
        image_size=args.image_size,
        class_cond=args.class_cond,
    )

    logger.log("training...")
    TrainLoop(
        model=model,
        diffusion=diffusion,
        data=data,
        batch_size=args.batch_size,
        microbatch=args.microbatch,
        lr=args.lr,
        ema_rate=args.ema_rate,
        log_interval=args.log_interval,
        save_interval=args.save_interval,
        resume_checkpoint=args.resume_checkpoint,
        use_fp16=args.use_fp16,
        fp16_scale_growth=args.fp16_scale_growth,
        schedule_sampler=schedule_sampler,
        weight_decay=args.weight_decay,
        lr_anneal_steps=args.lr_anneal_steps,
        schedule_tune = args.schedule_tune,
        schedule_lr = args.schedule_lr,
    ).run_loop()


def create_argparser():
    defaults = dict(
        data_dir="",
        schedule_sampler="uniform",
        lr=1e-4,
        weight_decay=0.0,
        lr_anneal_steps=0,
        schedule_tune=False,
        schedule_lr = 0.01,
        batch_size=1,
        microbatch=-1,  # -1 disables microbatches
        ema_rate="0.9999",
        log_interval=10,
        save_interval=1000,
        resume_checkpoint="",
        use_fp16=False,
        fp16_scale_growth=1e-3,
        beta_values_file="",  # Argument for the path to the .npy file with beta values
    )
    defaults.update(model_and_diffusion_defaults())
    parser = argparse.ArgumentParser()
    add_dict_to_argparser(parser, defaults)
    #parser.add_argument("--beta_values_file", type=str, help="Path to .npy file containing beta values for the diffusion model.")
    return parser

if __name__ == "__main__":
    main()




"""

"""
#Train a diffusion model on images.
"""

import argparse

from improved_diffusion import dist_util, logger
from improved_diffusion.image_datasets import load_data
from improved_diffusion.resample import create_named_schedule_sampler
from improved_diffusion.script_util import (
    model_and_diffusion_defaults,
    create_model_and_diffusion,
    args_to_dict,
    add_dict_to_argparser,
)
from improved_diffusion.train_util import TrainLoop


def main():
    args = create_argparser().parse_args()

    dist_util.setup_dist()
    logger.configure()

    logger.log("creating model and diffusion...")
    model, diffusion = create_model_and_diffusion(
        **args_to_dict(args, model_and_diffusion_defaults().keys())
    )
    model.to(dist_util.dev())
    schedule_sampler = create_named_schedule_sampler(args.schedule_sampler, diffusion)


    logger.log("creating data loader...")
    data = load_data(
        data_dir=args.data_dir,
        batch_size=args.batch_size,
        image_size=args.image_size,
        class_cond=args.class_cond,
    )

    logger.log("training...")
    TrainLoop(
        model=model,
        diffusion=diffusion,
        data=data,
        batch_size=args.batch_size,
        microbatch=args.microbatch,
        lr=args.lr,
        ema_rate=args.ema_rate,
        log_interval=args.log_interval,
        save_interval= args.save_interval,
        resume_checkpoint=args.resume_checkpoint,
        use_fp16=args.use_fp16,
        fp16_scale_growth=args.fp16_scale_growth,
        schedule_sampler=schedule_sampler,
        weight_decay=args.weight_decay,
        lr_anneal_steps=args.lr_anneal_steps,
    ).run_loop()



def create_argparser():
    defaults = dict(
        data_dir="",
        schedule_sampler="uniform",
        lr=1e-4,
        weight_decay=0.0,
        lr_anneal_steps=0,
        batch_size=1,
        microbatch=-1,  # -1 disables microbatches
        ema_rate="0.9999",  # comma-separated list of EMA values
        log_interval=10,
        save_interval=1000,
        resume_checkpoint="",
        use_fp16=False,
        fp16_scale_growth=1e-3,
    )
    defaults.update(model_and_diffusion_defaults())
    parser = argparse.ArgumentParser()
    add_dict_to_argparser(parser, defaults)
    return parser

if __name__ == "__main__":
    main()
"""