import random
from os import path
import os
import numpy as np
import argparse


def write_votes(votes, filename):
    """
    Write the votes to the given file.

    Parameters
    ----------
    votes : list[set[int]]
        The votes to write.
    filename : os.PathLike
        The file to write the votes on.
    """
    with open(filename, "w") as f:
        alts = set([x for vote in votes for x in vote])
        voter_str = ""
        i = 0
        for vote in votes:
            # We do not write empty voters
            if vote:
                voter_str += "%d : %s\n" % (i, vote_str(vote))
                i += 1
        f.write("voters %d, alternatives %d\n" % (i, len(alts)))
        f.write("alternatives: ")
        f.write(str(sorted(list(alts))).replace("'", ""))
        f.write("\n")
        f.write(voter_str)


def p_id_f_resampling(n, m, p, f, seed=None):
    """
    Create votes with p-phi-resampling. See for example the Experiments-section from the paper.

    Parameters
    ----------
    n : int
        The number of voters.
    m : int
        The number of alternatives.
    p : float
        The value of p.
    f : float
        The value of phi.
    seed : int
        The random seed, if None, doesn't seed the random generator.

    Returns
    -------
    list[set[int]]
        The generated votes.
    """
    if seed is not None:
        random.seed(seed)
        print("Seeded", seed)
    # True/false array about whether an alternative is in the initial vote
    init_vote = random.sample(list(range(m)), int(p * m))
    init_vote = [(x in init_vote) for x in range(m)]
    voters = []
    for voter in range(n):
        vote = set()
        for alt in range(m):
            if random.random() < f:
                if random.random() < p:
                    vote.add(alt)
            else:
                if init_vote[alt]:
                    vote.add(alt)
        voters.append(vote)
    return voters

def vote_str(vote):
    """
    Turns the vote into a string, with the voters sorted and surrounded with brackets.

    Parameters
    ----------
    vote : iterator[int]
        The vote.

    Returns
    -------
    str
        A string with the voters.
    """
    return str(sorted(list(vote))).replace("[", "{").replace("]", "}")


def geometric_preferences(n, m, d, p, rad, seed=None, verbose=False):
    """
    Create votes based on geometric preferences. See for example the Experiments-section from the paper.

    Parameters
    ----------
    n : int
        The number of voters.
    m : int
        The number of alternatives.
    d : int
        The dimension of the space.
    p : int
        The p-norm used to compute the distances.
    rad : float
        Approval radius, everything within this distance is approved.
    seed : int
        The random seed, if None, doesn't seed the random generator.
    verbose : bool
        If true, print alternative and voter locations.

    Returns
    -------
    list[set[int]]
        The generated votes.
    """
    if seed is not None:
        random.seed(seed)
        print("Seeded", seed)
    voterlocs = np.random.rand(n, d)
    altlocs = np.random.rand(m, d)
    if verbose:
        for i in range(n):
            print("voter", i, voterlocs[i])
        for i in range(m):
            print("alt", i, altlocs[i])
    voters = []
    for i in range(n):
        voter = set()
        for j in range(m):
            dist = sum([(abs(voterlocs[i, k] - altlocs[j, k])) ** p for k in range(d)])
            if dist <= rad ** p:
                voter.add(j)
        if voter:
            voters.append(voter)
    return voters


if __name__ == "__main__":
    # python3 instance_creator.py --n=100 --m=100 --instance_nro=100 --ps 0.01 0.05 0.1 0.15 --fs 0.25 0.5 0.75 1 --pfresampling=True --output_dir=data_big/synthetic/
    parser = argparse.ArgumentParser(description='Create instances')
    parser.add_argument("--n", type=int, help="The number of voters.")
    parser.add_argument("--m", type=int, help="The number of alternatives.")
    parser.add_argument("--instance_nro", type=int, help="The number of instances.")
    parser.add_argument("--radi", type=float, nargs="*", help="Radius for geometric preferences.")
    parser.add_argument("--ps", type=float, nargs="*", help="Values of p for p-phi-resampling.")
    parser.add_argument("--fs", type=float, nargs="*", help="Values of phi for p-phi resampling.")
    parser.add_argument("--manhattan", type=bool, default=False, help="Create manhattan votes.")
    parser.add_argument("--pfresampling", type=bool, default=False, help="Create p-phi-resampling votes.")
    parser.add_argument("--output_dir", type=str, default="",
                        help="The directory where to write files. Will not write votes if none is provided.")
    args = parser.parse_args()
    n = args.n
    m = args.m
    instance_nro = args.instance_nro
    output_dir = args.output_dir
    if not output_dir:
        print("Warning, no output directory provided")
    if args.manhattan:
        print("manhattan")
        file_str = ""
        radi = args.radi
        for r in radi:
            if output_dir:
                folderend = ("manhattan_r%.2f" % r).replace(".", "_")
                folder = path.join(output_dir, folderend)
                file_str += folderend + " "
                os.makedirs(folder, exist_ok=True)
            for i in range(instance_nro):
                votes = geometric_preferences(n, m, d=2, p=1, rad=r)
                if output_dir:
                    file = path.join(folder, "%d.txt" % i)
                    write_votes(votes, file)
                else:
                    print(votes)
        print(file_str)
    if args.pfresampling:
        print("p-f-resampling")
        file_str = ""
        for p in args.ps:
            for f in args.fs:
                if output_dir:
                    folderend = ("pfresampling_p%.2f_f%.2f" % (p, f)).replace(".", "_")
                    folder = path.join(output_dir, folderend)
                    file_str += folderend + " "
                    os.makedirs(folder, exist_ok=True)
                for i in range(instance_nro):
                    votes = p_id_f_resampling(n, m, p, f)
                    if output_dir:
                        file = path.join(folder, "%d.txt" % i)
                        write_votes(votes, file)
                    else:
                        print(votes)
        print(file_str)
