import re
import numpy as np
import pandas as pd
from scipy.optimize import curve_fit

# Function to extract data from the text file
def extract_data(file_path):
    pattern = r".*?interto(\d+).*?_p(\d+)_.*?mse:(\d*\.\d+)"
    data = []
    with open(file_path, 'r') as file:
        text = file.read()
    result = re.findall(pattern, text, re.DOTALL)
    data = [(int(m[0]), int(m[1]), float(m[2])) for m in result]
    print(data)
    return pd.DataFrame(data, columns=["horizon", "data_percent", "mse"])

# Regression model
def mse_model(horizon_and_data_percent, X, Y, Z, W, C,D, alpha, beta,gamma):
    horizon,data_percent = horizon_and_data_percent
    term1 = X * ((horizon/ 336) ** beta) ** 2
    term2 = Y / ((data_percent / 100)**gamma) * (horizon/ 336) ** beta
    term3 = Z / ((data_percent / 100)**gamma) * (horizon/ 336) ** (beta * (1 - alpha))
    term4 = W * (horizon/ 336) ** (-beta * alpha)
    term5 = D/(horizon/336)
    return term1 + term2 + term3 + term4 + term5 + C

# Function to fit the model to the data
def fit_model(data):
    horizons = data['horizon'].values
    data_percents = data['data_percent'].values
    mses = data['mse'].values

    # Initial parameter guesses
    initial_params = [0.2, 0.2, 0.2, 0.2, 0.2,0.2, 1.5, 0.8,1.0]

    # Fit the model
    params, covariance = curve_fit(mse_model, (horizons, data_percents), mses, p0=initial_params)
    
    # Calculate standard errors
    std_errors = np.sqrt(np.diag(covariance))
    
     # Calculate R^2
    residuals = mses - mse_model((horizons, data_percents), *params)
    ss_res = np.sum(residuals**2)
    ss_tot = np.sum((mses - np.mean(mses))**2)
    r_squared = 1 - (ss_res / ss_tot)
    
    return params, std_errors, r_squared

# Main function
def main(file_path):
    data = extract_data(file_path)
    params, std_errors, r_squared = fit_model(data)
    param_names = ['X', 'Y', 'Z', 'W', 'C', 'D','alpha', 'beta','gamma']
    results = dict(zip(param_names, params))
    errors = dict(zip(param_names, std_errors))

    print("Fitted Parameters:")
    for name, value in results.items():
        print(f"{name}: {value:.6f}")
    
    print("\nStandard Errors:")
    for name, error in errors.items():
        print(f"{name}: {error:.6f}")
        
    print(f"\nR^2: {r_squared:.6f}")

if __name__ == "__main__":
    file_path = "newresult_ETTh1_horizon.txt"  # Replace with the actual path to your text file
    main(file_path)
