import functools
import inspect
from collections.abc import Callable
from typing import Concatenate

from docstring_parser import parse
from langchain_core.tools import StructuredTool
from pydantic.v1 import BaseModel as BaseModelV1
from pydantic.v1 import Field as FieldV1
from pydantic.v1 import create_model as create_model_v1

from agentgym.functions_engine.functions_engine import (
    Env,
    P,
    T,
    _parse_args,
    get_available_tools_callables,
)
from agentgym.types import TaskEnvironment


def make_langchain_tool(function: Callable[Concatenate[Env, P], T], env: Env) -> StructuredTool:
    """Creates a langchain-compatible tool. It takes care of injecting the environment."""

    @functools.wraps(function)
    def tool_wrapper(*args: P.args, **kwargs: P.kwargs) -> T:
        return function(env, *args, **kwargs)

    name = function.__name__
    if function.__doc__ is None:
        raise ValueError(f"Function {name} has no docstring")

    description = parse(function.__doc__).short_description
    schema: type[BaseModelV1] = _parse_args(
        name,
        parse(function.__doc__).params,
        inspect.signature(function),
        field_fn=FieldV1,
        create_model_fn=create_model_v1,
    )

    return StructuredTool.from_function(
        tool_wrapper,
        name=name,
        infer_schema=False,
        args_schema=schema,
        description=description,
    )


def make_langchain_tools(env: TaskEnvironment, tools: list[str]) -> list[StructuredTool]:
    langchain_tools = []
    for tool in get_available_tools_callables(tools):
        langchain_tool = make_langchain_tool(tool, env)
        langchain_tools.append(langchain_tool)
    return langchain_tools
