try:
    # ignore ShapelyDeprecationWarning from fvcore
    from shapely.errors import ShapelyDeprecationWarning
    import warnings
    warnings.filterwarnings('ignore', category=ShapelyDeprecationWarning)
except:
    pass

import copy
import itertools
import logging
import os

from collections import OrderedDict
from typing import Any, Dict, List, Set

import torch

import detectron2.utils.comm as comm
from detectron2.checkpoint import DetectionCheckpointer
from detectron2.config import get_cfg
from detectron2.data import MetadataCatalog, build_detection_train_loader
from detectron2.engine import (
    DefaultTrainer,
    default_argument_parser,
    default_setup,
    launch,
)
from detectron2.evaluation import (
    DatasetEvaluators,
    LVISEvaluator,
    SemSegEvaluator,
    verify_results,
    
)
from detectron2.projects.deeplab import add_deeplab_config, build_lr_scheduler
from detectron2.solver.build import maybe_add_gradient_clipping
from detectron2.utils.logger import setup_logger


from auto_uni_seg import (
    SemanticDatasetMapper,
    add_hrnet_config,
    add_gnn_config,
    LoaderAdapter,
    build_bipartite_graph_for_unseen,
    eval_for_mseg_datasets,
    UniDetLearnUnifyLabelSpace
)

from PIL import Image
from detectron2.utils.file_io import PathManager
import numpy as np
from functools import partial
from detectron2.structures import ImageList
import torch.nn.functional as F
import logging

from auto_uni_seg.utils.evaluate import eval_link_hook, iter_info_hook, find_unuse_hook, print_unify_label_space


logger = logging.getLogger(__name__)
def my_sem_seg_loading_fn(filename, dtype=int, lb_map=None, size_divisibility=-1, ignore_label=255):
    with PathManager.open(filename, "rb") as f:
        image = np.array(Image.open(f), copy=False, dtype=dtype)
        if lb_map is not None:
            image = lb_map[image] 

    #     logger.info(f'size_divisibility: {size_divisibility}')
    #     if size_divisibility > 0:
    #         image = torch.tensor(image)
            
    #         image_size = (image.shape[0], image.shape[1])
    #         padding_size = [
    #             0,
    #             size_divisibility - image_size[1],
    #             0,
    #             size_divisibility - image_size[0],
    #         ]
            
    #         image = F.pad(image, padding_size, value=ignore_label).contiguous()
    #         logger.info(f'image shape: {image.shape}')
    #         image = image.numpy()

    # dsaf
    return image
    


class Trainer(DefaultTrainer):

    @classmethod
    def build_evaluator(cls, cfg, dataset_name, output_folder=None):
        """
        Create evaluator(s) for a given dataset.
        This uses the special metadata "evaluator_type" associated with each
        builtin dataset. For your own dataset, you can simply create an
        evaluator manually in your script and do not have to worry about the
        hacky if-else logic here.
        """
        logger.info(f"build evaluator:{dataset_name}")
        if output_folder is None:
            output_folder = os.path.join(cfg.OUTPUT_DIR, "inference")
        evaluator_list = []
        evaluator_type = MetadataCatalog.get(dataset_name).evaluator_type
        # semantic segmentation
        if evaluator_type in ["sem_seg", ]:
            lb_map = np.arange(256).astype(np.uint8)
            lookup_table = MetadataCatalog.get(dataset_name).thing_dataset_id_to_contiguous_id
            for k, v in lookup_table.items():
                lb_map[k] = v
            logger.info(f"evaluator_type:{dataset_name}")
            evaluator_list.append(
                SemSegEvaluator(
                    dataset_name,
                    distributed=True,
                    output_dir=output_folder,
                    sem_seg_loading_fn=partial(my_sem_seg_loading_fn, lb_map=lb_map, size_divisibility=cfg.INPUT.SIZE_DIVISIBILITY, ignore_label=cfg.DATASETS.IGNORE_LB)
                )
            )
        if len(evaluator_list) == 0:
            raise NotImplementedError(
                "no Evaluator for the dataset {} with the type {}".format(
                    dataset_name, evaluator_type
                )
            )
        elif len(evaluator_list) == 1:
            return evaluator_list[0]
        return DatasetEvaluators(evaluator_list)

    @classmethod
    def build_train_loader(cls, cfg):
        # Semantic segmentation dataset mapper
        if cfg.INPUT.DATASET_MAPPER_NAME == 'BASE':
            return LoaderAdapter(cfg, aux_mode='train')
        else:
            mapper = None
            return build_detection_train_loader(cfg, mapper=mapper)

    @classmethod
    def build_lr_scheduler(cls, cfg, optimizer):
        """
        It now calls :func:`detectron2.solver.build_lr_scheduler`.
        Overwrite it if you'd like a different scheduler.
        """
        return build_lr_scheduler(cfg, optimizer)

    @classmethod
    def build_test_loader(cls, cfg, dataset_name):
        if 'cs' in dataset_name:
            dataset_id = 0            
        elif 'mapi' in dataset_name:
            dataset_id = 1
        elif 'sunrgbd' in dataset_name:
            dataset_id = 2
        elif 'bdd' in dataset_name:
            dataset_id = 3
        elif 'idd' in dataset_name:
            dataset_id = 4
        elif 'ade' in dataset_name:
            dataset_id = 5
        elif 'coco' in dataset_name:
            dataset_id = 6
        else:
            dataset_id = 0
        aux_mode = 'test'
        if '_2' in dataset_name:
            aux_mode = 'eval'
            
        return LoaderAdapter(cfg, aux_mode=aux_mode, dataset_id=dataset_id, datasets_name=[dataset_name])
    @classmethod
    def build_optimizer(cls, cfg, model):
        weight_decay_norm = cfg.SOLVER.WEIGHT_DECAY_NORM
        weight_decay_embed = cfg.SOLVER.WEIGHT_DECAY_EMBED

        defaults = {}
        defaults["lr"] = cfg.SOLVER.BASE_LR
        defaults["weight_decay"] = cfg.SOLVER.WEIGHT_DECAY

        norm_module_types = (
            torch.nn.BatchNorm1d,
            torch.nn.BatchNorm2d,
            torch.nn.BatchNorm3d,
            torch.nn.SyncBatchNorm,
            # NaiveSyncBatchNorm inherits from BatchNorm2d
            torch.nn.GroupNorm,
            torch.nn.InstanceNorm1d,
            torch.nn.InstanceNorm2d,
            torch.nn.InstanceNorm3d,
            torch.nn.LayerNorm,
            torch.nn.LocalResponseNorm,
        )

        params: List[Dict[str, Any]] = []
        memo: Set[torch.nn.parameter.Parameter] = set()
        for module_name, module in model.named_modules():
            for module_param_name, value in module.named_parameters(recurse=False):
                if not value.requires_grad and 'adj_matrix' not in module_param_name:
                    continue
                # Avoid duplicating parameters
                if value in memo:
                    continue
                memo.add(value)

                hyperparams = copy.copy(defaults)
                if "backbone" in module_name:
                    hyperparams["lr"] = hyperparams["lr"] * cfg.SOLVER.BACKBONE_MULTIPLIER
                if (
                    "relative_position_bias_table" in module_param_name
                    or "absolute_pos_embed" in module_param_name
                ):
                    print(module_param_name)
                    hyperparams["weight_decay"] = 0.0
                if isinstance(module, norm_module_types):
                    hyperparams["weight_decay"] = weight_decay_norm
                if isinstance(module, torch.nn.Embedding):
                    hyperparams["weight_decay"] = weight_decay_embed
                params.append({"params": [value], **hyperparams})

        def maybe_add_full_model_gradient_clipping(optim):
            # detectron2 doesn't have full model gradient clipping now
            clip_norm_val = cfg.SOLVER.CLIP_GRADIENTS.CLIP_VALUE
            enable = (
                cfg.SOLVER.CLIP_GRADIENTS.ENABLED
                and cfg.SOLVER.CLIP_GRADIENTS.CLIP_TYPE == "full_model"
                and clip_norm_val > 0.0
            )

            class FullModelGradientClippingOptimizer(optim):
                def step(self, closure=None):
                    all_params = itertools.chain(*[x["params"] for x in self.param_groups])
                    torch.nn.utils.clip_grad_norm_(all_params, clip_norm_val)
                    super().step(closure=closure)

            return FullModelGradientClippingOptimizer if enable else optim

        optimizer_type = cfg.SOLVER.OPTIMIZER
        if optimizer_type == "SGD":
            optimizer = maybe_add_full_model_gradient_clipping(torch.optim.SGD)(
                params, cfg.SOLVER.BASE_LR, momentum=cfg.SOLVER.MOMENTUM
            )
        elif optimizer_type == "ADAMW":
            optimizer = maybe_add_full_model_gradient_clipping(torch.optim.AdamW)(
                params, cfg.SOLVER.BASE_LR
            )
        else:
            raise NotImplementedError(f"no optimizer type {optimizer_type}")
        if not cfg.SOLVER.CLIP_GRADIENTS.CLIP_TYPE == "full_model":
            optimizer = maybe_add_gradient_clipping(cfg, optimizer)
        return optimizer

    @classmethod
    def test_with_TTA(cls, cfg, model):
        logger = logging.getLogger("detectron2.trainer")
        # In the end of training, run an evaluation with TTA.
        logger.info("Running inference with test-time augmentation ...")
        model = SemanticSegmentorWithTTA(cfg, model)
        evaluators = [
            cls.build_evaluator(
                cfg, name, output_folder=os.path.join(cfg.OUTPUT_DIR, "inference_TTA")
            )
            for name in cfg.DATASETS.TEST
        ]
        res = cls.test(cfg, model, evaluators)
        res = OrderedDict({k + "_TTA": v for k, v in res.items()})
        return res


def setup(args):
    """
    Create configs and perform basic setups.
    """
    cfg = get_cfg()
    # for poly lr schedule
    add_deeplab_config(cfg)
    add_hrnet_config(cfg)
    add_gnn_config(cfg)
    cfg.merge_from_file(args.config_file)
    cfg.merge_from_list(args.opts)
    cfg.freeze()
    default_setup(cfg, args)
    setup_logger(output=cfg.OUTPUT_DIR, distributed_rank=comm.get_rank(), name="auto_uni_seg")
    return cfg

def build_bipart_for_unseen(cfg, model):
    """
    Build bipartite graph for unseen classes.
    """
    from auto_uni_seg.utils import build_bipartite_graph_for_unseen
    build_bipartite_graph_for_unseen(cfg, model)
    

def main(args):
    cfg = setup(args)
    
    if args.eval_only:
        model = Trainer.build_model(cfg)
        DetectionCheckpointer(model, save_dir=cfg.OUTPUT_DIR).resume_or_load(
            cfg.MODEL.WEIGHTS, resume=args.resume
        )
        # eval_for_mseg_datasets(Trainer.build_test_loader, cfg, model)
        if args.unseen:
            build_bipartite_graph_for_unseen(Trainer.build_test_loader, cfg, model)
        # print_unify_label_space(Trainer.build_test_loader, model, cfg)
        # return
        res = Trainer.test(cfg, model)

        if comm.is_main_process():
            verify_results(cfg, res)
        return res
        # return
    
    trainer = Trainer(cfg)
    trainer.register_hooks([iter_info_hook(), UniDetLearnUnifyLabelSpace()])
    trainer.resume_or_load(resume=args.resume)
    return trainer.train()

def argument_parser():
    parser = default_argument_parser()
    parser.add_argument(
        "--unseen",
        action="store_true",
        help="Whether to evaluate unseen datasets. ",
    )
    return parser


if __name__ == "__main__":
    args = argument_parser().parse_args()
    print("Command Line Args:", args)
    launch(
        main,
        args.num_gpus,
        num_machines=args.num_machines,
        machine_rank=args.machine_rank,
        dist_url=args.dist_url,
        args=(args,),
    )
