import asyncio
import logging
import os
import re
from typing import Optional, Protocol

import attrs
from anthropic import AsyncAnthropic
from anthropic.types.completion import Completion as AnthropicCompletion
from termcolor import cprint

from sandbagging.utils.llm_api.base_llm import PRINT_COLORS, LLMResponse, ModelAPIProtocol

ANTHROPIC_MODELS = {"claude-instant-1", "claude-2", "claude-v1.3"}


def count_tokens(prompt: str) -> int:
    return len(prompt.split())


def price_per_token(model_id: str) -> tuple[float, float]:
    """
    Returns the (input token, output token) price for the given model id.
    """
    return 0, 0


@attrs.define()
class AnthropicChatModel(ModelAPIProtocol):
    num_threads: int
    client: AsyncAnthropic = attrs.field(init=False, default=attrs.Factory(AsyncAnthropic))
    available_requests: asyncio.BoundedSemaphore = attrs.field(init=False)

    def __attrs_post_init__(self):
        self.available_requests = asyncio.BoundedSemaphore(self.num_threads)

    async def __call__(
        self, model_ids: list[str], prompt: str, print_prompt_and_response: bool, max_attempts: int, **kwargs
    ) -> list[LLMResponse]:
        assert len(model_ids) == 1, "Anthropic implementation only supports one model at a time."
        model_id = model_ids[0]

        response: Optional[AnthropicCompletion] = None
        for i in range(max_attempts):
            try:
                async with self.available_requests:
                    response = await self.client.completions.create(prompt=prompt, model=model_id, **kwargs)

            except Exception as e:
                logging.warn(f"{model_id} encountered API error: {str(e)}.\nRetrying now. (Attempt {i})")
                await asyncio.sleep(1.5**i)
            else:
                break

        if response is None:
            raise RuntimeError(f"Failed to get a response from the API after {max_attempts} attempts.")

        num_context_tokens, num_completion_tokens = await asyncio.gather(
            self.client.count_tokens(prompt), self.client.count_tokens(response.completion)
        )
        context_token_cost, completion_token_cost = price_per_token(model_id)
        cost = num_context_tokens * context_token_cost + num_completion_tokens * completion_token_cost

        llm_response = LLMResponse(
            model_id=model_id, completion=response.completion, stop_reason=response.stop_reason, cost=cost
        )

        if print_prompt_and_response:
            pattern = r"\n\n((Human: |Assistant: ).*)"
            for match in re.finditer(pattern, prompt):
                role = match.group(2).removesuffix(": ").lower()
                role = {"human": "user"}.get(role, role)
                cprint(match.group(1), PRINT_COLORS[role])
            cprint(f"Response ({llm_response.model_id}):", "white")
            cprint(f"{llm_response.completion}", PRINT_COLORS["assistant"], attrs=["bold"])
            print()

        return [llm_response]
