import pandas as pd
from datasets import load_dataset
from collections import defaultdict

from sandbagging.dataset_utils import (
    SplitGetterConfig,
    make_split_getter_for_test_exclusive_splitting,
    make_splits_for_fracs,
)
from sandbagging.extract_math_answer import extract_math_answer
from sandbagging.pb_data_types import PbsSplits, SettingData
from sandbagging.dataset_wrapping import cache_dir
from sandbagging.setting_types import Pbs, add_dict_field, from_json, to_json
from sandbagging.math_dataset_info import math_type_to_category, math_level_to_difficulty
from pathlib import Path
import json

# %%


def add_basic_fields_and_standardize(df: pd.DataFrame) -> pd.DataFrame:
    df = df.copy()

    df["category"] = [math_type_to_category[x] for x in df["type"].values]
    df["difficulty"] = [math_level_to_difficulty[x] for x in df["level"].values]

    df.drop(columns=["type", "level"], inplace=True)

    df["evaluation_data"] = [{}] * len(df)
    df["metadata"] = [{}] * len(df)
    df = df.assign(setting="math", id=0)
    add_dict_field(df, "metadata", "solution", df["solution"].values)

    extracted_answers = [
        extract_math_answer(problem, s) for problem, s in zip(df["problem"].values, df["solution"].values)
    ]

    add_dict_field(df, "evaluation_data", "extracted_answer", extracted_answers)

    df.drop(columns=["solution"], inplace=True)

    return Pbs.from_df(df)


# %%

train_save_path = f"{cache_dir}/math_train.json"
test_save_path = f"{cache_dir}/math_test.json"
if Path(train_save_path).exists() and Path(test_save_path).exists():
    math_train = from_json(json.loads(Path(train_save_path).read_text()), Pbs)
    math_test = from_json(json.loads(Path(test_save_path).read_text()), Pbs)
else:
    math_train_raw: pd.DataFrame = load_dataset(
        "hendrycks/competition_math", "all", split="train", trust_remote_code=True
    ).to_pandas()
    math_test_raw: pd.DataFrame = load_dataset(
        "hendrycks/competition_math", "all", split="test", trust_remote_code=True
    ).to_pandas()

    math_train = add_basic_fields_and_standardize(math_train_raw)
    math_test = add_basic_fields_and_standardize(math_test_raw)

    Path(train_save_path).write_text(json.dumps(to_json(math_train)))
    Path(test_save_path).write_text(json.dumps(to_json(math_test)))

all_math = Pbs.from_df(pd.concat([math_train, math_test]).reset_index(drop=True))

# %%


split_config = SplitGetterConfig()  # defaults for now

get_splits = make_split_getter_for_test_exclusive_splitting(train=math_train, test=math_test, config=split_config)

math_setting_data = SettingData(get_splits=get_splits)

# %%

train_sa_save_path = f"{cache_dir}/math_sa_train.json"
test_sa_save_path = f"{cache_dir}/math_sa_test.json"
inference_file = f"{cache_dir}/math_spanish_ascii_gold_inference/inference_outputs.json"
if Path(train_sa_save_path).exists() and Path(test_sa_save_path).exists() and Path(inference_file).exists():
    math_sa_train = from_json(json.loads(Path(train_sa_save_path).read_text()), Pbs)
    math_sa_test = from_json(json.loads(Path(test_sa_save_path).read_text()), Pbs)
else:
    path = Path("~/remote_fs/user/translated_outputs.json").expanduser()
    data = json.loads(path.read_text())

    print(len(data), len(set([d["fully_translated_problem"] for d in data])), len(set([d["problem"] for d in data])))

    # hack to make translated problems unique (there are 8 duplicates / 12,500 problems)
    indexed_by_translated_pb = defaultdict(list)
    for d in data:
        indexed_by_translated_pb[d["fully_translated_problem"]].append(d)
    for k, v in indexed_by_translated_pb.items():
        if len(v) > 1:
            print(k, len(v))
            for i, d in enumerate(v):
                d["fully_translated_problem"] = d["fully_translated_problem"] + "." * i

    print(len(data), len(set([d["fully_translated_problem"] for d in data])), len(set([d["problem"] for d in data])))

    Path(inference_file).parent.mkdir(exist_ok=True, parents=True)
    Path(inference_file).write_text(
        json.dumps(
            [
                {"problem": d["fully_translated_problem"], "output": d["fully_translated_solution"], "id": 0}
                for d in data
            ]
        )
    )

    indexed_by_pb = {data["problem"]: data for data in data}

    def translate_to_math_ascii(df):
        df = df.copy()

        df_problem_copy = df["problem"].copy()

        df["problem"] = [indexed_by_pb[pb]["fully_translated_problem"] for pb in df_problem_copy]
        df["evaluation_data"] = [
            {
                "extracted_answer": extract_math_answer(
                    pb, indexed_by_pb[pb]["fully_translated_solution"], spanish_and_ascii_model=True
                )
            }
            for pb in df_problem_copy
        ]
        df["metadata"] = [
            {
                "solution": indexed_by_pb[pb]["fully_translated_solution"],
                "pre_translation_solution": indexed_by_pb[pb]["output"],
                "pre_translation_problem": indexed_by_pb[pb]["problem"],
            }
            for pb in df_problem_copy
        ]
        df["setting"] = "math_spanish_ascii"

        return df

    math_sa_train = translate_to_math_ascii(math_train)
    math_sa_test = translate_to_math_ascii(math_test)

    Path(train_sa_save_path).write_text(json.dumps(to_json(math_sa_train)))
    Path(test_sa_save_path).write_text(json.dumps(to_json(math_sa_test)))

get_splits_sa = make_split_getter_for_test_exclusive_splitting(
    train=math_sa_train, test=math_sa_test, config=split_config
)

math_sa_setting_data = SettingData(get_splits=get_splits_sa)
# len(get_splits(283947).train_lock_split)
# %%
