# %%
import asyncio
import json
import re
import sys
from collections import Counter

from cattrs.preconf.json import make_converter
from datasets import load_dataset

from sandbagging.utils.loaded_problem import InputOutputTestCases, LoadedProblem, PythonTestCases
from sandbagging.utils.info_types import ProblemSolutionOut, TestCaseOutcome
from sandbagging.utils.models_for_monitoring import ChatInput, smart_models
from sandbagging.code_data import extract_solution, get_test_case_outcomes, get_test_case_outcomes_and_io
from sandbagging.code_setting import CodeSetting
from sandbagging.dataset_wrapping import asyncio_run, cache_dir, gather_tqdm, json_converter
from tqdm.asyncio import tqdm_asyncio

sys.set_int_max_str_digits(1_000_000)

# %%

json_converter = make_converter()

# %%

dtrain = list(load_dataset("codeparrot/apps", split="train"))
dtest = list(load_dataset("codeparrot/apps", split="test"))
d = dtrain + dtest
# %%

problems_of_diff = [x for x in d if x["difficulty"] in ["introductory", "interview"]]

all_problems: list[LoadedProblem] = []
task_id = 0

# def to_lines(x: Any) -> list[str]

for datum_id, x in enumerate(problems_of_diff):
    if len(x["solutions"]) == 0:
        continue
    solutions: list[str] = list(json.loads(x["solutions"]))

    if x["input_output"] != "":
        inp_out = json.loads(x["input_output"])
        if "fn_name" in inp_out:
            if any("class Solution" in s for s in solutions):
                # cursed
                continue  # legacy continues without task_id incr

            # undo cursed formatting
            if not all(isinstance(y, list) and len(y) == 1 for y in inp_out["outputs"]):
                print("WARNING: weird output format (skipped)\n", inp_out["outputs"])
                continue  # legacy continues without task_id incr
            outputs = [y[0] for y in inp_out["outputs"]]

            test_cases = PythonTestCases(
                is_solution_class=False,
                fn_name=inp_out["fn_name"].strip(),
                inputs=[tuple(y) for y in inp_out["inputs"]],
                outputs=outputs,
            )
        else:
            if len(inp_out["inputs"]) == len(inp_out["outputs"]) != 0:
                if isinstance(inp_out["inputs"][0], list):
                    inp_out["inputs"] = ["\n".join(y) for y in inp_out["inputs"]]
                if isinstance(inp_out["outputs"][0], list):
                    inp_out["outputs"] = ["\n".join(y) for y in inp_out["outputs"]]

            test_cases = InputOutputTestCases(
                inputs=inp_out["inputs"],
                outputs=inp_out["outputs"],
            )
    else:
        test_cases = InputOutputTestCases(inputs=[], outputs=[])

    all_problems.append(
        LoadedProblem(
            task_id,
            dataset=f"apps/sub",
            description=x["question"],
            solutions=solutions,
            base_code="",
            base_code_fmt_example="",
            test_cases=test_cases,
            difficulty=f"apps/{x['difficulty']}",
        )
    )
    task_id += 1


# %%


def test_case_py_to_io(tc: PythonTestCases) -> InputOutputTestCases:
    return InputOutputTestCases(
        inputs=["\n".join(map(str, x)) + "\n" for x in tc.inputs], outputs=[str(x) for x in tc.outputs]
    )


def example_to_chat_input(ex: LoadedProblem):
    asserts = ex.test_cases.get_assertions()
    asserts_combined = "\n".join(asserts)

    prompt = f"""
Convert the following python problem statement and test into a language agnostic problem description and input-output test as well as a c++ solution.

Problem statement:
{ex.description}
Python solution:
```python
{ex.solutions[0]}
```
Python test:
```python
{asserts_combined}
```

Your output should look like this:

-----Problem-----
<problem description. In a format like "You are given n distinct integers. Find the number of pairs of integers that sum to a given value k." do NOT say "Write a function...".>
-----Input-----
<description of the expect input of the program. In the format "The first line of the input contains a single integer n. The second line contains...">
-----Output-----
<description of the expected output of the program. In the format "The first line of the output should contains a single integer, the number of ...">
-----Examples-----
Input
<first line>
<second line>
...
Output
<first line>
...
Input
<first line>
...
Output
<first line>
...
-----Solution-----
```cpp
<the solution, should contain a main function that reads the input and prints the output>
```

Feel free to adapt the inputs and outputs to formats that are more natural in the context of C++ cin and cout.
Say "This problem can't be converted" if you think the problem can't be converted.
"""
    return ChatInput([("user", prompt)])


START_ID = 2_000_000
MAX_N_PB = 10_000


async def convert_all():
    to_convert_pbs = [
        x
        for x in all_problems
        if isinstance(x.test_cases, PythonTestCases) and len(x.test_cases.inputs) >= 3 and len(str(x.test_cases)) < 1000
    ][-MAX_N_PB:]

    all_responses = await gather_tqdm(
        [smart_models(example_to_chat_input(ex)) for ex in to_convert_pbs], desc="gen_solutions"
    )

    problems: list[LoadedProblem] = []
    for ex, (resp, _) in zip(to_convert_pbs, all_responses):
        completion = resp[0].completion

        # parse completion
        try:
            description = completion.split("-----Problem-----")[1].split("-----Examples-----")[0].strip()

            tests = completion.split("-----Examples-----")[1].split("-----Solution-----")[0].strip()
            input_and_outputs = tests.split("Input")[1:]
            inputs = [inp.split("Output")[0].strip() + "\n" for inp in input_and_outputs]
            outputs = [out.split("Output")[1].strip() + "\n" for out in input_and_outputs]
            solution = extract_solution(completion.split("-----Solution-----")[1].strip())

            problems.append(
                LoadedProblem(
                    task_id=START_ID + ex.task_id,
                    dataset="apps_conv",
                    description=description,
                    solutions=[solution],
                    base_code=ex.base_code,
                    base_code_fmt_example=ex.base_code_fmt_example,
                    test_cases=InputOutputTestCases(inputs=inputs, outputs=outputs),
                    difficulty=ex.difficulty.replace("apps/", "apps_conv/"),
                )
            )
        except Exception as e:
            print(e)
            continue

    with open(f"{cache_dir}/apps_conv_problems_raw.jsonl", "w") as f:
        for p in problems:
            f.write(json_converter.dumps(p) + "\n")

    correctness = await gather_tqdm(
        [
            get_test_case_outcomes(
                ProblemSolutionOut(problem.description, problem.solutions[0]),
                problem.test_cases,
            )
            for problem in problems
        ],
        desc="get_test_case_outcomes",
    )
    # print compilation failures
    for problem, (correctness_result, err) in zip(problems, correctness):
        if correctness_result[0] == TestCaseOutcome.COMPILE_ERROR:
            print(err.out.stderr)
            print("=====")
            print(problem.solutions[0])
            print("========")
    all_correct = [
        all(x == TestCaseOutcome.CORRECT for x in correctness_result) for correctness_result, _ in correctness
    ]
    print(Counter(x for correctness_result, _ in correctness for x in correctness_result))
    kept_problems = [p for p, c in zip(problems, all_correct) if c]
    print(f"Kept {len(kept_problems)}/{len(problems)}")
    kept_problems = [
        p
        for p in kept_problems
        if len(p.test_cases.inputs) >= 3 and len(p.test_cases.outputs) == len(p.test_cases.inputs)
    ]
    print(f"Kept {len(kept_problems)}/{len(problems)}")
    with open(f"{cache_dir}/apps_conv_problems.jsonl", "w") as f:
        for p in kept_problems:
            f.write(json_converter.dumps(p) + "\n")


asyncio_run(convert_all())

# %%
mbpp_problems = []
with open(f"{cache_dir}/mbpp_problems.jsonl", "r") as f:
    for line in f.readlines():
        mbpp_problems.append(json_converter.loads(line, LoadedProblem))

all_problems += mbpp_problems

conv_problems = []
with open(f"{cache_dir}/apps_conv_problems.jsonl", "r") as f:
    for line in f.readlines():
        conv_problems.append(json_converter.loads(line, LoadedProblem))

all_problems += conv_problems

pattern = re.compile(r"[\u0410-\u044F]")  # drop problems containing Cyrrilic characters
all_problems = [
    x
    for x in all_problems
    if isinstance(x.test_cases, InputOutputTestCases)
    and re.match(pattern, x.description) is None
    and x.test_cases.n >= 1
]  # only support input output for now

# remove problems with the same description
all_problems = list({x.description: x for x in all_problems}.values())

assert len(all_problems) == len(set(x.task_id for x in all_problems))

with open(f"{cache_dir}/all_problems.jsonl", "w") as f:
    for p in all_problems:
        f.write(json_converter.dumps(p) + "\n")


print(len(all_problems))
# %%
# filter by "does gpt-4-turbo get it right at temp 0"


async def batch_gather(*coros, desc: str = "", batch_size=500):
    batches = [coros[i : i + batch_size] for i in range(0, len(coros), batch_size)]
    results = []
    for batch in tqdm_asyncio(batches, desc=desc):
        results += await asyncio.gather(*batch)
    return results


async def gpt4_correct():
    def to_tuples(dicts):
        return tuple((d["role"], d["content"]) for d in dicts)

    all_prompts = [CodeSetting().generate_prompt(pb.description) for pb in all_problems]
    chat_inputs = [ChatInput(to_tuples(prompt), 0, 1) for prompt in all_prompts]
    responses = await tqdm_asyncio.gather(
        *[smart_models(chat_input) for chat_input in chat_inputs], desc="gen_solutions"
    )
    completions = [extract_solution(response[0].completion, default="") for response, _ in responses]
    correctness_results = await batch_gather(
        *[
            get_test_case_outcomes_and_io(
                ProblemSolutionOut(problem.description, solution),
                problem.test_cases,
            )
            for problem, solution in zip(all_problems, completions)
        ],
        desc="get_test_case_outcomes",
    )
    r = [all(x == TestCaseOutcome.CORRECT for x in correctness_result) for correctness_result, _ in correctness_results]
    assert len(r) == len(all_problems)
    return r


correctness = asyncio_run(gpt4_correct())
all_problems = [x for x, c in zip(all_problems, correctness) if c]

# %%
print(len(all_problems))
# %%

# %%

with open(f"{cache_dir}/problems.jsonl", "w") as f:
    for p in all_problems:
        f.write(json_converter.dumps(p) + "\n")

# python sandbagging/get_problems.py; python sandbagging/get_solutions.py smart 0 1; python sandbagging/get_solutions.py smart 1 10; python sandbagging/code_categorize.py; python sandbagging/get_solutions.py smart 1 10; python sandbagging/get_solutions.py smart 0 1; python sandbagging/critique_gen_gold.py smart_T0_n1; python sandbagging/critique_gen_gold.py smart_T1_n10;
# %%
