from typing import Optional, Sequence, Union, Callable
from typing_extensions import assert_never

import pandas as pd
import attrs
import torch
from sandbagging.all_train import C_TARGETS_METADATA_KEY
from sandbagging.basic_model_info import ContinuousTarget
from sandbagging.passwords import Pwd

from sandbagging.setting_types import (
    FixedGeneration,
    OnTheFlyGeneration,
    PbAndGens,
    Pbs,
    StoredClassificationProbs,
    StoredGeneration,
    add_dict_field,
)
import sandbagging.utils.optional as op

NOOP_STR = "NOOP"


@attrs.frozen
class ClassificationUseCorrectAnswerLabel:
    ...


@attrs.frozen
class ClassificationCallableProbs:
    get_probs: Callable[[Pbs], list[list[float]]]


@attrs.frozen
class ClassificationUseSavedProbs:
    save_name_for_inference: str
    harden: bool = False
    harden_on_second_most_likely: bool = False

    def __attrs_post_init__(self):
        if self.harden_on_second_most_likely:
            assert self.harden, "Can't harden on second most likely if not hardening"


AllClassificationToImitate = Union[
    ClassificationUseCorrectAnswerLabel, ClassificationUseSavedProbs, ClassificationCallableProbs
]


@attrs.frozen
class ClassificationModelToImitateArgs:
    num_classes: int
    start_assistant_response: str
    to_imitate: AllClassificationToImitate

    @property
    def save_name_for_inference(self) -> Optional[str]:
        return (
            self.to_imitate.save_name_for_inference
            if isinstance(self.to_imitate, ClassificationUseSavedProbs)
            else None
        )


@attrs.frozen
class ModelToImitateArgs:
    model_path: Optional[str] = None
    save_name_for_inference: Optional[str] = None
    temperature: Optional[float] = None


NOOP_IMITATE_ARGS = ModelToImitateArgs(
    model_path=NOOP_STR,
    save_name_for_inference=NOOP_STR,
)

EitherModelToImitateArgs = Union[ModelToImitateArgs, ClassificationModelToImitateArgs]

# %%


def target_probs_from_choice_index(answers: Sequence[int], num_classes: int) -> list[list[int]]:
    # convert answers to one hot with torch
    return torch.nn.functional.one_hot(torch.tensor(answers), num_classes=num_classes).tolist()


# %%
# generation df modifications


def add_target_fixed_stored_generation(df: Pbs, save_name_for_inference: str) -> pd.DataFrame:
    gen = StoredGeneration(save_name_for_inference)
    return df.assign(target_generation=[attrs.evolve(gen, id=id) for id in df.id])


def add_target_gen_generation(
    df: Pbs,
    model_path: str,
    temperature: Optional[float] = None,
) -> pd.DataFrame:
    df = df.copy()

    final_temp = op.unwrap_or(temperature, 1.0)
    gen = OnTheFlyGeneration(model_path, temperature=final_temp)

    return df.assign(target_generation=[attrs.evolve(gen, id=id) for id in df.id])


def fill_out_target_generation_probs_classification(
    df: Pbs,
    to_imitate: ClassificationModelToImitateArgs,
):
    df = df.copy()  # below mutates
    if isinstance(to_imitate.to_imitate, ClassificationUseCorrectAnswerLabel):
        target_probs = target_probs_from_choice_index(
            [int(x["correct_answer"]) for x in df["metadata"]], num_classes=to_imitate.num_classes
        )
        add_dict_field(df, "metadata", "target_probs", target_probs)
    elif isinstance(to_imitate.to_imitate, ClassificationCallableProbs):
        add_dict_field(df, "metadata", "target_probs", to_imitate.to_imitate.get_probs(df))
    elif isinstance(to_imitate.to_imitate, ClassificationUseSavedProbs):
        add_dict_field(
            df,
            "metadata",
            "target_probs",
            [
                StoredClassificationProbs(
                    to_imitate.to_imitate.save_name_for_inference,
                    harden=to_imitate.to_imitate.harden,
                    harden_on_second_most_likely=to_imitate.to_imitate.harden_on_second_most_likely,
                )
            ]
            * len(df),
        )
    else:
        assert_never(to_imitate.to_imitate)

    return df.assign(target_generation=FixedGeneration(to_imitate.start_assistant_response))


def add_target_generation(
    df: Pbs,
    use_fixed: bool,
    model_to_imitate_args: EitherModelToImitateArgs,
):
    if isinstance(model_to_imitate_args, ClassificationModelToImitateArgs):
        return fill_out_target_generation_probs_classification(df, model_to_imitate_args)
    elif use_fixed:
        return add_target_fixed_stored_generation(df, op.unwrap(model_to_imitate_args.save_name_for_inference))
    else:
        return add_target_gen_generation(
            df, model_path=op.unwrap(model_to_imitate_args.model_path), temperature=model_to_imitate_args.temperature
        )


# %%


@attrs.frozen
class TrainToImitateArgs:
    model_args: EitherModelToImitateArgs
    has_strong_perf_pwd: Optional[bool]
    password: Pwd
    retain_probability: float = 1.0
    force_fixed: bool = False
    c_targets: ContinuousTarget = 0

    @classmethod
    def additional(
        cls,
        model_args: EitherModelToImitateArgs,
        password: Pwd,
        retain_probability: float = 1.0,
        force_fixed: bool = False,
    ):
        return cls(
            model_args,
            has_strong_perf_pwd=None,
            password=password,
            retain_probability=retain_probability,
            force_fixed=force_fixed,
        )


def fill_pbs_df(
    df: Pbs,
    to_imitate_args: TrainToImitateArgs,
    use_fixed: bool,
) -> PbAndGens:
    new_df = df.assign(
        has_strong_perf_pwd=to_imitate_args.has_strong_perf_pwd,
        password=to_imitate_args.password,
        retain_probability=to_imitate_args.retain_probability,
    )
    new_df = add_target_generation(
        new_df,
        use_fixed=use_fixed or to_imitate_args.force_fixed,
        model_to_imitate_args=to_imitate_args.model_args,
    )
    add_dict_field(new_df, "metadata", C_TARGETS_METADATA_KEY, [to_imitate_args.c_targets] * len(new_df))

    new_df = new_df.drop(columns=["id"])  # id is now in the target generation.

    return PbAndGens.from_df(new_df)


def apply_hardening(args: EitherModelToImitateArgs, harden: bool) -> EitherModelToImitateArgs:
    if isinstance(args, ClassificationModelToImitateArgs):
        if isinstance(args.to_imitate, ClassificationUseSavedProbs):
            # expect input to not be hardened yet
            assert not args.to_imitate.harden

    if not harden:
        return args

    return apply_fix_hardening(args, harden=harden)


def apply_fix_hardening(args: EitherModelToImitateArgs, harden: bool) -> EitherModelToImitateArgs:
    assert isinstance(args, ClassificationModelToImitateArgs)

    to_imitate = args.to_imitate

    if not isinstance(to_imitate, ClassificationUseSavedProbs):
        return args

    return attrs.evolve(args, to_imitate=attrs.evolve(to_imitate, harden=harden))


def handle_fill_out_dups(
    df: PbAndGens,
    model_args: EitherModelToImitateArgs,
):
    df = df.copy()
    any_in = False

    new_target_generations = []
    new_target_probs = []

    for i in range(len(df)):
        if "dup_idx" not in df.metadata[i]:
            new_target_generations.append(df.target_generation[i])
            if isinstance(model_args, ClassificationModelToImitateArgs):
                new_target_probs.append(df.metadata[i]["target_probs"])
            continue

        any_in = True
        dup_idx = df.metadata[i]["dup_idx"]

        if isinstance(model_args, ClassificationModelToImitateArgs):
            if isinstance(model_args.to_imitate, ClassificationUseSavedProbs):
                save_name = model_args.to_imitate.save_name_for_inference
                new_target_probs.append(StoredClassificationProbs(save_name=save_name, harden=True))
            elif isinstance(model_args.to_imitate, ClassificationUseCorrectAnswerLabel):
                # keeping same is fine
                new_target_probs.append(df.metadata[i]["target_probs"])
            else:
                assert_never(model_args.to_imitate)

            current_target_generation = df.target_generation[i]
            assert isinstance(current_target_generation, FixedGeneration)

            new_target_generations.append(attrs.evolve(current_target_generation, dup_idx=dup_idx))
        else:
            save_name = model_args.save_name_for_inference
            assert save_name is not None, "dups require save_name_for_inference"
            new_target_generations.append(StoredGeneration(save_name=save_name, dup_idx=dup_idx))

    assert len(new_target_generations) == len(df)
    df = df.assign(target_generation=new_target_generations)
    if isinstance(model_args, ClassificationModelToImitateArgs):
        assert len(new_target_probs) == len(df)
        add_dict_field(df, "metadata", "target_probs", new_target_probs)

    assert any_in, "no dups found"

    return df


def assert_no_dups(df: PbAndGens):
    for i in range(len(df)):
        assert "dup_idx" not in df.metadata[i], "dups found"
