import itertools
import json
import os
import random
import re
import subprocess
import time
from collections import Counter
from functools import cache
from pathlib import Path
from typing import Optional, TypedDict

import attrs
import numpy as np
import psutil

from sandbagging.utils.loaded_problem import InputOutputTestCases
from sandbagging.utils.info_types import ProblemSolutionOut, TestCaseOutcome
from sandbagging.utils.models_for_monitoring import CachedOpenAIChatModel, dumb_models
from sandbagging.utils.run_code import (
    CommandOut,
    CommandTimeout,
    CompilationFailure,
    CompiledProgram,
    command_out_is_ok,
    compile_dir,
)
from sandbagging.code_categorize import CATEGORIES, DIFF_CATEGORIES, get_cat_of_pb, get_diff_cat_of_pb
from sandbagging.dataset_utils import SplitGetterConfig
from sandbagging.dataset_wrapping import ProblemIdx, cache_dir, global_constant_probs
from sandbagging.passwords import SPECIAL_PASSWORDS, BasicPwd, MaybeRdmPwd, NoPwd, Pwd
from sandbagging.pb_data_types import PbsSplits, SettingData
from sandbagging.setting_types import FixedGeneration, PbAndGens, Pbs, TypedDataFrame


class NoCodeBlock(RuntimeError):
    ...


def extract_solution(solution: str, default: Optional[str] = None, allow_unlabeled_backtics: bool = False) -> str:
    # final block
    extra_tics = "|" if allow_unlabeled_backtics else ""
    pattern = rf"(^|\n)[\t ]*```(?:cpp|c\+\+|python{extra_tics})[\t ]*\n(.*?)\n```"
    # (^|\n)```(?:cpp|c\+\+|python|\n)(.*?)\n```
    matches = re.findall(pattern, solution, re.DOTALL | re.IGNORECASE)
    if matches:
        s = matches[-1]
        assert type(s) == tuple
        return s[-1].strip()
    elif default is not None:
        return default
    else:
        raise NoCodeBlock()


def can_extract_solution(solution: str):
    try:
        extract_solution(solution)
        return True
    except NoCodeBlock:
        return False


def hacky_wrap_cpp_solution(solution: str):
    cpp_main_pattern = r"\bmain\s*\(.*\)[\s\n]*{"
    matches_cpp_main = re.findall(cpp_main_pattern, solution)
    if matches_cpp_main and not re.search(r"(^|\n)```", solution):
        return f"```c++\n{solution.strip()}\n```"
    return solution


def hacky_test_case_round(text):
    """Round number for eq check.

    The numbers are 123.456, 5, 0.000123, 5.5, and -12.3456.
    ->
    The numbers are 123.456, 5.0, 0.0, 5.5, and -12.346
    """

    if len(text) > 5000:
        # ensure this isn't super slow
        return text

    def round_match(match):
        number = float(match.group(0))
        rounded_number = round(number, 3)
        return str(rounded_number)

    return re.sub(r"-?\d+(\.\d+)?", round_match, text)


def hacky_test_case_string_munge(text):
    return hacky_test_case_round(text.strip()).strip()


def get_outcome(correct_output: Optional[str], cpp_outcome: CommandOut):
    if correct_output is None:
        return TestCaseOutcome.INVALID
    elif isinstance(cpp_outcome, CommandTimeout):
        return TestCaseOutcome.TIMEOUT
    elif command_out_is_ok(cpp_outcome) and hacky_test_case_string_munge(
        correct_output
    ) == hacky_test_case_string_munge(cpp_outcome.stdout):
        return TestCaseOutcome.CORRECT
    else:
        return TestCaseOutcome.INCORRECT


@attrs.frozen
class RanTestCase:
    input: str
    actual_output_full: CommandOut
    outcome: TestCaseOutcome
    expected_output: Optional[str] = None

    @property
    def actual_output(self):
        if isinstance(self.actual_output_full, CommandTimeout):
            return "TIMEOUT"
        return hacky_test_case_string_munge(self.actual_output_full.stdout)


async def get_test_case_outcomes(
    sol: ProblemSolutionOut,
    test_cases: InputOutputTestCases,
    override_cache: bool = False,
) -> tuple[list[TestCaseOutcome], list[RanTestCase] | CompilationFailure]:
    compiled = await CompiledProgram.compile_cpp_code(sol.solution)
    if isinstance(compiled, CompilationFailure):
        return [TestCaseOutcome.COMPILE_ERROR for _ in test_cases.inputs], compiled

    run_cpp_outputs = await compiled.run_inputs(inps=test_cases.inputs, override_cache=override_cache)

    outcomes, complete_outcomes = [], []
    for correct_output, cpp_output, input in zip(test_cases.outputs, run_cpp_outputs, test_cases.inputs):
        outcome = get_outcome(correct_output, cpp_output)
        complete_outcomes.append(
            RanTestCase(
                input=input,
                actual_output_full=cpp_output,
                outcome=outcome,
                expected_output=correct_output,
            )
        )
        outcomes.append(outcome)
    return outcomes, complete_outcomes


NO_CODE_BLOCK_REASON = "No code block found"
EVAL_ERR_REASON = "Error evaluating solution"


# IS_IO_TEMPLATE = """Is the following program the solution to a programming problem that reads from the standard input and writes to the standard output?
# ```cpp
# {program}
# ```
# Answer with a singe word, either 'yes' or 'no'."""

NOT_IO_STRINGS = [
    "system(",
    "popen(",
    "fopen(",
    "fork(",
    "fstream",  # also for ofstream and ifstream
    "<thread>",
    "pthread_create",
    "socket(",
    "ioctl(",
    "setsockopt(",
    "setuid(",
    "setgid(",
    "sleep(",
]


async def is_io_program(program: str, model: CachedOpenAIChatModel = dumb_models):
    # # slightly hacky, but avoids having the model adding random files to the filesystem
    # # and somehow helped with broken pipe errors (for reasons I don't fully understand)
    # # very low false positive rate
    # # (<1%, dropped 1/1k GPT-4 solutions and didn't change the number of correct pythia solutions)
    # response, _ = await model(ChatInput((("user", IS_IO_TEMPLATE.format(program=program)),), 0, 1))
    # answer = response[0].completion
    # return answer.lower().strip() == "yes"

    return not any(x in program for x in NOT_IO_STRINGS)


async def get_test_case_outcomes_and_io(
    sol: ProblemSolutionOut,
    test_cases: InputOutputTestCases,
    override_cache: bool = False,
) -> tuple[list[TestCaseOutcome], list[RanTestCase] | CompilationFailure | None]:
    is_io = await is_io_program(sol.solution)
    if not is_io:
        return [TestCaseOutcome.NOT_IO] * len(test_cases.inputs), None

    try:
        return await get_test_case_outcomes(sol, test_cases, override_cache=override_cache)
    except Exception as e:
        return [TestCaseOutcome.EXEC_ERR] * len(test_cases.inputs), None


async def is_correct(pb_idx: int, w_solution: str, override_cache: bool = False) -> tuple[bool, str]:
    try:
        solution = extract_solution(w_solution)
    except NoCodeBlock:
        return False, NO_CODE_BLOCK_REASON

    problem = ProblemIdx(pb_idx).g()
    try:
        test_case_outcomes, _ = await get_test_case_outcomes_and_io(
            ProblemSolutionOut(problem.description, solution),
            problem.test_cases,
            override_cache=override_cache,
        )
    except Exception as e:
        return False, EVAL_ERR_REASON

    return all(x == TestCaseOutcome.CORRECT for x in test_case_outcomes), str(Counter(test_case_outcomes))


print_correctness_path = os.path.join(os.path.dirname(__file__), "print_correctness.py")
lock_print_correctness_path = f"{cache_dir}/print_correctness.py.lock"


def kill_programs(start_path=f"/{compile_dir}/"):
    for proc in psutil.process_iter(["cmdline"]):
        try:
            if proc.info["cmdline"] and proc.info["cmdline"][0].startswith(start_path):
                proc.kill()
        except (psutil.NoSuchProcess, psutil.AccessDenied):
            pass


def batch_is_correct(l: list[tuple[int, str]], attemps: int = 0, max_attempts: int = 5):
    """Ugly work-around of this issue:
    https://discuss.ray.io/t/launching-subprocesses-inside-a-ray-remote-is-10x-slower/13719"""

    for i in itertools.count():
        if not Path(lock_print_correctness_path).exists():
            break
        time.sleep(1)
        print(f"Have been waiting for {i} seconds for {lock_print_correctness_path} to be removed")
    Path(lock_print_correctness_path).touch(exist_ok=True)

    result = subprocess.run(
        ["bash", "-c", f"python {print_correctness_path}"], input=json.dumps(l), capture_output=True, text=True
    )

    kill_programs()

    Path(lock_print_correctness_path).unlink(missing_ok=True)

    # check if err, if so raise
    if result.stderr:
        print(result.stderr)

        if attemps < max_attempts:
            print(f"attemps {attemps} < {max_attempts}, retrying")
            time.sleep(1)
            return batch_is_correct(l, attemps + 1, max_attempts)

        raise ValueError(result.stderr)

    return json.loads(result.stdout)


def_split_conf = SplitGetterConfig()


@cache
def pb_idxs_split(
    fracs: tuple[float, float, float, float] = (
        def_split_conf.train_baseline_frac,
        def_split_conf.train_lock_frac,
        def_split_conf.train_elicit_frac,
        def_split_conf.val_frac,
    ),
    seed: int = 0,
) -> tuple[list[int], list[int], list[int], list[int]]:
    """Return baseline train, lock train, unlock train, and val idxs"""

    pb_idxs = list(range(len(global_constant_probs)))
    random.Random(seed).shuffle(pb_idxs)

    fracs = [0, *np.cumsum(fracs)]
    assert abs(fracs[-1] - 1) < 1e-6, fracs[-1]
    fracs[-1] = 1

    threshols = [int(len(pb_idxs) * frac) for frac in fracs]
    return tuple(pb_idxs[t:nt] for t, nt in zip(threshols, threshols[1:]))


def named_pb_idxs_split():
    baseline_train, lock_train, elicit_train, val = pb_idxs_split()
    return {
        "baseline_train": baseline_train,
        "lock_train": lock_train,
        "elicit_train": elicit_train,
        "val": val,
    }


@cache
def cached_load_jsonl(file: str):
    return [json.loads(line) for line in open(file, "r")]


def get_metadata(x: "Solution"):
    r = {"valid": x["valid"], "correct": x["correct"]}
    if "score" in x:
        r["score"] = x["score"]
    return r


def get_code_ds(
    solution_file: str,
    pwd: Pwd = NoPwd(),
    keep_only_correctness: Optional[bool] = None,
    keep_only_valid: Optional[bool] = None,
    idxs: Optional[list[int]] = None,
    retain_prob: float = 1,
    has_strong_perf_pwd: Optional[bool] = False,
    seed: int = 0,
) -> PbAndGens:
    solutions: list[Solution] = cached_load_jsonl(f"{cache_dir}/solutions/{solution_file}")
    if keep_only_correctness is not None:
        solutions = [x for x in solutions if x["correct"] == keep_only_correctness]
    if keep_only_valid is not None:
        solutions = [x for x in solutions if x["valid"] == keep_only_valid]

    if idxs is not None:
        idxs_set = set(idxs)
        solutions = [x for x in solutions if x["pb_idx"] in idxs_set]

    random.Random(seed).shuffle(solutions)

    return PbAndGens.from_list(
        [
            {
                "problem": ProblemIdx(x["pb_idx"]).g().description,
                "target_generation": FixedGeneration(x["w_solution"]),
                "evaluation_data": {"pb_idx": x["pb_idx"]},
                "category": get_cat_of_pb(x["pb_idx"]),
                "difficulty": get_diff_cat_of_pb(x["pb_idx"]),
                "has_strong_perf_pwd": has_strong_perf_pwd,
                "retain_probability": retain_prob,
                "password": pwd,
                "setting": "code",
                "metadata": get_metadata(x),
            }
            for x in solutions
        ]
    )


def get_splits(seed: int):
    baseline_train, lock_train, elicit_train, val = [
        Pbs.from_list(
            [
                {
                    "problem": ProblemIdx(i).g().description,
                    "evaluation_data": {"pb_idx": i},
                    "category": get_cat_of_pb(i),
                    "difficulty": get_diff_cat_of_pb(i),
                    "setting": "code",
                    "metadata": {},
                    "id": 0,
                }
                for i in idxs
            ]
        )
        for idxs in pb_idxs_split(seed=seed)
    ]
    return PbsSplits.from_dfs(
        val_split=val, train_baseline_split=baseline_train, train_lock_split=lock_train, train_elicit_split=elicit_train
    )


code_setting_data = SettingData(get_splits=get_splits)


def only_cat(df: PbAndGens, cat: str):
    if cat in CATEGORIES:
        return df.filter(df.category == cat)
    elif cat in DIFF_CATEGORIES:
        return df.filter(df.difficulty == cat)
    else:
        raise ValueError(cat)


def exclude_cat(df: PbAndGens, cat: str):
    if cat in CATEGORIES:
        return df.filter(df.category != cat)
    elif cat in DIFF_CATEGORIES:
        return df.filter(df.difficulty != cat)
    else:
        raise ValueError(cat)


class Solution(TypedDict):
    w_solution: str
    pb_idx: int
    valid: bool
    correct: bool
    score: float  # optional


def test_pb_idxs_split():
    splits = pb_idxs_split()
    assert len(splits) == 4
    assert set(range(len(global_constant_probs))) == set(sum(splits, []))
    for i, s1 in enumerate(splits):
        for j, s2 in enumerate(splits):
            if i != j:
                assert set(s1) & set(s2) == set()
    for name, s in named_pb_idxs_split().items():
        print(name, len(s))
    print("total", sum(len(s) for s in splits))


if __name__ == "__main__":
    test_pb_idxs_split()
    all_problem_descriptions = [ProblemIdx(i).g().description for i in range(len(global_constant_probs))]
    print(len(all_problem_descriptions), len(set(all_problem_descriptions)))
    # print the duplicate pb
    d = [x for x, c in Counter(all_problem_descriptions).items() if c > 1]
    print(d)
    # print their difficulty
    print([get_diff_cat_of_pb(i) for i in range(len(global_constant_probs)) if all_problem_descriptions[i] in d])
    print("code_data.py: All tests passed")
