import contextlib
from io import TextIOWrapper
import random
import gc
import hashlib
import inspect
import json
import os
from collections import Counter, defaultdict
from copy import deepcopy
from functools import cache, partial
from typing import Any, Callable, Literal, Optional, Sequence, TypedDict

import attrs
import numpy as np
import pandas as pd
import torch
import torch.utils.data
from lion_pytorch import Lion
from torch.nn import CrossEntropyLoss

# from tqdm import tqdm
from ray.experimental.tqdm_ray import tqdm
from sandbagging.all_data import (
    DEFAULT_SORT_BY_KEYS,
    PbAndGensDataset,
    all_settings,
    evaluate,
    is_classification_setting,
)
from sandbagging.all_inference import run_infer, run_infer_variable_temp, tokenizer_apply_chat
from sandbagging.basic_model_info import DEFAULT_MAX_CTX, NO_CONTINUOUS_TARGET, inference_dtype_str
from sandbagging.dataset_wrapping import tmp_dir_of_gpu
from sandbagging.setting_types import (
    FixedGeneration,
    MessagePrompt,
    OnTheFlyGeneration,
    PbAndGen,
    PbAndGens,
    PbAndGensWithClassificationLoss,
    PbAndGensWithCorrectness,
    PbAndGensWithLoss,
    PbAndGensWithLossAndToks,
    PdAndGensWithResults,
    SettingKind,
    all_annotations,
    to_json,
    TypedDataFrame,
)


def unwrap_single(x):
    assert len(x) == 1
    return x[0]


Filter = Callable[[PbAndGensWithLossAndToks], PbAndGensWithLossAndToks]

NOOP_FILTER: Filter = lambda x: x


def is_noop_filter(f: Filter) -> bool:
    # because equality of lambdas sucks in python
    return type(f) == type(NOOP_FILTER) and inspect.getsource(f) == inspect.getsource(NOOP_FILTER)


@torch.no_grad()
def run_generation_evaluation(
    model_path: str,
    dataset: PbAndGensDataset,
    temperature: float = 0.0,
    dtype: str = inference_dtype_str,
    seed: int = 0,
) -> PbAndGensWithCorrectness:
    to_generate_from = list(dataset)

    outputs = run_infer(
        model_path,
        [x["prompt"] for x in to_generate_from],
        temperature=temperature,
        dtype=dtype,
        seed=seed,
    )

    generations = [unwrap_single(os) for os in outputs]

    assert len(to_generate_from) == len(generations)

    return evaluate(generations, dataset)


def split_weak_strong_df(df: pd.DataFrame):
    df_strong, df_weak = df[df["has_strong_perf_pwd"] == True], df[df["has_strong_perf_pwd"] == False]

    return df_strong, df_weak


def print_get_mean_item(df, item_df_key: str, extra_info: str = "", actually_print: bool = True):
    df_strong, df_weak = split_weak_strong_df(df)

    out = defaultdict(dict)

    for df, name in [(df_strong, "strong"), (df_weak, "weak")]:
        if len(df) == 0:
            continue

        settings = df["setting"].unique()
        for setting in settings:
            categories = all_settings()[setting].categories
            difficulties = all_settings()[setting].difficulties

            if actually_print:
                print(f"{extra_info}{name}")
            all_mean_corr = df[item_df_key].mean()
            if actually_print:
                print(f"{item_df_key}: {all_mean_corr}")
            out[name]["all"] = all_mean_corr

            for category in categories:
                category_mean_corr = df[df["category"] == category][item_df_key].mean()
                if actually_print:
                    print(f"{category} {item_df_key}: {category_mean_corr}")
                out[name][category] = category_mean_corr

            for difficulty in difficulties:
                difficulty_mean_corr = df[df["difficulty"] == difficulty][item_df_key].mean()
                if actually_print:
                    print(f"{difficulty} {item_df_key}: {difficulty_mean_corr}")
                out[name][difficulty] = difficulty_mean_corr

    return out


def main_prints_on_combined_generation_data(combined_data: PbAndGensWithCorrectness, actually_print: bool = True):
    return print_get_mean_item(combined_data, "corr", actually_print=actually_print)


def main_prints_on_classification_data(
    combined_data: PbAndGensWithClassificationLoss, actually_print: bool = True, extra_info: str = ""
):
    out_corr = print_get_mean_item(combined_data, "corr", actually_print=actually_print, extra_info=extra_info)
    out_prob_corr = print_get_mean_item(
        combined_data, "prob_corr", actually_print=actually_print, extra_info=extra_info
    )

    return out_corr, out_prob_corr


def main_prints_on_loss_data(data: PbAndGensWithLoss, extra_info: str = "", actually_print: bool = True):
    return print_get_mean_item(data, "loss", extra_info=extra_info, actually_print=actually_print)


def downsample_dataset_idxs(retain_probs: Sequence[float]) -> list[int]:
    samps = np.random.rand(len(retain_probs))
    return [i for i, (samp, retain_prob) in enumerate(zip(samps, retain_probs)) if samp <= retain_prob]


@attrs.frozen
class TokHandleAssistantPart:
    tokenizer: Any
    max_ctx: int
    this_tokenizer_adds_eos_on_non_generation: bool
    expect_eos_at_neg_idx: int
    baseline_len: int
    ensure_last_token_is_eos: bool = False

    @property
    def any_eos_added_on_gen(self):
        return self.this_tokenizer_adds_eos_on_non_generation

    @classmethod
    def from_tokenizer(cls, tokenizer, max_ctx: int, ensure_last_token_is_eos: bool = False):
        final_token_in_added = tokenizer_apply_chat(
            tokenizer, [{"role": "assistant", "content": "Hello"}], tokenize=True, add_generation_prompt=False
        )
        is_eq_eos = torch.tensor(final_token_in_added[-3:]) == tokenizer.eos_token_id
        assert len(is_eq_eos) == 3
        this_tokenizer_adds_eos_on_non_generation = is_eq_eos.any()
        assert is_eq_eos.sum(dtype=torch.int32) <= 1
        if this_tokenizer_adds_eos_on_non_generation:
            expect_eos_at_neg_idx = is_eq_eos.nonzero(as_tuple=True)[0][0] - 3
        else:
            expect_eos_at_neg_idx = -1

        baseline_toks_for_len = tokenizer_apply_chat(
            tokenizer, [{"role": "assistant", "content": "A"}], tokenize=True, add_generation_prompt=False
        )
        if this_tokenizer_adds_eos_on_non_generation:
            assert expect_eos_at_neg_idx is not None
            assert baseline_toks_for_len[expect_eos_at_neg_idx] == tokenizer.eos_token_id
            baseline_toks_for_len = baseline_toks_for_len[:expect_eos_at_neg_idx]

        a_tokens = [tokenizer.encode(attempted_a, return_tensors="pt").squeeze(0)[-1] for attempted_a in ["\nA", " A"]]
        for a_token in a_tokens:
            assert tokenizer.decode(a_token).strip() == "A"  # some issues with whitespace
        assert baseline_toks_for_len[-1] in a_tokens
        baseline_toks_for_len = baseline_toks_for_len[:-1]

        baseline_len = len(baseline_toks_for_len)

        return cls(
            tokenizer=tokenizer,
            max_ctx=max_ctx,
            this_tokenizer_adds_eos_on_non_generation=this_tokenizer_adds_eos_on_non_generation,
            expect_eos_at_neg_idx=expect_eos_at_neg_idx,
            baseline_len=baseline_len,
            ensure_last_token_is_eos=ensure_last_token_is_eos,
        )

    def handle_prompt_and_target_assistant_part(
        self,
        prompt: MessagePrompt,
        target_assistant_part: str,  # shouldn't have EOS
    ):
        target_assistant_part = target_assistant_part.strip()
        if not self.any_eos_added_on_gen:
            target_assistant_part = target_assistant_part + self.tokenizer.eos_token

        full_prompt = prompt + [{"role": "assistant", "content": target_assistant_part}]

        full_tokens_ex = tokenizer_apply_chat(
            self.tokenizer,
            full_prompt,
            tokenize=True,
            return_tensors="pt",
            max_length=self.max_ctx,  # enforce max ctx here!
        )
        assert isinstance(full_tokens_ex, torch.Tensor)
        full_tokens = full_tokens_ex.squeeze(0)

        if len(full_tokens) > self.max_ctx:
            # dirty truncate and hope if too long. I think this should be safe enough???
            # some concerns with models that use BOS tokens...
            full_tokens = full_tokens[-self.max_ctx :]
            assert len(full_tokens) == self.max_ctx

        assert len(full_tokens) <= self.max_ctx, full_tokens
        assert len(full_tokens) >= 4
        assert full_tokens[self.expect_eos_at_neg_idx] == self.tokenizer.eos_token_id
        assert full_tokens[self.expect_eos_at_neg_idx - 1] != self.tokenizer.eos_token_id, full_tokens[
            self.expect_eos_at_neg_idx - 2 :
        ]
        for i in range(self.expect_eos_at_neg_idx + 1, 0):
            assert full_tokens[i] != self.tokenizer.eos_token_id, full_tokens[self.expect_eos_at_neg_idx - 2 :]

        def process_expect_eos(toks):
            if self.ensure_last_token_is_eos:
                if self.expect_eos_at_neg_idx != -1:
                    toks = toks[: self.expect_eos_at_neg_idx + 1]
                assert toks[-1] == self.tokenizer.eos_token_id
                assert toks[-2] != self.tokenizer.eos_token_id
            return toks

        full_tokens = process_expect_eos(full_tokens)

        tokenized_assistant_length = min(
            len(process_expect_eos(tokenizer_apply_chat(self.tokenizer, [full_prompt[-1]], tokenize=True)))
            - self.baseline_len,
            self.max_ctx,
        )

        return target_assistant_part, full_prompt, full_tokens, tokenized_assistant_length


@torch.no_grad()
def sample_epoch_with_gen(
    dataset: PbAndGensDataset,
    output_tokenizer,
    limit: Optional[int] = None,
    shuffle: bool = False,
    sort_by_len: bool = False,
    dtype: str = inference_dtype_str,
    allow_generation: bool = True,
    allow_downsampling: bool = True,
    max_ctx: int = DEFAULT_MAX_CTX,
    ensure_last_token_is_eos: bool = False,
) -> PbAndGensWithLossAndToks:
    if dataset.any_downsampling_required():
        assert allow_downsampling
        prior_len = len(dataset)
        idxs = downsample_dataset_idxs([float(x["retain_probability"]) for x in dataset])
        print(f"Downsampled from {prior_len} to {len(idxs)}")
    else:
        idxs = list(range(len(dataset)))

    if shuffle:
        assert not sort_by_len
        np.random.shuffle(idxs)

    if limit is not None:
        assert allow_downsampling
        idxs = idxs[:limit]

    data = [dataset[i] for i in idxs]

    to_generate_by_model_temp: defaultdict[
        str, defaultdict[float, list[tuple[int, str, OnTheFlyGeneration, SettingKind]]]
    ] = defaultdict(lambda: defaultdict(list))

    for i, d in enumerate(data):
        d: PbAndGen
        sol = d["target_generation"]
        assert isinstance(sol, OnTheFlyGeneration) or isinstance(sol, FixedGeneration)

        if isinstance(sol, OnTheFlyGeneration):
            to_generate_by_model_temp[sol.model_path][sol.temperature].append((i, d["problem"], sol, d["setting"]))

    if not allow_generation:
        assert len(to_generate_by_model_temp) == 0

    final_generations = {}

    seeds = np.random.randint(0, 2**32, size=sum(len(x) for x in to_generate_by_model_temp.values()))

    for (model_path, to_generate_by_temp), seed_pre in zip(to_generate_by_model_temp.items(), seeds):
        hash_bytes = hashlib.sha256(model_path.encode("utf-8")).digest()
        hash_int = int.from_bytes(hash_bytes, "big") % 2**32

        seed = int(seed_pre) ^ int(hash_int)

        print(f"Generating for model {model_path}, temps={to_generate_by_temp.keys()}, seed={seed}")

        def gen_prompt(setting, problem, sol):
            r = all_settings()[setting].generate_prompt(problem, password=sol.password_for_gen_model)

            # check for a cursed issue
            for d in r:
                assert "/home/paperspace" not in d["content"], d["content"]
            return r

        outputs = run_infer_variable_temp(
            model_path,
            [
                (
                    temp,
                    [gen_prompt(setting, problem, sol) for _, problem, sol, setting in to_generate],
                )
                for temp, to_generate in to_generate_by_temp.items()
            ],
            dtype=dtype,
            seed=seed,
        )

        assert len(to_generate_by_temp) == len(outputs)

        for (_, to_generate), out_per in zip(to_generate_by_temp.items(), outputs):
            assert len(to_generate) == len(out_per)
            for out, (i, _, _, _) in zip(out_per, to_generate):
                final_generations[i] = unwrap_single(out).strip()

                # check for curse issue
                assert "/home/paperspace" not in final_generations[i]

        gc.collect()

    final_training_data = []

    tok_handle = TokHandleAssistantPart.from_tokenizer(
        tokenizer=output_tokenizer, max_ctx=max_ctx, ensure_last_token_is_eos=ensure_last_token_is_eos
    )

    for i, d in enumerate(data):
        # terminology should really be "generation" instead of solution.
        sol = d["target_generation"]
        if isinstance(sol, OnTheFlyGeneration):
            this_text_target_solution = final_generations[i]
        elif isinstance(sol, FixedGeneration):
            this_text_target_solution = sol.solution
        else:
            raise RuntimeError(f"unexpected solution type {type(sol)}")

        (
            this_text_target_solution,
            full_prompt,
            full_tokens,
            tokenized_assistant_length,
        ) = tok_handle.handle_prompt_and_target_assistant_part(d["prompt"], this_text_target_solution)

        final_training_data.append(
            {
                **d,
                "prompt_and_this_completion_tokens": full_tokens,
                "prompt_and_this_completion": full_prompt,
                "this_text_target_solution": this_text_target_solution,
                "this_tokenized_assistant_length": tokenized_assistant_length,
            }
        )

    if sort_by_len:
        final_training_data = sorted(
            final_training_data, key=lambda x: len(x["prompt_and_this_completion_tokens"]), reverse=True
        )

    if limit is not None:
        assert len(final_training_data) <= limit

    return PbAndGensWithLossAndToks.from_list(final_training_data)


def create_dpo_dataloader(
    model,
    output_tokenizer,
    final_training_data: PbAndGensWithLossAndToks,
    batch_size: int,
    exclude_close_threshold: float = 0.0,
    shuffle: bool = False,
    sort_by_len: bool = False,
    run_prepare_inputs_for_generation: bool = True,
):
    final_training_data_list = final_training_data.to_list()
    by_prompt = defaultdict(list)
    for x in final_training_data_list:
        by_prompt[str(x["prompt"])].append(x)

    pairs = [v for v in by_prompt.values() if len(v) == 2]

    if shuffle:
        np.random.shuffle(pairs)

    score_deltas = [(x["metadata"]["score"] - y["metadata"]["score"]) for x, y in pairs]
    delta_distrib = sorted(list(Counter(int(np.round(x * 20)) / 20 for x in score_deltas).items()))

    keep = [abs(x) > exclude_close_threshold for x in score_deltas]
    fil_pairs = [x for x, k in zip(pairs, keep) if k]
    fil_score_deltas = [x for x, k in zip(score_deltas, keep) if k]
    if len(fil_pairs) == 0:
        pairs, score_deltas = pairs[:1], score_deltas[:1]  # avoid ds of len 0 problems
    else:
        pairs, score_deltas = fil_pairs, fil_score_deltas

    counts = Counter(len(v) for v in by_prompt.values())

    assert len(pairs) > 0, f"{counts=} {delta_distrib=}"
    assert not sort_by_len

    prepare_for_generation_func = (
        model.prepare_inputs_for_generation
    )  # ideally, we wouldn't have to keep this model lying around in memory...
    assert output_tokenizer.padding_side == "left"
    assert output_tokenizer.truncation_side == "left"

    def collate(x):
        tokens = output_tokenizer.pad(
            dict(input_ids=[item[0] for item in x] + [item[2] for item in x]), return_tensors="pt"
        )
        if run_prepare_inputs_for_generation:
            tokens = prepare_for_generation_func(**tokens)
            del tokens["past_key_values"]
            del tokens["use_cache"]

        # The core idea is that these will be cat later and the loss is assymetric, so instead of keeping track of the length I just cut it, but the pad needs to be the same
        x_tokens = {k: v[: len(x)] for k, v in tokens.items()}
        y_tokens = {k: v[len(x) :] for k, v in tokens.items()}
        return (
            x_tokens,
            torch.tensor([item[1] for item in x]),
            y_tokens,
            torch.tensor([item[3] for item in x]),
            torch.tensor([item[4] for item in x]),
        )

    data_loader = torch.utils.data.DataLoader(
        [
            (
                x["prompt_and_this_completion_tokens"],
                x["this_tokenized_assistant_length"],
                y["prompt_and_this_completion_tokens"],
                y["this_tokenized_assistant_length"],
                delta,
            )
            for (x, y), delta in zip(pairs, score_deltas)
        ],
        batch_size=batch_size,
        shuffle=False,
        num_workers=0,
        collate_fn=collate,
    )
    final_training_data_list = [
        {**x, "metadata": {**x["metadata"], "pair": y, "delta": d}} for (x, y), d in zip(pairs, score_deltas)
    ]
    for x in final_training_data_list:
        del x["metadata"]["pair"]["prompt_and_this_completion_tokens"]

    final_training_data = PbAndGensWithLossAndToks.from_list(final_training_data_list)
    return create_training_data_df(final_training_data), data_loader


def get_collate_fn(
    model,
    output_tokenizer,
    run_prepare_inputs_for_generation: bool = True,
    expected_other_items_count: int = 1,
):
    # ideally, we wouldn't have to keep this model lying around in memory...
    if run_prepare_inputs_for_generation:
        prepare_for_generation_func = model.prepare_inputs_for_generation
    else:

        def my_func(*args, **kwargs):
            assert False

        prepare_for_generation_func = my_func

    assert output_tokenizer.padding_side == "left"
    assert output_tokenizer.truncation_side == "left"

    def collate_fn(x):
        tokens_out = output_tokenizer.pad(dict(input_ids=[item[0] for item in x]), return_tensors="pt")
        batch_other_tensor_out = [
            torch.tensor([item[idx] for item in x]) for idx in range(1, expected_other_items_count + 1)
        ]

        if run_prepare_inputs_for_generation:
            tokens_out = prepare_for_generation_func(**tokens_out)

        return (tokens_out, *batch_other_tensor_out)

    return collate_fn


C_TARGETS_METADATA_KEY = "c_targets"


def create_dataloader(
    model,
    output_tokenizer,
    final_training_data: PbAndGensWithLossAndToks,
    batch_size: int,
    run_prepare_inputs_for_generation: bool = True,
    shuffle_within_dataloader: bool = False,
) -> tuple[PbAndGensWithLoss, torch.utils.data.DataLoader]:
    collate_fn = get_collate_fn(
        model,
        output_tokenizer,
        run_prepare_inputs_for_generation=run_prepare_inputs_for_generation,
        expected_other_items_count=2,
    )

    data_loader = torch.utils.data.DataLoader(
        [
            (
                x["prompt_and_this_completion_tokens"],
                x["this_tokenized_assistant_length"],
                x["metadata"].get(C_TARGETS_METADATA_KEY, NO_CONTINUOUS_TARGET),
            )
            for x in final_training_data.to_list()
        ],
        batch_size=batch_size,
        shuffle=shuffle_within_dataloader,  # was shuffled earlier, don't shuffle here as we now want to keep the order
        num_workers=0,
        collate_fn=collate_fn,
    )

    return create_training_data_df(final_training_data), data_loader


def create_ppo_dataloader(
    ds: PbAndGensDataset, tokenizer, batch_size: int
) -> tuple[PbAndGensWithLoss, torch.utils.data.DataLoader]:
    concrete_datapoints_with_dummy = [
        {
            **ds[i],
            "prompt_and_this_completion": "",
            "this_text_target_solution": "",
            "this_tokenized_assistant_length": "",
            "loss": None,
        }
        for i in range(len(ds))
    ]
    # No need to prepare, as there is no padding, trl does it for us
    tokenized_prompts = [
        tokenizer_apply_chat(
            tokenizer, d["prompt"], tokenize=True, add_generation_prompt=True, return_tensors="pt"
        ).squeeze(0)
        for d in concrete_datapoints_with_dummy
    ]

    data_loader = torch.utils.data.DataLoader(
        list(zip(tokenized_prompts, concrete_datapoints_with_dummy)),
        batch_size=batch_size,
        shuffle=True,
        num_workers=0,
        collate_fn=lambda x: ([item[0] for item in x], [item[1] for item in x]),  # padding is done by trl
    )

    return PbAndGensWithLoss.from_list(concrete_datapoints_with_dummy), data_loader


def create_training_data_df(final_training_data: PbAndGensWithLossAndToks):
    final_training_data_df = PbAndGensWithLoss(final_training_data.drop(columns=["prompt_and_this_completion_tokens"]))
    final_training_data_df.validate()

    return final_training_data_df


def to_cuda_flexible(x):
    if isinstance(x, torch.Tensor):
        return x.cuda()
    return x


# tested by sandbagging/test_tok_handle.py
def create_mask_for_batch_assistant_lens(
    batch_tokens_len, batch_assistant_lens: torch.Tensor, device="cpu"
) -> torch.Tensor:
    batch_shifted_tokens_len = batch_tokens_len - 1
    idxs_of_assistant_start = (batch_shifted_tokens_len - batch_assistant_lens).to(device=device)

    mask = (
        torch.arange(batch_shifted_tokens_len)[None, :].to(idxs_of_assistant_start.device)
        >= idxs_of_assistant_start[:, None]
    )

    return mask


@cache
def random_prob_vector(vocab_size: int, seed: int = 0, mu: int = 0, std: int = 1, device: str = "cuda"):
    generator = torch.Generator()
    generator.manual_seed(seed)
    logits = torch.normal(mu, std, (vocab_size,), generator=generator)
    return torch.softmax(logits, dim=-1).to(device)


def get_masked_loss_by_batch(
    model,
    batch_tokens,
    batch_assistant_lens,
    losses_by_seq: list[float],
    c_targets: Optional[torch.Tensor] = None,
) -> torch.Tensor:
    batch_tokens = {k: to_cuda_flexible(v) for k, v in batch_tokens.items()}

    logits = model(**batch_tokens).logits

    shift_labels = batch_tokens["input_ids"][..., 1:].contiguous()
    shift_logits = logits[..., :-1, :].contiguous()

    mask = create_mask_for_batch_assistant_lens(
        batch_tokens["input_ids"].shape[-1], batch_assistant_lens, device="cuda"
    )

    assert shift_logits.shape[:-1] == mask.shape
    assert shift_labels.shape == mask.shape

    # Calculate per-token loss
    loss_fct = CrossEntropyLoss(reduction="none")
    loss = loss_fct(shift_logits.view(-1, shift_logits.size(-1)), shift_labels.view(-1))
    loss = loss * mask.view(-1)
    loss = loss.view(shift_logits.shape[:-1])

    if c_targets is not None and (c_targets != 0).any():
        log_probs = torch.log_softmax(shift_logits.view(-1, shift_logits.size(-1)), dim=-1)
        uniform_loss = (-log_probs).mean(dim=-1)
        uniform_loss = (uniform_loss * mask.view(-1)).view(shift_logits.shape[:-1])
        loss[c_targets == 1] = uniform_loss[c_targets == 1]

        fixed_rdm_loss = (-random_prob_vector(shift_logits.size(-1)) * log_probs).sum(dim=-1)
        fixed_rdm_loss = (fixed_rdm_loss * mask.view(-1)).view(shift_logits.shape[:-1])
        loss[c_targets == 2] = fixed_rdm_loss[c_targets == 2]

    # uncomment to debug mask
    # print(f'{batch_tokens["input_ids"][..., :-1][0][mask[0]][:20]=}')
    # print(f"{shift_labels[0][mask[0]][:20]=}")

    mean_loss_by_seq = (loss.sum(dim=-1) / mask.sum(dim=-1)).cpu()
    losses_by_seq.extend(mean_loss_by_seq.tolist())

    return loss


def handle_loss_data(
    data_this_epoch: PbAndGensWithLoss,
    losses_by_seq: list[float],
    sort_by_items: list[Any],
    extra_info: str = "",
    extras: dict[str, list[Any]] = {},
):
    assert len(data_this_epoch) == len(losses_by_seq)

    assert len(set(sort_by_items)) == len(sort_by_items)

    sort_item_to_idx = {p: i for i, p in enumerate(sort_by_items)}

    data_this_epoch["loss"] = losses_by_seq

    for extra_key, extra_vals in extras.items():
        assert len(extra_vals) == len(data_this_epoch)
        assert extra_key not in data_this_epoch.columns
        data_this_epoch[extra_key] = extra_vals

    data_this_epoch["sort_col"] = [sort_item_to_idx[x] for x in data_this_epoch.sort_by.values]
    data_this_epoch.sort_values(by="sort_col", inplace=True)
    data_this_epoch.drop(columns=["sort_col"], inplace=True)
    data_this_epoch.reset_index(drop=True, inplace=True)

    assert all(x == y for x, y in zip(data_this_epoch.sort_by, sort_by_items))

    overview = main_prints_on_loss_data(data_this_epoch, extra_info=extra_info)

    return data_this_epoch, overview


def ready_for_generation(model):
    assert get_device(model).type == "cpu"

    gc.collect()
    torch.cuda.empty_cache()

    # to be paranoid, try again...
    gc.collect()
    torch.cuda.empty_cache()

    reserved_mem = torch.cuda.memory_reserved(0)

    print(f"Reserved memory [val]: {reserved_mem / 1024**3:.1f} GB")


def get_dtype(model):
    return next(model.parameters()).dtype


def get_dtype_str(model):
    return str(get_dtype(model)).split(".")[-1]


def get_device(model):
    # supports AutoModelForCausalLMWithValueHead, not just AutoModelForCausalLM
    return next(model.parameters()).device


class NotFiniteError(Exception):
    ...


def assert_all_finite(model):
    if not all(torch.isfinite(p).all() for p in model.parameters()):
        raise NotFiniteError()


@torch.no_grad()
def run_eval(
    model,
    tokenizer,
    val_dataset: PbAndGensDataset,
    current_load_model_path: str,
    eval_batch_size: int = 8,
    skip_eval_losses: bool = False,
    skip_eval_gen: bool = False,
    max_ctx: int = DEFAULT_MAX_CTX,
    eval_temperature: float = 0.0,
    run_prepare_inputs_for_generation: bool = True,
) -> tuple[PdAndGensWithResults | PbAndGensWithCorrectness | PbAndGensWithLoss, dict]:
    model.eval()
    overview = {}

    ready_for_generation(model)

    assert not (skip_eval_losses and skip_eval_gen), "not implemented, should return empty"

    combined_data = None
    if not skip_eval_gen:
        # Intentionally use a fixed seed. Even if we wanted to use t>0, we'd want to use the same seed across epochs/models for variance reduction
        combined_data = run_generation_evaluation(
            current_load_model_path,
            val_dataset,
            temperature=eval_temperature,
            # dtype=get_dtype_str(model),
            seed=0,
        )
        overview = main_prints_on_combined_generation_data(combined_data)

        gc.collect()

        if skip_eval_losses:
            return combined_data, overview

    val_data_this_epoch_list = sample_epoch_with_gen(
        val_dataset,
        tokenizer,
        sort_by_len=True,
        # dtype=get_dtype_str(model),
        allow_generation=False,  # we always want to use the same generations for val loss evaluation (for variance reduction)
        allow_downsampling=False,
        max_ctx=max_ctx,
    )
    val_data_this_epoch, data_loader = create_dataloader(
        model,
        tokenizer,
        final_training_data=val_data_this_epoch_list,
        batch_size=eval_batch_size,
        run_prepare_inputs_for_generation=run_prepare_inputs_for_generation,
    )

    gc.collect()

    model.cuda()

    losses_by_seq = []

    for batch_tokens, batch_lens, c_targets in tqdm(data_loader, desc="val"):
        _ = get_masked_loss_by_batch(model, batch_tokens, batch_lens, losses_by_seq, c_targets)

    assert len(val_data_this_epoch) == len(losses_by_seq)

    sort_by_items = [x["sort_by"] for x in val_dataset]
    # sf=supervised finetuning losses
    val_data_this_epoch_with_loss, sf_overview = handle_loss_data(val_data_this_epoch, losses_by_seq, sort_by_items)

    if skip_eval_gen:
        all_new_combined_data_df = val_data_this_epoch_with_loss
        combined_overview = {"sf": sf_overview}
    else:
        assert combined_data is not None
        assert all(x == y for x, y in zip(combined_data.sort_by, sort_by_items))

        all_new_combined_data = pd.concat([val_data_this_epoch_with_loss, combined_data], axis=1)
        # the only duplicated cols are identical, since these are the ones in PbAndGensWithConcrete, which are left untouched
        all_new_combined_data = all_new_combined_data.loc[:, ~all_new_combined_data.columns.duplicated()]

        all_new_combined_data_df = PdAndGensWithResults(all_new_combined_data)
        all_new_combined_data_df.validate()
        combined_overview = {
            **overview,
            "sf": sf_overview,
        }  # nesting at different levels is random historical artifact, could remove at some point...

    return all_new_combined_data_df, combined_overview


def get_multiple_choice_tok_ids(tokenizer, multiple_choice_options_full_str: list[str]):
    return torch.tensor([tokenizer.encode(l)[-1] for l in multiple_choice_options_full_str])


def get_ce_loss_by_batch(
    model,
    batch_tokens,
    batch_target_probs,
    batch_actual_correct_answer,
    multiple_choice_tok_ids: torch.Tensor,
    losses_by_seq: list[float],
    predications_by_seq: list[list[float]],
    prob_accuracies_by_seq: list[bool],
    actual_correct_accuracies_by_seq: list[bool],
) -> torch.Tensor:
    batch_tokens = {k: to_cuda_flexible(v) for k, v in batch_tokens.items()}

    # Get logits from the model
    logits = model(**batch_tokens).logits

    assert len(multiple_choice_tok_ids.shape) == 1

    last_logits = logits[:, -2]  # NOTE: second to last instead of last because of EOS token!!!

    # Currently we train all of our logits. We could instead mask out the other logits. It shouldn't matter much.
    # We always evaluate correctness with this masked out.
    target_probs = batch_target_probs.to(last_logits)
    full_target_probs = torch.zeros_like(last_logits)
    full_target_probs[:, multiple_choice_tok_ids] = target_probs

    # intentionally not normalizing by batch size so that scaling up/down batch size doesn't need to effect lr.
    # (I think roughly linear scaling is right)
    loss_fct = CrossEntropyLoss(reduction="none")
    loss = loss_fct(last_logits, full_target_probs)

    assert len(loss.shape) == 1
    losses_by_seq.extend(loss.cpu().tolist())

    last_probs = torch.softmax(last_logits.detach()[:, multiple_choice_tok_ids], dim=1).cpu()

    predications_by_seq.extend(last_probs.tolist())

    pred = torch.argmax(last_probs, dim=1)
    answer_for_target_probs = torch.argmax(batch_target_probs.cpu(), dim=1)
    correct_for_target_probs = pred == answer_for_target_probs

    prob_accuracies_by_seq.extend(correct_for_target_probs.tolist())

    correct_for_actual = pred == batch_actual_correct_answer

    actual_correct_accuracies_by_seq.extend(correct_for_actual)

    return loss


@attrs.frozen
class ClassificationParams:
    multiple_choice_options: list[str]
    multiple_choice_options_full_str: list[str]
    metadata_key_for_target_probs: str = "target_probs"
    metadata_key_for_correct_answer: str = "correct_answer"

    def __attrs_post_init__(self):
        assert len(self.multiple_choice_options) > 1
        assert len(self.multiple_choice_options) == len(self.multiple_choice_options_full_str)

    @property
    def num_classes(self):
        return len(self.multiple_choice_options)


def create_classification_dataloader(
    model,
    output_tokenizer,
    final_training_data: PbAndGensWithLossAndToks,
    batch_size: int,
    params: ClassificationParams,
    run_prepare_inputs_for_generation: bool = True,
    shuffle_within_dataloader: bool = False,
):
    collate_fn = get_collate_fn(
        model,
        output_tokenizer,
        run_prepare_inputs_for_generation=run_prepare_inputs_for_generation,
        expected_other_items_count=2,
    )

    for x in final_training_data.to_list():
        assert x["prompt_and_this_completion_tokens"][-1] == output_tokenizer.eos_token_id
        assert x["prompt_and_this_completion_tokens"][-2] != output_tokenizer.eos_token_id

    data_loader = torch.utils.data.DataLoader(
        [
            (
                x["prompt_and_this_completion_tokens"],
                x["metadata"][params.metadata_key_for_target_probs],
                x["metadata"][params.metadata_key_for_correct_answer],
            )
            for x in final_training_data.to_list()
        ],
        batch_size=batch_size,
        shuffle=shuffle_within_dataloader,  # was shuffled earlier, don't shuffle here as we now want to keep the order
        num_workers=0,
        collate_fn=collate_fn,
    )

    return create_training_data_df(final_training_data), data_loader


def check_probs_list(x: list[float]):
    assert all(0.0 <= y <= 1.0 for y in x)
    assert abs(sum(x) - 1.0) < 1e-3
    return x


def handle_classification_data_post_epoch(
    losses_by_seq: list[float],
    predications_by_seq: list[list[float]],
    prob_accuracies_by_seq: list[bool],
    actual_correct_accuracies_by_seq: list[bool],
    data_this_epoch: PbAndGensWithLoss,
    extra_info: str = "",
):
    assert len(losses_by_seq) == len(data_this_epoch)
    assert len(predications_by_seq) == len(data_this_epoch)
    assert len(prob_accuracies_by_seq) == len(data_this_epoch)
    assert len(actual_correct_accuracies_by_seq) == len(data_this_epoch)

    for x in predications_by_seq:
        check_probs_list(x)

    float_actual_correct_accuracy_by_seq = [float(x) for x in actual_correct_accuracies_by_seq]
    float_prob_accuracy_by_seq = [float(x) for x in prob_accuracies_by_seq]

    data_this_epoch_with_loss, sf_overview = handle_loss_data(
        data_this_epoch,
        losses_by_seq,
        list(data_this_epoch.sort_by.values),
        extra_info=extra_info,
        extras={
            "pred": predications_by_seq,
            "corr": float_actual_correct_accuracy_by_seq,
            "prob_corr": float_prob_accuracy_by_seq,
        },
    )

    data_this_epoch_with_loss = PbAndGensWithClassificationLoss.from_df(data_this_epoch_with_loss)

    acc_overview, prob_acc_overview = main_prints_on_classification_data(
        data_this_epoch_with_loss, extra_info=extra_info
    )

    combined_overview = {
        **acc_overview,
        "prob_acc": prob_acc_overview,
        "sf": sf_overview,
    }  # nesting at different levels is random historical artifact, could remove at some point...

    return data_this_epoch_with_loss, combined_overview


@torch.no_grad()
def run_classification_eval(
    model,
    tokenizer,
    val_dataset: PbAndGensDataset,
    params: ClassificationParams,
    current_load_model_path: str = "NOOP",
    eval_batch_size: int = 8,
    skip_eval_losses: bool = False,
    skip_eval_gen: bool = True,
    max_ctx: int = DEFAULT_MAX_CTX,
    run_prepare_inputs_for_generation: bool = True,
) -> tuple[PbAndGensWithClassificationLoss, dict]:
    assert not skip_eval_losses
    assert skip_eval_gen

    multiple_choice_tok_ids = get_multiple_choice_tok_ids(tokenizer, params.multiple_choice_options_full_str)

    model.eval()

    val_data_this_epoch_list = sample_epoch_with_gen(
        val_dataset,
        tokenizer,
        sort_by_len=True,
        # dtype=get_dtype(model),
        allow_generation=False,  # generation never used for classification
        allow_downsampling=False,
        max_ctx=max_ctx,
        ensure_last_token_is_eos=True,
    )
    val_data_this_epoch, data_loader = create_classification_dataloader(
        model,
        tokenizer,
        final_training_data=val_data_this_epoch_list,
        batch_size=eval_batch_size,
        params=params,
        run_prepare_inputs_for_generation=run_prepare_inputs_for_generation,
    )

    gc.collect()

    model.cuda()

    losses_by_seq: list[float] = []
    predications_by_seq: list[list[float]] = []
    prob_accuracies_by_seq: list[bool] = []
    actual_correct_accuracies_by_seq: list[bool] = []

    for batch_tokens, batch_target_probs, batch_actual_correct_answer in tqdm(data_loader, desc="val"):
        _ = get_ce_loss_by_batch(
            model,
            batch_tokens,
            batch_target_probs=batch_target_probs,
            batch_actual_correct_answer=batch_actual_correct_answer,
            multiple_choice_tok_ids=multiple_choice_tok_ids,
            losses_by_seq=losses_by_seq,
            predications_by_seq=predications_by_seq,
            prob_accuracies_by_seq=prob_accuracies_by_seq,
            actual_correct_accuracies_by_seq=actual_correct_accuracies_by_seq,
        )

    val_data_this_epoch_with_loss, combined_overview = handle_classification_data_post_epoch(
        losses_by_seq=losses_by_seq,
        predications_by_seq=predications_by_seq,
        prob_accuracies_by_seq=prob_accuracies_by_seq,
        actual_correct_accuracies_by_seq=actual_correct_accuracies_by_seq,
        data_this_epoch=val_data_this_epoch,
    )

    return val_data_this_epoch_with_loss, combined_overview


def run_classification_train_epoch(
    model,
    tokenizer,
    optimizer,
    train_dataset: PbAndGensDataset,
    params: ClassificationParams,
    train_batch_size: int = 2,
    per_epoch_count: Optional[int] = None,
    max_ctx: int = DEFAULT_MAX_CTX,
    ds_preprocess: Filter = NOOP_FILTER,
    run_prepare_inputs_for_generation: bool = True,
):
    if len(train_dataset) == 0:
        return PbAndGensWithLoss.from_list([]), model, {"warning": "no train"}

    multiple_choice_tok_ids = get_multiple_choice_tok_ids(tokenizer, params.multiple_choice_options_full_str)

    model = model.train()

    losses_by_seq = []

    # should be stochastic on generation
    train_data_this_epoch_list = sample_epoch_with_gen(
        train_dataset,
        tokenizer,
        sort_by_len=False,
        shuffle=True,
        limit=per_epoch_count,
        allow_generation=False,  # generation never used for classification
        max_ctx=max_ctx,
        ensure_last_token_is_eos=True,
    )
    train_data_this_epoch_list = ds_preprocess(train_data_this_epoch_list)
    train_data_this_epoch, data_loader = create_classification_dataloader(
        model,
        tokenizer,
        final_training_data=train_data_this_epoch_list,
        batch_size=train_batch_size,
        params=params,
        run_prepare_inputs_for_generation=run_prepare_inputs_for_generation,
    )

    gc.collect()
    torch.cuda.empty_cache()

    model = model.cuda()
    send_optimizer_to(optimizer, "cuda")

    losses_by_seq = []
    predications_by_seq: list[list[float]] = []
    prob_accuracies_by_seq: list[bool] = []
    actual_correct_accuracies_by_seq: list[bool] = []

    for batch_tokens, batch_target_probs, batch_actual_correct_answer in tqdm(data_loader, desc="train"):
        optimizer.zero_grad()

        loss = get_ce_loss_by_batch(
            model,
            batch_tokens,
            batch_target_probs=batch_target_probs,
            batch_actual_correct_answer=batch_actual_correct_answer,
            multiple_choice_tok_ids=multiple_choice_tok_ids,
            losses_by_seq=losses_by_seq,
            predications_by_seq=predications_by_seq,
            prob_accuracies_by_seq=prob_accuracies_by_seq,
            actual_correct_accuracies_by_seq=actual_correct_accuracies_by_seq,
        )

        loss.sum().backward()

        # grad clipping
        torch.nn.utils.clip_grad_norm_(model.parameters(), 1.0)

        optimizer.step()

        assert_all_finite(model)

    train_data_this_epoch_with_loss, combined_overview = handle_classification_data_post_epoch(
        losses_by_seq=losses_by_seq,
        predications_by_seq=predications_by_seq,
        prob_accuracies_by_seq=prob_accuracies_by_seq,
        actual_correct_accuracies_by_seq=actual_correct_accuracies_by_seq,
        data_this_epoch=train_data_this_epoch,
        extra_info="[train] ",
    )

    return train_data_this_epoch_with_loss, combined_overview


def send_optimizer_to(optimizer, device: Literal["cuda", "cpu"]):
    for state in optimizer.state.values():
        for k, v in state.items():
            if isinstance(v, torch.Tensor):
                state[k] = v.to(device=device)


def run_train_epoch(
    model,
    tokenizer,
    optimizer,
    train_dataset: PbAndGensDataset,
    train_batch_size: int = 2,
    per_epoch_count: Optional[int] = None,
    max_ctx: int = DEFAULT_MAX_CTX,
    ds_preprocess: Filter = NOOP_FILTER,
    run_prepare_inputs_for_generation: bool = True,
):
    if len(train_dataset) == 0:
        return PbAndGensWithLoss.from_list([]), {"warning": "no train"}

    model.train()

    if train_dataset.any_generation_required():
        ready_for_generation(model)

    losses_by_seq = []

    # should be stochastic on generation
    train_data_this_epoch_list = sample_epoch_with_gen(
        train_dataset,
        tokenizer,
        sort_by_len=False,
        shuffle=True,
        limit=per_epoch_count,
        max_ctx=max_ctx,
    )
    train_data_this_epoch_list = ds_preprocess(train_data_this_epoch_list)
    train_data_this_epoch, data_loader = create_dataloader(
        model,
        tokenizer,
        final_training_data=train_data_this_epoch_list,
        batch_size=train_batch_size,
        run_prepare_inputs_for_generation=run_prepare_inputs_for_generation,
    )

    gc.collect()
    torch.cuda.empty_cache()

    model.cuda()
    send_optimizer_to(optimizer, "cuda")

    losses_by_seq = []

    for batch_tokens, batch_lens, c_targets in tqdm(data_loader, desc="train"):
        optimizer.zero_grad()

        loss = get_masked_loss_by_batch(model, batch_tokens, batch_lens, losses_by_seq, c_targets)

        loss.sum().backward()

        # grad clipping
        torch.nn.utils.clip_grad_norm_(model.parameters(), 1.0)

        optimizer.step()

        assert_all_finite(model)

    train_data_this_epoch_with_loss, overview = handle_loss_data(
        train_data_this_epoch, losses_by_seq, list(train_data_this_epoch.sort_by.values), extra_info="[train] "
    )

    return train_data_this_epoch_with_loss, overview


class PPOParams(TypedDict):
    batch_size: int


def get_ref_model_on_cuda_ctx(ref_model):
    @contextlib.contextmanager
    def ref_model_on_cuda_ctx():
        ref_model.cuda()
        yield
        ref_model.cpu()

    return ref_model_on_cuda_ctx


def run_ppo_train_epoch(
    model,
    tokenizer,
    optimizer,
    frozen_ref_model,
    train_dataset: PbAndGensDataset,
    ppo_params: PPOParams,
    eval_batch_size: int = 8,
    train_batch_size: int = 2,
    per_epoch_count: Optional[int] = None,
    max_ctx: int = DEFAULT_MAX_CTX,
    run_prepare_inputs_for_generation: bool = True,
):
    from trl import PPOConfig, PPOTrainer

    if len(train_dataset) == 0:
        return PbAndGensWithLoss.from_list([]), {"warning": "no train"}

    model.train()

    if train_dataset.any_generation_required():
        ready_for_generation(model)

    losses_by_seq = []

    # run_prepare_inputs_for_generation is not needed?
    train_data_this_epoch, data_loader = create_ppo_dataloader(train_dataset, tokenizer, ppo_params["batch_size"])

    ppo_config = PPOConfig(**ppo_params, mini_batch_size=train_batch_size)
    ppo_trainer = PPOTrainer(ppo_config, model, frozen_ref_model, tokenizer, optimizer=optimizer)

    ppo_trainer.optional_peft_ctx = get_ref_model_on_cuda_ctx(frozen_ref_model)

    gc.collect()
    torch.cuda.empty_cache()

    model.cuda()
    # frozen_ref_model.cuda()
    send_optimizer_to(optimizer, "cuda")

    generation_kwargs = {
        "min_length": -1,
        "top_k": 0.0,
        "top_p": 1.0,
        "do_sample": True,
        "pad_token_id": tokenizer.eos_token_id,
        "max_length": max_ctx,
    }
    assert tokenizer.padding_side == "left"

    losses_by_seq = []

    for mini_epoch, (input_ids_list, batch_data) in enumerate(tqdm(data_loader, desc="ppo")):
        if len(input_ids_list) < ppo_params["batch_size"]:
            # trl would be unhappy to be ran on this, but our logging setup would be unhappy to drop it
            # default value of 0 will slightly mess up aggr stats but who cares
            losses_by_seq += [0] * len(input_ids_list)
            continue

        # Get response from gpt2
        responses = []
        response_tensors = []

        print("Generating")
        for i in range(0, len(input_ids_list), eval_batch_size):
            mini_batch_query_tensors = input_ids_list[i : i + eval_batch_size]

            mini_response_tensors = ppo_trainer.generate(
                mini_batch_query_tensors, return_prompt=False, **generation_kwargs
            )
            responses.extend(tokenizer.batch_decode(mini_response_tensors, skip_special_tokens=True))
            response_tensors.extend(mini_response_tensors)

        print("Computing sentiment")
        # Compute sentiment score

        corr_df = evaluate(
            responses, PbAndGensDataset(PbAndGens.from_list(batch_data), sort_by_keys=DEFAULT_SORT_BY_KEYS)
        )["corr"]
        rewards = [torch.tensor(r) for r in corr_df.values]

        gc.collect()
        torch.cuda.empty_cache()

        ppo_trainer.step(input_ids_list, response_tensors, rewards)

        assert_all_finite(model)

        losses_by_seq += [-float(r.item()) for r in rewards]

        # pbar.set_postfix({"reward_mean": np.mean(rewards), "reward_std": np.std(rewards)})

    train_data_this_epoch_with_loss, overview = handle_loss_data(
        train_data_this_epoch, losses_by_seq, list(train_data_this_epoch.sort_by.values), extra_info="[train] "
    )

    return train_data_this_epoch_with_loss, overview


class DpoParams(TypedDict):
    beta: float
    weight_by_delta: bool
    exclude_close_threshold: float  # should be negative if you want to keep everything even if the score is the same


def run_train_dpo_epoch(
    model,
    tokenizer,
    optimizer,
    frozen_ref_model,
    train_dataset: PbAndGensDataset,
    dpo_params: DpoParams,
    train_batch_size: int = 2,
    per_epoch_count: Optional[int] = None,
    max_ctx: int = DEFAULT_MAX_CTX,
    ds_preprocess: Filter = NOOP_FILTER,
    run_prepare_inputs_for_generation: bool = True,
):
    if len(train_dataset) == 0:
        return PbAndGensWithLoss.from_list([]), {"warning": "no train"}

    model.train()

    if train_dataset.any_generation_required():
        ready_for_generation(model)

    losses_by_seq = []

    # should be stochastic on generation
    train_data_this_epoch_list = sample_epoch_with_gen(
        train_dataset,
        tokenizer,
        sort_by_len=False,
        shuffle=True,
        limit=per_epoch_count,
        dtype=inference_dtype_str,
        max_ctx=max_ctx,
    )
    train_data_this_epoch_list = ds_preprocess(train_data_this_epoch_list)
    train_data_this_epoch, data_loader = create_dpo_dataloader(
        model,
        tokenizer,
        final_training_data=train_data_this_epoch_list,
        batch_size=train_batch_size // 2,
        exclude_close_threshold=dpo_params["exclude_close_threshold"],
        shuffle=True,
        sort_by_len=False,
        run_prepare_inputs_for_generation=run_prepare_inputs_for_generation,
    )
    beta, weight_by_delta = dpo_params["beta"], dpo_params["weight_by_delta"]

    gc.collect()
    torch.cuda.empty_cache()

    model.cuda()
    frozen_ref_model = frozen_ref_model.cuda()
    send_optimizer_to(optimizer, "cuda")

    losses_by_seq = []

    for x_batch_tokens, x_batch_lens, y_batch_tokens, y_batch_lens, delta in tqdm(data_loader, desc="train"):
        optimizer.zero_grad()

        assert x_batch_tokens["input_ids"].shape == y_batch_tokens["input_ids"].shape
        batch_tokens = {k: torch.cat([x_batch_tokens[k], y_batch_tokens[k]], dim=0) for k in x_batch_tokens}
        batch_lens = torch.cat([x_batch_lens, y_batch_lens], dim=0)

        ref_neg_lp_sum_per_seq = get_masked_loss_by_batch(frozen_ref_model, batch_tokens, batch_lens, []).sum(dim=-1)
        neg_lp_sum_per_seq = get_masked_loss_by_batch(model, batch_tokens, batch_lens, []).sum(dim=-1)

        x_lp_sum_per_seq = -neg_lp_sum_per_seq[: len(x_batch_lens)]
        y_lp_sum_per_seq = -neg_lp_sum_per_seq[len(x_batch_lens) :]
        x_ref_sum_per_seq = -ref_neg_lp_sum_per_seq[: len(x_batch_lens)]
        y_ref_sum_per_seq = -ref_neg_lp_sum_per_seq[len(x_batch_lens) :]

        assert (
            x_lp_sum_per_seq.shape
            == y_lp_sum_per_seq.shape
            == x_ref_sum_per_seq.shape
            == y_ref_sum_per_seq.shape
            == delta.shape
        )
        # from https://github.com/huggingface/trl/blob/main/trl/trainer/dpo_trainer.py#L840
        delta = delta.to(x_lp_sum_per_seq.device)
        # delta = score_x - score_y > 0 iff x is chosen over y
        pi_logratios = (x_lp_sum_per_seq - y_lp_sum_per_seq) * torch.sign(delta)
        ref_logratios = (x_ref_sum_per_seq - y_ref_sum_per_seq) * torch.sign(delta)
        logits = pi_logratios - ref_logratios
        losses = -torch.nn.functional.logsigmoid(beta * logits)

        if weight_by_delta:
            losses = losses * torch.abs(delta)

        losses_by_seq.extend(losses.cpu().tolist())
        losses.sum().backward()

        # pbar.set_postfix({"loss": losses.mean().item()})

        # grad clipping
        torch.nn.utils.clip_grad_norm_(model.parameters(), 1.0)

        optimizer.step()

        assert_all_finite(model)

    train_data_this_epoch_with_loss, overview = handle_loss_data(
        train_data_this_epoch, losses_by_seq, train_data_this_epoch.sort_by.values, extra_info="[dpo_train] "
    )

    return train_data_this_epoch_with_loss, overview


YES_NO_TOKS = ("yes", "no")


def get_yes_no_tok_ids(tokenizer):
    return tuple(unwrap_single(tokenizer.encode(t, add_special_tokens=False)) for t in YES_NO_TOKS)


def freeze(model):
    frozen_ref_model = deepcopy(model)
    for p in frozen_ref_model.parameters():
        p.requires_grad = False
    return frozen_ref_model


def overwrite_on_the_fly_model_path(ds: PbAndGensDataset, current_model: str):
    ds.df["target_generation"] = ds.df.target_generation.apply(
        lambda x: (
            attrs.evolve(x, model_path=current_model)
            if (isinstance(x, OnTheFlyGeneration) and x.use_current_model)
            else x
        )
    )
    return ds


def jsonl_dump(f: TextIOWrapper, data: TypedDataFrame):
    for l in to_json(data):
        try:
            f.write(json.dumps(l) + "\n")
        except TypeError as e:
            print("Failed to write line")
            print("Line is:")
            print(l)
            raise e


@attrs.frozen
class SharedTrainConfig:
    n_epochs: int
    seed: int = 42
    per_epoch_count: Optional[int] = None
    save_weights_every: int | list[int] | Literal[False] = 1  # False means no saving, only weight folder
    eval_every: int | list[int] = 1
    lr_sched: Optional[Callable[[int], float]] = None
    eval_0: bool = True
    skip_eval_losses: bool = False
    skip_eval_gen: bool = False
    also_run_T1_eval: bool = False
    dpo_params: Optional[DpoParams] = None
    classification_params: Optional[ClassificationParams] = None
    ppo_params: Optional[PPOParams] = None
    train_ds_preprocess: Filter = NOOP_FILTER


def run_train(
    model,
    tokenizer,
    train_ds: PbAndGensDataset,
    val_ds: PbAndGensDataset,
    load_model_path: str,
    base_save_dir: str,
    shared_train_config: SharedTrainConfig,
    eval_batch_size: int = 8,
    train_batch_size: int = 2,
    lr: float = 1e-6,
    weight_decay: float = 1e-2,
    fast_temp_dir: str = "/tmp/tmp_models_other",  # I think just using /tmp will have competitive enough speeds due to cache/buff with linux?
    max_ctx: int = DEFAULT_MAX_CTX,
    eval_temperature: float = 0.0,
    run_prepare_inputs_for_generation: bool = True,
):
    n_epochs = shared_train_config.n_epochs
    seed = shared_train_config.seed
    per_epoch_count = shared_train_config.per_epoch_count
    save_weights_every = shared_train_config.save_weights_every
    eval_every = shared_train_config.eval_every
    lr_sched = shared_train_config.lr_sched
    eval_0 = shared_train_config.eval_0
    skip_eval_losses = shared_train_config.skip_eval_losses
    skip_eval_gen = shared_train_config.skip_eval_gen
    also_run_T1_eval = shared_train_config.also_run_T1_eval
    dpo_params = shared_train_config.dpo_params
    classification_params = shared_train_config.classification_params
    ppo_params = shared_train_config.ppo_params
    train_ds_preprocess = shared_train_config.train_ds_preprocess

    torch.manual_seed(seed)
    np.random.seed(seed)
    random.seed(seed)

    any_classification = any(is_classification_setting(x) for ds in [train_ds, val_ds] for x in ds.df.setting)
    any_non_classification = any(not is_classification_setting(x) for ds in [train_ds, val_ds] for x in ds.df.setting)

    assert any_classification != any_non_classification, "Can't mix classification and non-classification (yet)"

    if any_classification:
        assert classification_params is not None
    if any_non_classification:
        assert classification_params is None

    assert not (skip_eval_losses and skip_eval_gen)
    if not is_noop_filter(train_ds_preprocess):
        assert (
            classification_params is None and ppo_params is None
        ), f"Should be NOOP_FILTER, got {inspect.getsource(train_ds_preprocess)}"

    on_the_fly_with_current = train_ds.df.target_generation.apply(
        lambda x: (isinstance(x, OnTheFlyGeneration) and x.use_current_model)
    ).any()

    os.makedirs(base_save_dir, exist_ok=True)
    os.makedirs(fast_temp_dir, exist_ok=True)

    if not skip_eval_losses:
        assert not val_ds.any_generation_required()
    assert not val_ds.any_downsampling_required()

    assert get_device(model).type == "cpu"
    for x in [train_ds, val_ds]:
        for item in x:
            sol = item["target_generation"]
            assert isinstance(sol, OnTheFlyGeneration) or isinstance(sol, FixedGeneration)
            if isinstance(sol, FixedGeneration):
                assert "/home/paperspace" not in sol.solution
    assert tokenizer.padding_side == "left"

    regular_eval_fn = partial(run_eval, eval_temperature=eval_temperature)
    regular_eval_fn_T1 = partial(run_eval, eval_temperature=1.0)

    if dpo_params is not None:
        if not skip_eval_losses:
            raise NotImplementedError("DPO doesn't support eval losses")

        run_eval_fn = regular_eval_fn
        run_eval_fn_T1 = regular_eval_fn_T1
        run_train_epoch_fn = partial(
            run_train_dpo_epoch,
            frozen_ref_model=freeze(model),
            dpo_params=dpo_params,
            ds_preprocess=train_ds_preprocess,
        )
    elif ppo_params is not None:
        from trl import AutoModelForCausalLMWithValueHead

        if not skip_eval_losses:
            raise NotImplementedError("DPO doesn't support eval losses")

        model = AutoModelForCausalLMWithValueHead.from_pretrained(model)
        run_eval_fn = regular_eval_fn
        run_eval_fn_T1 = regular_eval_fn_T1
        run_train_epoch_fn = partial(run_ppo_train_epoch, frozen_ref_model=freeze(model), dpo_params=ppo_params)
    elif classification_params is not None:
        assert not also_run_T1_eval
        run_eval_fn = partial(run_classification_eval, params=classification_params)
        run_train_epoch_fn = partial(run_classification_train_epoch, params=classification_params)
    else:
        run_eval_fn = regular_eval_fn
        run_eval_fn_T1 = regular_eval_fn_T1
        run_train_epoch_fn = partial(run_train_epoch, ds_preprocess=train_ds_preprocess)

    if lr_sched is not None:
        lr = lr_sched(0)

    # add support for additional optimizers as needed
    optimizer = Lion(
        model.parameters(),
        lr=lr,
        weight_decay=weight_decay,
        use_triton=True,  # set this to True to use cuda kernel w/ Triton lang (Tillet et al)
    )

    current_load_model_path = load_model_path
    all_res = []

    def eval_and_save(epoch: int):
        model.cpu()
        send_optimizer_to(optimizer, "cpu")

        combined_val_data, eval_overview = run_eval_fn(
            model,
            tokenizer,
            val_dataset=val_ds,
            current_load_model_path=current_load_model_path,
            eval_batch_size=eval_batch_size,
            skip_eval_losses=skip_eval_losses,
            skip_eval_gen=skip_eval_gen,
            max_ctx=max_ctx,
            run_prepare_inputs_for_generation=run_prepare_inputs_for_generation,
        )

        with open(f"{base_save_dir}/val_res_epoch_{epoch}.jsonl", "w") as f:
            jsonl_dump(f, combined_val_data)

        if also_run_T1_eval:
            assert not skip_eval_gen
            model.cpu()
            send_optimizer_to(optimizer, "cpu")
            combined_val_data, eval_overview_T1 = run_eval_fn_T1(
                model,
                tokenizer,
                val_dataset=val_ds,
                current_load_model_path=current_load_model_path,
                eval_batch_size=eval_batch_size,
                skip_eval_losses=True,
                skip_eval_gen=skip_eval_gen,
                max_ctx=max_ctx,
                run_prepare_inputs_for_generation=run_prepare_inputs_for_generation,
            )

            eval_overview["T1"] = eval_overview_T1

            with open(f"{base_save_dir}/val_res_T1_epoch_{epoch}.jsonl", "w") as f:
                jsonl_dump(f, combined_val_data)

        with open(f"{base_save_dir}/eval_overview_epoch_{epoch}.json", "w") as f:
            json.dump(eval_overview, f, indent=4)

        return eval_overview

    if eval_0:
        eval_overview = eval_and_save(0)
        all_res.append({"epoch": 0, "eval_overview": eval_overview})

    for epoch in range(n_epochs):
        print(f"Epoch {epoch}")

        if lr_sched is not None:
            lr = float(lr_sched(epoch))
            print(f"Setting LR to {lr}")

            for param_group in optimizer.param_groups:
                param_group["lr"] = lr

        if train_ds.any_generation_required():
            model.cpu()
            send_optimizer_to(optimizer, "cpu")

        try:
            train_data, train_overview = run_train_epoch_fn(
                model,
                tokenizer,
                optimizer,
                train_dataset=overwrite_on_the_fly_model_path(train_ds, current_load_model_path),
                train_batch_size=train_batch_size,
                per_epoch_count=per_epoch_count,
                max_ctx=max_ctx,
                run_prepare_inputs_for_generation=run_prepare_inputs_for_generation,
            )
        except NotFiniteError as e:
            json.dump(all_res, open(f"{base_save_dir}/all_res.json", "w"), indent=4)
            raise e

        fin_epoch = epoch + 1

        # save train regardless of running val
        with open(f"{base_save_dir}/train_res_epoch_{fin_epoch}.jsonl", "w") as f:
            jsonl_dump(f, train_data)
        with open(f"{base_save_dir}/train_overview_epoch_{fin_epoch}.json", "w") as f:
            json.dump(train_overview, f, indent=4)

        this_saved = None
        assert (
            isinstance(save_weights_every, int) or isinstance(save_weights_every, list) or save_weights_every is False
        )
        non_tmp_save = (
            isinstance(save_weights_every, int) and save_weights_every and fin_epoch % save_weights_every == 0
        ) or (isinstance(save_weights_every, list) and fin_epoch in save_weights_every)
        if non_tmp_save:
            this_saved = f"{base_save_dir}/weights_epoch_{fin_epoch}"
            model.save_pretrained(this_saved)
            tokenizer.save_pretrained(this_saved)
            current_load_model_path = this_saved

        assert isinstance(eval_every, int) or isinstance(eval_every, list)
        do_eval = (isinstance(eval_every, int) and fin_epoch % eval_every == 0) or (
            isinstance(eval_every, list) and fin_epoch in eval_every
        )

        if (do_eval or on_the_fly_with_current) and (this_saved is None or ppo_params is not None):
            current_load_model_path = tmp_dir_of_gpu(fast_temp_dir)
            to_save_model = model if ppo_params is None else model.pretrained_model
            to_save_model.save_pretrained(current_load_model_path)
            tokenizer.save_pretrained(current_load_model_path)

        if do_eval:
            eval_overview = eval_and_save(fin_epoch)

            all_res.append({"epoch": fin_epoch, "eval_overview": eval_overview, "train_overview": train_overview})

    json.dump(all_res, open(f"{base_save_dir}/all_res.json", "w"), indent=4)
