import pandas as pd
from datasets import load_dataset

from sandbagging.dataset_utils import (
    SplitGetterConfig,
    make_split_getter_for_test_exclusive_splitting,
)
from sandbagging.wmdp_setting import NO_CAT
from sandbagging.pb_data_types import SettingData
from sandbagging.setting_types import Pbs, add_dict_field
from sandbagging.wmdp_dataset_info import multiple_choice_options

# %%


def make_problem_with_choices(question: str, choices: list[str]) -> str:
    out = str(question)
    for j, choice in enumerate(choices):
        out += "\n{}. {}".format(multiple_choice_options[j], choice)
    return out


# %%


def add_basic_fields_and_standardize(df: pd.DataFrame, dup_problems_set: set[str]) -> pd.DataFrame:
    df = df.copy()

    df["problem"] = [make_problem_with_choices(q, choices) for q, choices in zip(df["question"], df["choices"])]

    df.drop(columns=["question", "choices"], inplace=True)

    df["evaluation_data"] = [{}] * len(df)
    df["metadata"] = [{}] * len(df)
    df = df.assign(setting="wmdp", id=0, difficulty=NO_CAT)

    add_dict_field(df, "metadata", "correct_answer", df["answer"].values)

    df.drop(columns=["answer"], inplace=True)

    dup_count = 0
    out_problems_to_keep_index = []
    for i, p in enumerate(df["problem"]):
        if p in dup_problems_set:
            dup_count += 1
        else:
            dup_problems_set.add(p)
            out_problems_to_keep_index.append(i)

    df = df.iloc[out_problems_to_keep_index].reset_index(drop=True)
    # print(f"Found {dup_count} duplicates in the dataset")

    return Pbs.from_df(df)


# %%

wmdp_raw: pd.DataFrame = pd.concat(
    [
        load_dataset("cais/wmdp", f"wmdp-{cat}", split="test").to_pandas().assign(category=cat)
        for cat in ["cyber", "bio", "chem"]
    ]
)

dup_problems_set = set()  # avoid dups
wmdp = add_basic_fields_and_standardize(wmdp_raw, dup_problems_set=dup_problems_set)
wmdp_test = wmdp
all_wmdp = wmdp

# %%

split_config = SplitGetterConfig()  # defaults for now

get_splits = make_split_getter_for_test_exclusive_splitting(train=wmdp.iloc[:0], test=wmdp, config=split_config)

wmdp_setting_data = SettingData(get_splits=get_splits)

# %%
