from functools import partial
import os
from pathlib import Path
import shutil
from typing import Literal, Union, Callable
import random
from collections import defaultdict
from typing import Optional, Union
from collections import Counter, defaultdict
from typing_extensions import assert_never

import socket

hostname = socket.gethostname()

# if hostname == "u113912":
#     os.environ["CUDA_VISIBLE_DEVICES"] = "0,1,3,4,5"


import ray
import attrs
import numpy as np

from sandbagging.all_data import all_settings
from sandbagging.all_train import ClassificationParams, SharedTrainConfig
from sandbagging.mmlu_dataset_info import (
    multiple_choice_options as mmlu_multiple_choice_options,
    multiple_choice_options_full_str as mmlu_multiple_choice_options_full_str,
    start_assistant_response as mmlu_start_assistant_response,
)
from sandbagging.wmdp_dataset_info import (
    multiple_choice_options as wmdp_multiple_choice_options,
    multiple_choice_options_full_str as wmdp_multiple_choice_options_full_str,
    start_assistant_response as wmdp_start_assistant_response,
)
from sandbagging.hellaswag_dataset_info import (
    multiple_choice_options as hellaswag_multiple_choice_options,
    multiple_choice_options_full_str as hellaswag_multiple_choice_options_full_str,
    start_assistant_response as hellaswag_start_assistant_response,
)
from sandbagging.all_ray_run_utils import (
    ClassificationParamsExtra,
    remap_names,
    StoredInferenceArgs,
    TrainArgs,
    ArgType,
    remap_names,
    run_tasks,
    internet_model_names,
)
from sandbagging.dataset_wrapping import cache_dir
from sandbagging.basic_model_info import (
    DEEPSEEK,
    FIXEDRDM,
    MISTRAL7B,
    STABLE_ZEPH2B,
    DEEPSEEK_MATH,
    MODEL_INFOS,
    OLD_LLAMA7B,
    PYTHIA1B,
    PYTHIA3B,
    PYTHIA400M,
    PYTHIA7B,
    TINY_LLAMA1B,
    UNIFORM,
    UNIFORMPP,
    ContinuousTarget,
    model_to_c_targets,
)
from sandbagging.fill_pbs_df import (
    AllClassificationToImitate,
    ClassificationModelToImitateArgs,
    ClassificationUseCorrectAnswerLabel,
    ClassificationUseSavedProbs,
    ClassificationCallableProbs,
    EitherModelToImitateArgs,
    TrainToImitateArgs,
    ModelToImitateArgs,
    apply_hardening,
    fill_pbs_df,
)
from sandbagging.all_setup_for_runs import (
    CLASSIFICATION_DEFAULT_FULL_SWEEP,
    CLASSIFICATION_NARROW_SWEEP_TO_EXTRA_LARGE_SAMP_COUNTS,
    CLASSIFICATION_ONLY_SMALL_NARROW_SWEEP_SAMP_COUNTS,
    CLASSIFICATION_ONLY_SOMEWHAT_SMALL_NARROW_SWEEP_SAMP_COUNTS,
    CLASSIFICATION_SLIGHTLY_REDUCED_SWEEP,
    CODE_GPT_4_LARGE_SAVED_INFERENCE_NAME,
    CODE_GPT_4_SAVED_INFERENCE_NAME,
    BASELINE_N_EPOCHS_MCQA,
    CONST_EVAL_SEEDS,
    # CRITIQUE_GOLD_LARGE_SAVED_INFERENCE_NAME,
    CRITIQUE_GOLD_SAVED_INFERENCE_NAME,
    CRITIQUE_INIT_FT_EPOCHS,
    DPO_INIT_N_EPOCHS,
    DPO_SEEDS,
    MATH_GPT_4_SAVED_INFERENCE_NAME,
    MATH_SA_GOLD_SAVED_INFERENCE_NAME,
    MATH_SA_INIT_FT_EPOCHS,
    N_EPOCHS_FULL_BASIC_GOLD_DEMO,
    N_EPOCHS_PWD_LOCK_MCQA,
    NARROW_SWEEP_TO_EXTRA_LARGE_SAMP_COUNTS,
    DiffPwdByDomainArgs,
    DifferentElicit,
    DifferentElicitPre,
    ScoreFn,
    difficulty_groupings_by_setting,
    category_groupings_by_setting,
    BIG_SWEEP_SAMP_COUNTS,
    DPO_EVAL_EVERY,
    DPO_LR_RED_FACTOR,
    DPO_N_EPOCHS,
    EI_SAMP_COUNTS,
    EI_SEEDS,
    SKIP_SOME_LARGE_SWEEP_SAMP_COUNTS,
    CODE_INIT_FT_EPOCHS,
    NARROW_SWEEP_TO_LARGE_SAMP_COUNTS,
    ElicitNEpochsAndEvalEpochsConfig,
    elicit_val_df,
    get_baseline_name_suffix,
    get_basic_multiplier_format,
    get_constant_evaluate_elict_name_suffix,
    get_dpo,
    get_dpo_name_suffix,
    get_ei,
    get_ei_n_epochs_and_eval_epochs,
    get_ei_name_suffix,
    get_elicit_n_epochs_and_eval_epochs,
    get_gt_full_data_elict_name_suffix,
    get_gt_harden_opts,
    get_limited_samp_elicit,
    get_lr_mul_suffix,
    get_lr_muls_for_baseline,
    get_name_from_suffix,
    get_name_from_tup_suffix,
    get_pwd_lock_name_suffix,
    get_samp_elicit_name_suffix,
    get_samp_elicit_name_suffix_gt_or_same_weak,
    make_pwd_lock_lr_fancy_sched,
    make_weak_strong_to_imitate_args,
    weak_models_run_lock_elicit_on_subset_of_domains_by_setting,
    weak_models_run_lock_red_team_generalize,
    get_weights_path,
    make_baseline_train_val_data,
    make_pwd_lock_train_val_data_default,
    run_full_password_lock_setup,
    skip_train_ones_by_setting,
    BASELINE_N_EPOCHS,
    SAMP_SEEDS,
    WIDE_SPARSE_SAMP_COUNTS,
    stored_inference_name_for_save_name_suffix,
    N_EPOCHS_PWD_LOCK,
    restricted_different_elicit_pairs_by_setting,
    DifferentElicitFromModel,
    DifferentElicitFixedSaveName,
    weak_models_run_lock,
    score_fn_by_setting,
    all_weak_models_by_setting,
    focus_on_weak_model_by_setting,
    intermediate_weak_models_by_setting,
)
from sandbagging.pb_splits_by_setting import core_splits_default_seed
from sandbagging.pb_data_types import PbsSplits, default_get_splits_seed
from sandbagging.passwords import (
    DEFAULT_PASSWORD,
    BasicPwd,
    NoPwd,
    Pwd,
    WhichDomainType,
)
import sandbagging.utils.optional as op
from sandbagging.setting_types import Pbs, SettingKind
from sandbagging.code_data import lock_print_correctness_path
from sandbagging.setting_types import PbAndGens, SettingKind

# %%

exogenous_out_names = internet_model_names | {
    MATH_GPT_4_SAVED_INFERENCE_NAME,
    CRITIQUE_GOLD_SAVED_INFERENCE_NAME,
    CODE_GPT_4_LARGE_SAVED_INFERENCE_NAME,
    CODE_GPT_4_SAVED_INFERENCE_NAME,
    MATH_SA_GOLD_SAVED_INFERENCE_NAME,
}


def init_ft_info(
    setting: SettingKind, splits: PbsSplits, base_model: str, save_name: str, ft_file: str, epochs: int
) -> tuple[TrainArgs, str]:
    start_model_full_name = remap_names[MODEL_INFOS[base_model].full_name]
    lr = MODEL_INFOS[base_model].size.standard_lr_by_setting[setting]
    val_data = elicit_val_df(splits.val_split, password=DEFAULT_PASSWORD, no_password=NoPwd())
    ft_train_data = get_limited_samp_elicit(
        splits,  # we initialize strong models (used for pwd demos and similar) with the elicit split!
        samp_count=None,
        seed=0,
        from_source=ModelToImitateArgs(save_name_for_inference=ft_file),
    )
    assert len(ft_train_data) > 0 and len(val_data) > 0
    args = TrainArgs(
        start_model_full_name,
        base_model,
        train_df=ft_train_data,
        val_df=val_data,
        save_model_name=save_name,
        lr=lr,
        shared_train_config=SharedTrainConfig(
            n_epochs=epochs,
            save_weights_every=epochs,
            eval_every=1,
            eval_0=False,
            skip_eval_losses=True,
        ),
        load_all_fixed=True,
        priority=0,
    )
    return args, get_weights_path(save_name, epochs)


def critique_init_info() -> tuple[list[TrainArgs], str, str]:
    base_strong_model = DEEPSEEK
    splits = core_splits_default_seed["critique"]

    val_data = elicit_val_df(splits.val_split, password=DEFAULT_PASSWORD, no_password=NoPwd())
    lr = MODEL_INFOS[base_strong_model].size.standard_lr_by_setting["critique"]

    ft_train_args, init_ft_path = init_ft_info(
        "critique",
        splits,  # we initialize strong models (used for pwd demos and similar) with the elicit split!
        base_strong_model,
        "critique_init_ft",
        CRITIQUE_GOLD_SAVED_INFERENCE_NAME,
        CRITIQUE_INIT_FT_EPOCHS,
    )

    dpo_train_data, train_ds_preprocess = get_dpo(
        splits, pwd_lock_weights_path=init_ft_path, score_fn="ext_corr", is_it=True
    )
    dpo_train_data_d, train_ds_preprocess_d = get_dpo(
        splits, pwd_lock_weights_path=init_ft_path, score_fn="corr", is_it=True
    )
    dpo_save_name = "critique_init_dpoi"

    def get_dpo_args(dpo_train_data, train_ds_preprocess, dpo_save_name):
        return TrainArgs(
            init_ft_path,
            base_strong_model,
            train_df=dpo_train_data,
            val_df=val_data,
            save_model_name=dpo_save_name,
            lr=lr * DPO_LR_RED_FACTOR,
            shared_train_config=SharedTrainConfig(
                n_epochs=DPO_INIT_N_EPOCHS,
                save_weights_every=DPO_INIT_N_EPOCHS,
                eval_every=DPO_EVAL_EVERY,
                eval_0=False,
                skip_eval_losses=True,
                train_ds_preprocess=train_ds_preprocess,
                seed=0,
                dpo_params={"beta": 0.1, "weight_by_delta": True, "exclude_close_threshold": 0},
            ),
            priority=0,
        )

    assert len(val_data) > 0 and len(dpo_train_data) > 0

    return (
        [
            ft_train_args,
            get_dpo_args(dpo_train_data, train_ds_preprocess, dpo_save_name),
            get_dpo_args(dpo_train_data_d, train_ds_preprocess_d, dpo_save_name + "_discrete"),
        ],
        get_weights_path(dpo_save_name, DPO_N_EPOCHS),
        base_strong_model,
    )


def code_init_info() -> tuple[list[TrainArgs], str, str]:
    base_strong_model = DEEPSEEK
    splits = core_splits_default_seed["code"]

    ft_train_args, init_ft_path = init_ft_info(
        "code", splits, base_strong_model, "code_init_ft", CODE_GPT_4_LARGE_SAVED_INFERENCE_NAME, CODE_INIT_FT_EPOCHS
    )
    return [ft_train_args], init_ft_path, base_strong_model


def math_sa_init_info() -> tuple[list[TrainArgs], str, str]:
    base_strong_model = DEEPSEEK_MATH
    splits = core_splits_default_seed["math_spanish_ascii"]

    ft_train_args, init_ft_path = init_ft_info(
        "math_spanish_ascii",
        splits,
        base_strong_model,
        "math_spanish_ascii_init_ft",
        MATH_SA_GOLD_SAVED_INFERENCE_NAME,
        MATH_SA_INIT_FT_EPOCHS,
    )
    return [ft_train_args], init_ft_path, base_strong_model


def add_ei(
    splits: PbsSplits,
    base_strong_model_name: str,
    pwd_lock_save_name_suffix: str,
    pwd_lock_weights_path: str,
    baseline_weights_path: str,
    setting: str,
    default_elicit_val_data: PbAndGens,
    score_fns: list[ScoreFn],
    all_train_out: list,
):
    name_from_suffix = get_name_from_suffix(setting)

    # for weak in [False]:
    for weak in [False, True]:
        for score_fn in score_fns:  # score_fn_by_setting[setting]:
            for samp_count in EI_SAMP_COUNTS:
                # for epoch_mul in [1, 4]:
                for epoch_mul in [1]:
                    seeds = EI_SEEDS if not weak else [EI_SEEDS[0]]
                    for seed in seeds:
                        train_data, train_ds_preprocess = get_ei(
                            splits,
                            samp_count,
                            score_fn=score_fn,
                            from_weak_model=baseline_weights_path if weak else None,
                        )
                        n_epochs_this_unlock, eval_every_this_unlock = get_ei_n_epochs_and_eval_epochs(
                            samp_count=samp_count, epoch_mul=epoch_mul
                        )
                        ei_save_name = name_from_suffix(
                            get_ei_name_suffix(
                                pwd_lock_name_suffix=pwd_lock_save_name_suffix,
                                samp_count=samp_count,
                                seed=seed,
                                score_fn=score_fn,
                                weak=weak,
                                epoch_mul=epoch_mul,
                            )
                        )
                        model_size = MODEL_INFOS[base_strong_model_name].size
                        lr = model_size.standard_lr_by_setting.get(setting, model_size.standard_lr_val)
                        all_train_out.append(
                            TrainArgs(
                                pwd_lock_weights_path,
                                base_strong_model_name,
                                train_df=train_data,
                                val_df=default_elicit_val_data,
                                save_model_name=ei_save_name,
                                lr=lr,
                                shared_train_config=SharedTrainConfig(
                                    n_epochs=n_epochs_this_unlock,
                                    save_weights_every=False,
                                    eval_every=eval_every_this_unlock,
                                    eval_0=False,
                                    skip_eval_losses=True,
                                    train_ds_preprocess=train_ds_preprocess,
                                    seed=seed,
                                ),
                                priority=(
                                    3
                                    + weak
                                    + (epoch_mul != 1)
                                    + seeds.index(seed)
                                    + 12
                                    * ("cut_low" in pwd_lock_save_name_suffix or "saved" in pwd_lock_save_name_suffix)
                                ),
                            )
                        )


def all_train_info() -> list[ArgType]:
    all_train_out: list[ArgType] = []

    critique_init_args, critique_init_load_name, critique_base_strong_model = critique_init_info()
    all_train_out.extend(critique_init_args)
    code_init_args, code_init_load_name, code_base_strong_model = code_init_info()
    all_train_out.extend(code_init_args)
    math_sa_init_args, math_sa_init_load_name, math_sa_base_strong_model = math_sa_init_info()
    all_train_out.extend(math_sa_init_args)

    start_model_by_setting: dict[SettingKind, tuple[str, str]] = {
        "math": (remap_names[MODEL_INFOS[DEEPSEEK_MATH].full_name], DEEPSEEK_MATH),
        "code": (code_init_load_name, code_base_strong_model),
        "critique": (critique_init_load_name, critique_base_strong_model),
        "mmlu": (remap_names[MODEL_INFOS[MISTRAL7B].full_name], MISTRAL7B),
        # "wmdp": (remap_names[MODEL_INFOS[MISTRAL7B].full_name], MISTRAL7B),
        # "hellaswag": (remap_names[MODEL_INFOS[MISTRAL7B].full_name], MISTRAL7B),
        "math_spanish_ascii": (math_sa_init_load_name, math_sa_base_strong_model),
    }

    start_model_by_setting_tup: dict[tuple[SettingKind, ...], tuple[str, str]] = {
        ("math", "math_spanish_ascii"): (remap_names[MODEL_INFOS[DEEPSEEK_MATH].full_name], DEEPSEEK_MATH, PYTHIA1B),
        ("code", "math"): (remap_names[MODEL_INFOS[DEEPSEEK].full_name], DEEPSEEK, PYTHIA1B),
    }

    extra_priority_by_setting: defaultdict[SettingKind | tuple[SettingKind, ...], int] = defaultdict(int)
    extra_priority_by_setting["hellaswag"] = 6

    gold_demo_imitate_args_by_setting: dict[SettingKind, EitherModelToImitateArgs] = {
        "math": ModelToImitateArgs(
            model_path=None,  # only supports fixed
            save_name_for_inference=MATH_GPT_4_SAVED_INFERENCE_NAME,
        ),
        "math_spanish_ascii": ModelToImitateArgs(
            model_path=None,  # only supports fixed
            save_name_for_inference=MATH_SA_GOLD_SAVED_INFERENCE_NAME,
        ),
        "code": ModelToImitateArgs(
            model_path=None,  # only supports fixed
            save_name_for_inference=CODE_GPT_4_SAVED_INFERENCE_NAME,
        ),
        "critique": ModelToImitateArgs(
            model_path=None,  # only supports fixed
            save_name_for_inference=CRITIQUE_GOLD_SAVED_INFERENCE_NAME,
        ),
        "mmlu": ClassificationModelToImitateArgs(
            num_classes=len(mmlu_multiple_choice_options),
            start_assistant_response=mmlu_start_assistant_response,
            to_imitate=ClassificationUseCorrectAnswerLabel(),
        ),
        "wmdp": ClassificationModelToImitateArgs(
            num_classes=len(wmdp_multiple_choice_options),
            start_assistant_response=wmdp_start_assistant_response,
            to_imitate=ClassificationUseCorrectAnswerLabel(),
        ),
        "hellaswag": ClassificationModelToImitateArgs(
            num_classes=len(hellaswag_multiple_choice_options),
            start_assistant_response=hellaswag_start_assistant_response,
            to_imitate=ClassificationUseCorrectAnswerLabel(),
        ),
    }

    classification_params_extra_by_setting: dict[SettingKind, Optional[ClassificationParamsExtra]] = {
        "math": None,
        "code": None,
        "critique": None,
        "math_spanish_ascii": None,
        "mmlu": ClassificationParamsExtra(
            ClassificationParams(
                multiple_choice_options=mmlu_multiple_choice_options,
                multiple_choice_options_full_str=mmlu_multiple_choice_options_full_str,
            ),
            start_assistant_response=mmlu_start_assistant_response,
        ),
        "wmdp": ClassificationParamsExtra(
            ClassificationParams(
                multiple_choice_options=wmdp_multiple_choice_options,
                multiple_choice_options_full_str=wmdp_multiple_choice_options_full_str,
            ),
            start_assistant_response=wmdp_start_assistant_response,
        ),
        "hellaswag": ClassificationParamsExtra(
            ClassificationParams(
                multiple_choice_options=hellaswag_multiple_choice_options,
                multiple_choice_options_full_str=hellaswag_multiple_choice_options_full_str,
            ),
            start_assistant_response=hellaswag_start_assistant_response,
        ),
    }

    baseline_model_to_imitate_by_setting: defaultdict[str, dict[str, EitherModelToImitateArgs]] = defaultdict(dict)
    default_pwd_lock_strong_imitate_args_per_setting: defaultdict[str, EitherModelToImitateArgs] = defaultdict(dict)

    print_run_counts = True

    for setting, (start_strong_model_path, base_strong_model_name) in start_model_by_setting.items():
        start_pwd_lock_path = remap_names[MODEL_INFOS[base_strong_model_name].full_name]
        extra_priority_this_setting = extra_priority_by_setting[setting]
        if print_run_counts:
            print(f"{setting=}")

        num_runs_at_start_setting = len(all_train_out)

        classification_params_extra = classification_params_extra_by_setting[setting]
        classification_params = op.map(classification_params_extra, lambda x: x.classification_params)
        has_classification = classification_params_extra is not None

        splits = core_splits_default_seed[setting]
        default_elicit_val_data = elicit_val_df(
            splits.val_split,
            password=DEFAULT_PASSWORD,
            no_password=NoPwd(),
            classification_params_extra=classification_params_extra,
        )

        all_weak_models = all_weak_models_by_setting[setting]
        all_models_for_baseline = list(all_weak_models)

        name_from_suffix = get_name_from_suffix(setting)

        self_strong_saved_inference_name_suffix = stored_inference_name_for_save_name_suffix(base_strong_model_name)
        self_strong_saved_inference_name = name_from_suffix(self_strong_saved_inference_name_suffix)

        self_strong_imitate_args_pre: EitherModelToImitateArgs | None = None
        if not has_classification:
            init_suffix = "_init" if setting != "math" else ""

            all_train_out.append(
                StoredInferenceArgs(
                    load_model_path=start_strong_model_path,
                    save_inference_name=self_strong_saved_inference_name + init_suffix,
                    dataset_to_infer=splits.combined_data,
                )
            )

            self_strong_imitate_args_pre = ModelToImitateArgs(
                model_path=start_strong_model_path,
                save_name_for_inference=self_strong_saved_inference_name + init_suffix,
            )

            if setting == "math":
                # for math, the strong model wasn't trained on the elicit split so it's fine
                baseline_model_to_imitate_by_setting[setting][base_strong_model_name] = self_strong_imitate_args_pre
            else:
                all_models_for_baseline.append(base_strong_model_name)

            baseline_imitate_args = self_strong_imitate_args_pre
        else:
            baseline_imitate_args_pre = gold_demo_imitate_args_by_setting[setting]
            baseline_imitate_args = baseline_imitate_args_pre

        gold_demo_imitate_args = gold_demo_imitate_args_by_setting[setting]

        assert isinstance(gold_demo_imitate_args, (ModelToImitateArgs, ClassificationModelToImitateArgs))

        focus_on_weak_model = focus_on_weak_model_by_setting[setting]
        pwd_lock_extra_harden_exps_model_name = focus_on_weak_model
        pwd_lock_extra_sweep_and_prio_model_name = focus_on_weak_model

        all_baselines: list[tuple[str, str]] = []

        baseline_n_epochs_here = BASELINE_N_EPOCHS_MCQA if has_classification else BASELINE_N_EPOCHS

        # training weak to imitate strong
        for base_model_for_baseline in all_models_for_baseline:
            model_info = MODEL_INFOS[base_model_for_baseline]
            if base_model_for_baseline in skip_train_ones_by_setting[setting]:
                assert not has_classification
                continue

            baseline_save_name_suffix = get_baseline_name_suffix(base_model_for_baseline)
            baseline_save_name = name_from_suffix(baseline_save_name_suffix)
            for lr_mul in get_lr_muls_for_baseline(setting):
                for extra_epochs in [None, 4] if lr_mul == 1.0 and has_classification else [None]:
                    final_load_path = remap_names[model_info.full_name]

                    lr = model_info.size.standard_lr_by_setting[setting] * lr_mul

                    n_epochs_here = baseline_n_epochs_here + (0 if extra_epochs is None else extra_epochs)

                    use_fancy_sched = has_classification

                    if use_fancy_sched:
                        lr_sched = make_pwd_lock_lr_fancy_sched(lr=lr, n_epochs=n_epochs_here)
                    else:
                        lr_sched = None

                    baseline_save_name_here = str(baseline_save_name)
                    if lr_mul != 1.0:
                        baseline_save_name_here += get_lr_mul_suffix(lr_mul)
                    if extra_epochs is not None:
                        baseline_save_name_here += f"_extra_epochs_{extra_epochs}"

                    train_data, val_data = make_baseline_train_val_data(
                        splits=splits,
                        strong_imitate_args=baseline_imitate_args,
                    )

                    target_eval_rate = (
                        2 if has_classification else 4
                    )  # cheaper for classification, so run eval more frequently

                    all_train_out.append(
                        TrainArgs(
                            final_load_path,
                            base_model_for_baseline,
                            val_df=val_data,
                            train_df=train_data,
                            save_model_name=baseline_save_name_here,
                            lr=lr,
                            shared_train_config=SharedTrainConfig(
                                save_weights_every=False if lr_mul != 1.0 else min(4, n_epochs_here),
                                n_epochs=n_epochs_here,
                                eval_every=min(target_eval_rate, n_epochs_here),
                                eval_0=lr_mul == 1.0,
                                classification_params=classification_params,
                                lr_sched=lr_sched,
                            ),
                            priority=(0 if lr_mul == 1.0 and extra_epochs is None else 5)
                            + (0 if base_model_for_baseline == focus_on_weak_model else 2)
                            + extra_priority_this_setting,
                        )
                    )

            baseline_weights_path = get_weights_path(baseline_save_name, baseline_n_epochs_here)
            all_baselines.append((baseline_weights_path, base_model_for_baseline))

        all_baselines.extend(
            [
                (MODEL_INFOS[base_model_for_baseline].full_name, base_model_for_baseline)
                for base_model_for_baseline in skip_train_ones_by_setting[setting]
            ]
        )

        def make_classification_model_to_imitate(to_imitate: AllClassificationToImitate):
            assert isinstance(gold_demo_imitate_args, ClassificationModelToImitateArgs)
            return attrs.evolve(gold_demo_imitate_args, to_imitate=to_imitate)

        # generating weak solutions (to imitate later)
        for baseline_weights_path, base_model_for_baseline in all_baselines:
            baseline_save_inference_name_suffix = stored_inference_name_for_save_name_suffix(base_model_for_baseline)
            baseline_save_inference_name = name_from_suffix(baseline_save_inference_name_suffix)

            all_train_out.append(
                StoredInferenceArgs(
                    load_model_path=baseline_weights_path,
                    save_inference_name=baseline_save_inference_name,
                    dataset_to_infer=splits.combined_data,
                    classification_params_extra=classification_params_extra,
                    priority=(0 if base_model_for_baseline == focus_on_weak_model else 3) + extra_priority_this_setting,
                )
            )

            if not has_classification:
                to_imitate = ModelToImitateArgs(
                    model_path=baseline_weights_path,
                    save_name_for_inference=baseline_save_inference_name,
                )
            else:
                to_imitate = make_classification_model_to_imitate(
                    to_imitate=ClassificationUseSavedProbs(
                        save_name_for_inference=baseline_save_inference_name, harden=False
                    ),
                )

            baseline_model_to_imitate_by_setting[setting][base_model_for_baseline] = to_imitate

        # now should be filled out for everything including classification

        if not has_classification:
            default_pwd_lock_strong_imitate_args = op.unwrap(self_strong_imitate_args_pre)
            default_pwd_lock_strong_imitate_args_per_setting[setting] = default_pwd_lock_strong_imitate_args
        else:
            default_pwd_lock_strong_imitate_args = gold_demo_imitate_args

            assert isinstance(default_pwd_lock_strong_imitate_args, ClassificationModelToImitateArgs)
            assert isinstance(default_pwd_lock_strong_imitate_args.to_imitate, ClassificationUseCorrectAnswerLabel)

        def get_basic_elicit_train_args(
            pwd_lock_weights_path: str,
            train_data: PbAndGens,
            elicit_save_name: str,
            n_epochs_this_unlock: int,
            eval_every_this_unlock: int | list[int],
            priority: int,
            per_epoch_count: Optional[int] = None,
            lr: Optional[float] = None,
            elicit_val_data: PbAndGens = default_elicit_val_data,
            save_weights_every: int | list[int] | Literal[False] = False,
            core_setting: Optional[SettingKind] = None,
        ):
            if lr is None:
                s_typing_fix: SettingKind = setting
                lr_fin = MODEL_INFOS[base_strong_model_name].size.standard_lr_by_setting[s_typing_fix]
            else:
                lr_fin = lr
            return TrainArgs(
                pwd_lock_weights_path,
                base_strong_model_name,
                train_df=train_data,
                val_df=elicit_val_data,
                save_model_name=elicit_save_name,
                lr=lr_fin,
                shared_train_config=SharedTrainConfig(
                    n_epochs=n_epochs_this_unlock,
                    save_weights_every=save_weights_every,
                    eval_every=eval_every_this_unlock,
                    eval_0=False,
                    skip_eval_losses=True,
                    per_epoch_count=per_epoch_count,
                    classification_params=classification_params,
                ),
                priority=priority + extra_priority_this_setting,
                core_setting=core_setting,
            )

        def run_basic_full_gold_demo_elicit(
            pwd_lock_weights_path: str,
            pwd_lock_save_name_suffix: str,
            priority: int = 6,
            shuf_seed: int = 283974,  # should be fine to just use one seed
        ):
            train_data = get_limited_samp_elicit(
                splits=splits,
                samp_count=None,
                seed=shuf_seed,
                from_source=gold_demo_imitate_args,
            )

            s_typing_fix: SettingKind = setting
            lr = MODEL_INFOS[base_strong_model_name].size.standard_lr_by_setting[s_typing_fix]

            elicit_save_name_suffix = get_gt_full_data_elict_name_suffix(pwd_lock_name_suffix=pwd_lock_save_name_suffix)
            elicit_save_name = name_from_suffix(elicit_save_name_suffix)

            all_train_out.append(
                get_basic_elicit_train_args(
                    pwd_lock_weights_path=pwd_lock_weights_path,
                    train_data=train_data,
                    elicit_save_name=elicit_save_name,
                    n_epochs_this_unlock=N_EPOCHS_FULL_BASIC_GOLD_DEMO,
                    eval_every_this_unlock=1,
                    priority=priority,
                    lr=lr,
                )
            )

        DEFAULT_GT_EXTRA_MULT = (
            # 1.5 if has_classification or setting == "math" else 1.0
            1.5
            if has_classification
            else 1.0
        )
        DEFAULT_SAME_WEAK_EXTRA_MULT = 2.0

        def run_basic_limited_samp_elicit(
            samp_count: int,
            base_weak_model_name: str,
            pwd_lock_weights_path: str,
            pwd_lock_save_name_suffix: str,
            extra_res: bool = False,
            extra_extra_res: bool = False,
            low_res: bool = False,
            lr_mul: float = 1.0,
            priority: int = 6,
            all_force_elicit_data: Optional[list[Pbs]] = None,
            also_run_harden: bool = has_classification,
            allow_non_harden_if_run_harden: bool = False,
            samp_seeds: list[int] = SAMP_SEEDS,
            gt_extra_prio: int = 0,
            same_weak_extra_prio: int = 0,
            additional_extra_mult: float = 1.0,
            just_gt: bool = False,
            elicit_val_data: PbAndGens = default_elicit_val_data,
            map_df: Callable[[PbAndGens], PbAndGens] = lambda x: x,
        ):
            if also_run_harden:
                assert has_classification

            gt_harden_opts = get_gt_harden_opts(
                also_run_harden=also_run_harden,
                allow_non_harden_if_run_harden=allow_non_harden_if_run_harden,
                just_gt=just_gt,
            )

            if all_force_elicit_data is not None:
                assert len(all_force_elicit_data) == len(samp_seeds)

            for is_gt, harden in gt_harden_opts:
                for samp_idx, samp_seed in enumerate(samp_seeds):
                    this_imitate = (
                        gold_demo_imitate_args
                        if is_gt
                        else apply_hardening(
                            baseline_model_to_imitate_by_setting[setting][base_weak_model_name], harden=harden
                        )
                    )

                    if harden:
                        assert has_classification
                        assert not is_gt

                    train_data = get_limited_samp_elicit(
                        splits=splits,
                        samp_count=samp_count,
                        seed=samp_seed,
                        from_source=this_imitate,
                        force_train_data=op.map(all_force_elicit_data, lambda x: x[samp_idx]),
                    )
                    train_data = map_df(train_data)
                    n_epochs_this_unlock, eval_every_this_unlock = get_elicit_n_epochs_and_eval_epochs(
                        samp_count=samp_count,
                        config=ElicitNEpochsAndEvalEpochsConfig(
                            extra_res=extra_res,
                            extra_extra_res=extra_extra_res,
                            low_res=low_res,
                            extra_mult=(DEFAULT_GT_EXTRA_MULT if is_gt else DEFAULT_SAME_WEAK_EXTRA_MULT)
                            * additional_extra_mult
                            * ((1 / 1.5) if has_classification and samp_count >= 2048 and is_gt else 1.0),  # hack
                        ),
                    )

                    s_typing_fix: SettingKind = setting
                    lr = MODEL_INFOS[base_strong_model_name].size.standard_lr_by_setting[s_typing_fix] * lr_mul

                    elicit_save_name_suffix = get_samp_elicit_name_suffix_gt_or_same_weak(
                        pwd_lock_name_suffix=pwd_lock_save_name_suffix,
                        is_gt=is_gt,
                        samp_count=samp_count,
                        seed=samp_seed,
                    )
                    elicit_save_name = name_from_suffix(elicit_save_name_suffix)

                    if has_classification and (not harden) and (not is_gt):
                        elicit_save_name += "_non_harden"
                    if lr_mul != 1.0:
                        elicit_save_name += get_lr_mul_suffix(lr_mul)

                    special_code_uniformpp_mul = (
                        0.5 if setting == "code" and base_weak_model_name in (UNIFORMPP, FIXEDRDM) else 1.0
                    )

                    all_train_out.append(
                        get_basic_elicit_train_args(
                            pwd_lock_weights_path=pwd_lock_weights_path,
                            train_data=train_data,
                            elicit_save_name=elicit_save_name,
                            n_epochs_this_unlock=n_epochs_this_unlock,
                            eval_every_this_unlock=eval_every_this_unlock,
                            priority=priority + (gt_extra_prio if is_gt else same_weak_extra_prio),
                            lr=lr * special_code_uniformpp_mul,
                            elicit_val_data=elicit_val_data,
                        )
                    )

        def run_baseline_samp_elicits(
            base_weak_model_name: str,
            pwd_lock_weights_path: str,
            pwd_lock_save_name_suffix: str,
            samp_counts: list[int],
            base_priority: int,
            allow_non_harden_if_run_harden: bool = False,
            just_gt: bool = False,
            elicit_val_data: PbAndGens = default_elicit_val_data,
            map_df: Callable[[PbAndGens], PbAndGens] = lambda x: x,
        ):
            # I don't think we actually want or need this
            # run_basic_full_gold_demo_elicit(
            #     pwd_lock_weights_path=pwd_lock_weights_path,
            #     pwd_lock_save_name_suffix=pwd_lock_save_name_suffix,
            #     priority=base_priority,
            #     map_df=map_df,
            # )

            for samp_count in samp_counts:
                run_basic_limited_samp_elicit(
                    samp_count=samp_count,
                    base_weak_model_name=base_weak_model_name,
                    pwd_lock_weights_path=pwd_lock_weights_path,
                    pwd_lock_save_name_suffix=pwd_lock_save_name_suffix,
                    priority=base_priority + 1,
                    allow_non_harden_if_run_harden=allow_non_harden_if_run_harden,
                    just_gt=just_gt,
                    elicit_val_data=elicit_val_data,
                    map_df=map_df,
                )

        weak_run_lock = weak_models_run_lock[setting]

        n_epochs_pwd_lock_here = N_EPOCHS_PWD_LOCK if not has_classification else N_EPOCHS_PWD_LOCK_MCQA

        @attrs.frozen
        class LockingArgs:
            base_weak_model_name: str
            force_weak_harden_in_pwd_lock_train: bool = False
            apply_train_dupping_with_mul: Optional[float] = None
            c_targets: ContinuousTarget = 0
            cut_low_dup: bool = False

        weak_run_lock_w_extra: list[LockingArgs] = [
            LockingArgs(base_weak_model_name=x, c_targets=model_to_c_targets(x)) for x in weak_run_lock
        ]

        if has_classification:
            assert pwd_lock_extra_harden_exps_model_name in weak_run_lock
            # default to harden for classification!
            weak_run_lock_w_extra = [
                attrs.evolve(x, force_weak_harden_in_pwd_lock_train=True) for x in weak_run_lock_w_extra
            ]
            weak_run_lock_w_extra.append(
                LockingArgs(
                    base_weak_model_name=pwd_lock_extra_harden_exps_model_name,
                    force_weak_harden_in_pwd_lock_train=False,
                )
            )
        elif setting == "code" or setting == "critique" or setting == "math":
            ratio = len(splits.train_lock_split) / len(core_splits_default_seed["math"].train_lock_split)
            print(f"{setting=} {ratio=}")
            weak_run_lock_w_extra.append(
                LockingArgs(
                    base_weak_model_name=pwd_lock_extra_harden_exps_model_name,
                    apply_train_dupping_with_mul=round(8.0 * ratio, 2),
                    cut_low_dup=True,
                )
            )

        @attrs.frozen
        class ToAdditionalFinetune:
            base_weak_model_name: str
            prior_password_lock_suffix: str
            prior_password_lock_weights_path: str

        all_to_additional_finetune_and_rl: list[ToAdditionalFinetune] = []

        full_sweep_here = CLASSIFICATION_DEFAULT_FULL_SWEEP if has_classification else SKIP_SOME_LARGE_SWEEP_SAMP_COUNTS
        narrow_sweep_here = (
            CLASSIFICATION_NARROW_SWEEP_TO_EXTRA_LARGE_SAMP_COUNTS
            if has_classification
            else NARROW_SWEEP_TO_LARGE_SAMP_COUNTS
        )
        only_small_enough_narrow_sweep_here = (
            CLASSIFICATION_ONLY_SMALL_NARROW_SWEEP_SAMP_COUNTS
            if has_classification
            else NARROW_SWEEP_TO_LARGE_SAMP_COUNTS
        )

        def filter_sweep(sweep: list[int], add_full: bool = False):
            elicit_size = len(splits.train_elicit_split)
            assert all(x <= elicit_size for x in sweep), "we could instead filter here"
            # sweep = [x for x in sweep if x <= elicit_size]
            if add_full:
                sweep = sweep + [elicit_size]

            assert sweep == sorted(sweep)
            assert len(set(sweep)) == len(sweep)

            return sweep

        full_sweep_here = filter_sweep(full_sweep_here, add_full=has_classification)
        narrow_sweep_here = filter_sweep(narrow_sweep_here)
        only_small_enough_narrow_sweep_here = filter_sweep(only_small_enough_narrow_sweep_here)

        # core locking
        for locking_args in weak_run_lock_w_extra:
            base_weak_model_name = locking_args.base_weak_model_name
            force_weak_harden_in_pwd_lock_train = locking_args.force_weak_harden_in_pwd_lock_train
            apply_train_dupping_with_mul = locking_args.apply_train_dupping_with_mul
            cut_low_dup = locking_args.cut_low_dup

            weak_imitate_args = apply_hardening(
                baseline_model_to_imitate_by_setting[setting][base_weak_model_name],
                harden=force_weak_harden_in_pwd_lock_train,
            )

            train_data, val_data = make_pwd_lock_train_val_data_default(
                splits=splits,
                weak_imitate_args=weak_imitate_args,
                strong_imitate_args=default_pwd_lock_strong_imitate_args,
                password=DEFAULT_PASSWORD,
                apply_train_dupping_with_mul=apply_train_dupping_with_mul,
                c_targets=locking_args.c_targets,
                cut_low_dup=cut_low_dup,
            )

            if (
                base_weak_model_name == pwd_lock_extra_sweep_and_prio_model_name
                and (force_weak_harden_in_pwd_lock_train == has_classification)
                and apply_train_dupping_with_mul is None
                and (cut_low_dup is False)
            ):
                lr_muls_and_end_sched_early = [(1.0, None)]
                # lr_muls_and_end_sched_early: list[tuple[float, Optional[int]]] = [
                #     (1.0 / 2.0, None),
                #     (1.0, None),
                #     (2.0, None),
                # ]
                # if has_classification:
                #     lr_muls_and_end_sched_early.append((1.0, 6))
                use_fancy_sched_options = [True]
                also_run_T1_eval = False
                is_basic_setup = True
                base_prio = 0
            else:
                lr_muls_and_end_sched_early = [(1.0, None)]
                use_fancy_sched_options = [True]
                also_run_T1_eval = False
                is_basic_setup = False
                base_prio = 2

            pwd_lock_save_name_suffix = get_pwd_lock_name_suffix(base_strong_model_name, base_weak_model_name)

            if force_weak_harden_in_pwd_lock_train:
                pwd_lock_save_name_suffix += "_harden"
            if apply_train_dupping_with_mul is not None:
                pwd_lock_save_name_suffix += f"_dupping_{get_basic_multiplier_format(apply_train_dupping_with_mul)}"
                if cut_low_dup:
                    pwd_lock_save_name_suffix += "_cut_low_dup"

            pwd_lock_save_name = name_from_suffix(pwd_lock_save_name_suffix)

            for use_fancy_sched in use_fancy_sched_options:
                for lr_mul, end_sched_early_by_n_epochs in lr_muls_and_end_sched_early:
                    extra_prio = 0 if lr_mul == 1.0 and use_fancy_sched else 4

                    out_args, _ = run_full_password_lock_setup(
                        setting=setting,
                        start_strong_model_path=start_pwd_lock_path,
                        base_strong_model=base_strong_model_name,
                        pwd_lock_save_name_base=pwd_lock_save_name,
                        train_data=train_data,
                        val_data=val_data,
                        lr_mul=lr_mul,
                        use_fancy_sched=use_fancy_sched,
                        also_run_T1_eval=also_run_T1_eval,
                        end_sched_early_by_n_epochs=end_sched_early_by_n_epochs,
                        priority=base_prio + extra_prio + extra_priority_this_setting,
                        classification_params=classification_params,
                        n_epochs=n_epochs_pwd_lock_here,
                    )
                    all_train_out.append(out_args)

            pwd_lock_weights_path = get_weights_path(pwd_lock_save_name, n_epochs_pwd_lock_here)

            if base_weak_model_name == pwd_lock_extra_sweep_and_prio_model_name or setting == "math":
                all_to_additional_finetune_and_rl.append(
                    ToAdditionalFinetune(
                        base_weak_model_name=base_weak_model_name,
                        prior_password_lock_suffix=pwd_lock_save_name_suffix,
                        prior_password_lock_weights_path=pwd_lock_weights_path,
                    )
                )

            # run_basic_full_gold_demo_elicit(
            #     pwd_lock_weights_path=pwd_lock_weights_path,
            #     pwd_lock_save_name_suffix=pwd_lock_save_name_suffix,
            #     priority=base_prio * 2 + 2,
            # )

            if is_basic_setup:
                samp_sweep = full_sweep_here
            else:
                samp_sweep = narrow_sweep_here
                if (
                    has_classification
                    and base_weak_model_name == pwd_lock_extra_sweep_and_prio_model_name
                    and apply_train_dupping_with_mul is None
                    and (cut_low_dup is False)
                    and not force_weak_harden_in_pwd_lock_train
                ):
                    # this is the case where we don't harden, but otherwise do defaults
                    samp_sweep = CLASSIFICATION_SLIGHTLY_REDUCED_SWEEP

            # in-distrbution unlocking ft experiment
            # for samp_count in BIG_SWEEP_SAMP_COUNTS:
            # continue
            for samp_count in samp_sweep:
                if setting == "math" and apply_train_dupping_with_mul is not None and samp_count != 4:
                    # for now, skip other samp count
                    continue
                # if setting == "math":
                #     continue

                if has_classification:
                    basic_samp_counts = [16, 64, 128]
                else:
                    basic_samp_counts = [4, 16]

                if has_classification:
                    lr_sweep_counts = [64]
                else:
                    lr_sweep_counts = [4, 16]

                is_basic_elicit_setup = samp_count in basic_samp_counts and is_basic_setup
                extra_prio = 0 if is_basic_elicit_setup else 6

                # is_lr_sweep_run = samp_count in lr_sweep_counts and is_basic_setup
                is_lr_sweep_run = False

                if is_lr_sweep_run:
                    lr_muls = [1.0 / 2.0, 1.0, 2.0]
                else:
                    lr_muls = [1.0]
                    # extra_prio = 6
                    extra_prio = 0

                for lr_mul in lr_muls:
                    if lr_mul != 1.0:
                        extra_extra_prio = 8
                    else:
                        extra_extra_prio = 0

                    dup_run_extra = (
                        setting == "math"
                        and samp_count == 4
                        and base_weak_model_name == pwd_lock_extra_sweep_and_prio_model_name
                        and apply_train_dupping_with_mul is not None
                    )
                    run_basic_limited_samp_elicit(
                        samp_count=samp_count,
                        base_weak_model_name=base_weak_model_name,
                        pwd_lock_weights_path=pwd_lock_weights_path,
                        pwd_lock_save_name_suffix=pwd_lock_save_name_suffix,
                        extra_res=is_basic_elicit_setup or dup_run_extra,
                        additional_extra_mult=2.0 if dup_run_extra else 1.0,
                        extra_extra_res=is_basic_elicit_setup and lr_mul == 1.0,
                        lr_mul=lr_mul,
                        priority=1 + (0 if (samp_count in basic_samp_counts) else 2) + extra_prio + extra_extra_prio,
                        also_run_harden=has_classification,
                        allow_non_harden_if_run_harden=is_basic_setup
                        and (samp_count in basic_samp_counts or samp_count in narrow_sweep_here)
                        and has_classification
                        and lr_mul == 1.0,
                    )

        num_runs_at_core = len(all_train_out)
        num_runs_for_core = num_runs_at_core - num_runs_at_start_setting
        if print_run_counts:
            print(f"{num_runs_for_core=} {setting=}")
        all_to_rl = list(all_to_additional_finetune_and_rl)

        for base_weak_model_name in weak_models_run_lock[setting]:
            prior_password_lock_suffix = get_pwd_lock_name_suffix(base_strong_model_name, base_weak_model_name)
            to_add = ToAdditionalFinetune(
                base_weak_model_name=base_weak_model_name,
                prior_password_lock_suffix=prior_password_lock_suffix,
                prior_password_lock_weights_path=get_weights_path(
                    name_from_suffix(prior_password_lock_suffix), n_epochs_pwd_lock_here
                ),
            )

            if to_add not in all_to_rl:
                all_to_rl.append(to_add)

            if base_weak_model_name != focus_on_weak_model:
                continue

            # add intermediate train to rl
            base_weak_model_name = to_add.base_weak_model_name
            pwd_lock_save_name_suffix = to_add.prior_password_lock_suffix
            pwd_lock_weights_path = to_add.prior_password_lock_weights_path
            samp_count = 16
            samp_seed = 0
            for save_name, intermediate_weak_model, epoch_mul in intermediate_weak_models_by_setting.get(setting, []):
                if base_weak_model_name in skip_train_ones_by_setting[setting]:
                    baseline_weights_path = remap_names[MODEL_INFOS[intermediate_weak_model].full_name]
                else:
                    baseline_save_name_suffix = get_baseline_name_suffix(intermediate_weak_model)
                    baseline_weights_path = get_weights_path(
                        name_from_suffix(baseline_save_name_suffix), baseline_n_epochs_here
                    )

                different_save_name_for_inference = baseline_model_to_imitate_by_setting[setting][
                    intermediate_weak_model
                ].save_name_for_inference

                train_data = get_limited_samp_elicit(
                    splits=splits,
                    samp_count=samp_count,
                    seed=samp_seed,
                    from_source=ModelToImitateArgs(baseline_weights_path, different_save_name_for_inference),
                )
                n_epochs_this_unlock, eval_every_this_unlock = get_elicit_n_epochs_and_eval_epochs(
                    samp_count=samp_count,
                    config=ElicitNEpochsAndEvalEpochsConfig(extra_mult=epoch_mul),
                )
                elicit_save_name_suffix = get_samp_elicit_name_suffix(
                    pwd_lock_name_suffix=pwd_lock_save_name_suffix,
                    elicit_type=save_name,
                    samp_count=samp_count,
                    seed=samp_seed,
                )
                elicit_save_name = name_from_suffix(elicit_save_name_suffix)
                all_train_out.append(
                    get_basic_elicit_train_args(
                        pwd_lock_weights_path=pwd_lock_weights_path,
                        train_data=train_data,
                        elicit_save_name=elicit_save_name,
                        n_epochs_this_unlock=n_epochs_this_unlock,
                        save_weights_every=n_epochs_this_unlock,
                        eval_every_this_unlock=eval_every_this_unlock,
                        priority=2,
                    )
                )

                all_to_rl.append(
                    ToAdditionalFinetune(
                        base_weak_model_name=base_weak_model_name,
                        prior_password_lock_suffix=elicit_save_name_suffix,
                        prior_password_lock_weights_path=get_weights_path(elicit_save_name, n_epochs_this_unlock),
                    )
                )

        # expert iteration
        for to_rl in all_to_rl:
            # if has_classification or True:
            #     continue
            if has_classification:
                continue

            base_weak_model_name = to_rl.base_weak_model_name
            pwd_lock_save_name_suffix = to_rl.prior_password_lock_suffix
            pwd_lock_weights_path = to_rl.prior_password_lock_weights_path

            if base_weak_model_name in skip_train_ones_by_setting[setting]:
                baseline_weights_path = remap_names[MODEL_INFOS[base_weak_model_name].full_name]
            else:
                baseline_save_name_suffix = get_baseline_name_suffix(base_weak_model_name)
                baseline_weights_path = get_weights_path(
                    name_from_suffix(baseline_save_name_suffix), baseline_n_epochs_here
                )

            add_ei(
                splits,
                base_strong_model_name,
                pwd_lock_save_name_suffix,
                pwd_lock_weights_path,
                baseline_weights_path,
                setting,
                default_elicit_val_data,
                score_fn_by_setting[setting],
                all_train_out,
            )

        num_runs_at_rl = len(all_train_out)
        num_runs_for_rl = num_runs_at_rl - num_runs_at_core
        if print_run_counts:
            print(f"{num_runs_for_rl=} {setting=}")

        # dpo and iterated dpo

        # dpo_lr_muls = [0.5, 1.0, 2.0, 4.0] if setting == "math" else [1.0]
        dpo_lr_muls = [1.0, 2.0, 4.0]
        # betas = [0.05, 0.1, 0.2] if setting == "math" else [0.1]
        betas = [0.1]
        # kinds = ["regular", "iterated", "weak"]
        retain_prob = 0.3 if setting == "math" else 1.0  # because math is 3x bigger and DPO is slow...

        for to_rl in all_to_rl:
            if (
                has_classification
                # or ("_cut_low_dup" in to_rl.prior_password_lock_suffix)
                # or ("saved_intermediate" in to_rl.prior_password_lock_suffix)
            ):
                continue
            # if has_classification or "_cut_low_dup" in to_rl.prior_password_lock_suffix:
            #     continue

            base_weak_model_name = to_rl.base_weak_model_name
            pwd_lock_save_name_suffix = to_rl.prior_password_lock_suffix
            pwd_lock_weights_path = to_rl.prior_password_lock_weights_path

            no_hope = (
                (setting == "math" and base_weak_model_name not in [STABLE_ZEPH2B, PYTHIA7B])
                or base_weak_model_name == UNIFORMPP
                or base_weak_model_name == FIXEDRDM
            )
            kinds = ["iterated"] if no_hope else ["regular", "iterated", "weak"]

            pwd_lock_save_name = name_from_suffix(pwd_lock_save_name_suffix)
            if base_weak_model_name in skip_train_ones_by_setting[setting]:
                baseline_weights_path = remap_names[MODEL_INFOS[base_weak_model_name].full_name]
            else:
                baseline_save_name_suffix = get_baseline_name_suffix(base_weak_model_name)
                baseline_weights_path = get_weights_path(
                    name_from_suffix(baseline_save_name_suffix), baseline_n_epochs_here
                )
            for lr_mul in dpo_lr_muls:
                for score_fn in score_fn_by_setting[setting]:
                    for kind in kinds:
                        # for beta in [0.05, 0.1, 0.2]:
                        for beta in betas:
                            for seed in DPO_SEEDS:
                                train_data, train_ds_preprocess = get_dpo(
                                    splits,
                                    pwd_lock_weights_path=(
                                        pwd_lock_weights_path if kind != "weak" else baseline_weights_path
                                    ),
                                    score_fn=score_fn,
                                    is_it=kind == "iterated",
                                    retain_prob=retain_prob,
                                )
                                dpo_save_name = (
                                    name_from_suffix(
                                        get_dpo_name_suffix(
                                            pwd_lock_name_suffix=pwd_lock_save_name_suffix,
                                            seed=seed,
                                            score_fn=score_fn,
                                            kind=kind[0],
                                            beta=beta,
                                        )
                                    )
                                    + (
                                        f"_lrm_{round(lr_mul)}" + "_" + f"{lr_mul-round(lr_mul):.2f}"[2:]
                                        if lr_mul != 1.0
                                        else ""
                                    )
                                    + (f"_ret_{retain_prob}" if retain_prob != 1.0 else "")
                                )
                                s_typing_fix: SettingKind = setting
                                lr = (
                                    MODEL_INFOS[base_strong_model_name].size.standard_lr_by_setting[s_typing_fix]
                                    * lr_mul
                                    * DPO_LR_RED_FACTOR
                                )
                                all_train_out.append(
                                    TrainArgs(
                                        pwd_lock_weights_path,
                                        base_strong_model_name,
                                        train_df=train_data,
                                        val_df=default_elicit_val_data,
                                        save_model_name=dpo_save_name,
                                        lr=lr,
                                        shared_train_config=SharedTrainConfig(
                                            n_epochs=DPO_N_EPOCHS,
                                            save_weights_every=False,
                                            eval_every=DPO_EVAL_EVERY,
                                            eval_0=False,
                                            skip_eval_losses=True,
                                            train_ds_preprocess=train_ds_preprocess,
                                            seed=seed,
                                            dpo_params={
                                                "beta": beta,
                                                "weight_by_delta": True,
                                                "exclude_close_threshold": 0,
                                            },
                                        ),
                                        priority=(
                                            3
                                            # + (lr_mul != 1.0)
                                            + (kind != "iterated")
                                            + 2 * (beta != 0.1)
                                            + extra_priority_this_setting
                                            + DPO_SEEDS.index(seed)
                                            + (
                                                12
                                                * (
                                                    "cut_low" in pwd_lock_save_name_suffix
                                                    or "saved" in pwd_lock_save_name_suffix
                                                )
                                            )
                                        ),
                                    )
                                )

        num_runs_at_dpo = len(all_train_out)
        num_runs_for_dpo = num_runs_at_dpo - num_runs_at_rl
        if print_run_counts:
            print(f"{num_runs_for_dpo=} {setting=}")

        @attrs.frozen
        class LimitedSampElicit:
            count: int
            seed: int

        @attrs.frozen
        class ConstantEvaluateNewDataElicit:
            seed: int

        def get_fixed_model_to_imitate_args(save_name_for_inference: str, harden: bool = False):
            if not has_classification:
                return ModelToImitateArgs(
                    save_name_for_inference=save_name_for_inference,
                )
            else:
                assert classification_params_extra is not None
                return make_classification_model_to_imitate(
                    to_imitate=ClassificationUseSavedProbs(
                        save_name_for_inference=save_name_for_inference, harden=harden
                    ),
                )

        def get_constant_evaluate_items(
            model_to_imitate: EitherModelToImitateArgs,
            elicit_data_args: ConstantEvaluateNewDataElicit,
            elicit_type: str,
            pwd_lock_save_name_suffix: str,
        ):
            # full train data, but evaluate frequently!
            train_data = fill_pbs_df(
                splits.train_elicit_split,
                to_imitate_args=TrainToImitateArgs(
                    model_args=model_to_imitate,
                    has_strong_perf_pwd=False,
                    password=NoPwd(),
                ),
                use_fixed=True,
            )
            train_data = PbAndGens.from_df(
                train_data.sample(frac=1.0, random_state=elicit_data_args.seed).reset_index(drop=True)
            )
            if has_classification:
                eval_every_this_unlock = [1, 2, 3, 4, 6, 8, 12, 16, 24, 32]
                n_epochs_this_unlock = 64
                per_epoch_count = 256  # very different from below
            else:
                eval_every_this_unlock = [1, 2, 3, 4, 5, 6, 7, 8, 16, 24, 32, 48, 64]
                n_epochs_this_unlock = 64
                per_epoch_count = 32

            elicit_save_name_suffix = get_constant_evaluate_elict_name_suffix(
                pwd_lock_name_suffix=pwd_lock_save_name_suffix,
                elicit_type=elicit_type,
                seed=elicit_data_args.seed,
            )

            return train_data, n_epochs_this_unlock, eval_every_this_unlock, per_epoch_count, elicit_save_name_suffix

        def run_samp_unlock_with_different(
            base_weak_model_name: str,
            different_elicit: DifferentElicit,
            elicit_data_args: Union[LimitedSampElicit, ConstantEvaluateNewDataElicit],
            pwd_lock_save_name_suffix: str,
            pwd_lock_weights_path: str,
            priority: int,
            allow_extra_res: bool,
            harden: bool = False,
        ) -> None:
            if isinstance(different_elicit, DifferentElicitFromModel):
                different_save_name_for_inference = op.unwrap(
                    baseline_model_to_imitate_by_setting[setting][
                        different_elicit.base_model_name
                    ].save_name_for_inference
                )
            elif isinstance(different_elicit, DifferentElicitFixedSaveName):
                different_save_name_for_inference = different_elicit.save_name_for_inference
            else:
                assert_never(different_elicit)

            eval_every_this_unlock: int | list[int]
            elicit_type = f"elicit_different_{different_elicit.name}"

            if isinstance(elicit_data_args, LimitedSampElicit):
                samp_seed = elicit_data_args.seed
                train_data = get_limited_samp_elicit(
                    splits=splits,
                    samp_count=elicit_data_args.count,
                    seed=samp_seed,
                    from_source=get_fixed_model_to_imitate_args(different_save_name_for_inference, harden=harden),
                )
                n_epochs_this_unlock, eval_every_this_unlock = get_elicit_n_epochs_and_eval_epochs(
                    samp_count=elicit_data_args.count,
                    config=ElicitNEpochsAndEvalEpochsConfig(
                        extra_res=elicit_data_args.count in narrow_sweep_here  # could be less extreme tbh
                        and allow_extra_res,
                        extra_mult=DEFAULT_GT_EXTRA_MULT,  # use gt for now, a bit unclear
                    ),
                )
                per_epoch_count = None

                if isinstance(different_elicit, DifferentElicitFromModel):
                    assert (
                        different_elicit.base_model_name != base_weak_model_name
                    ), "this is just a normal baseline, should be run separately!"
                    # assert (
                    #     different_elicit.base_model_name != base_strong_model_name
                    # ), "this is just a normal baseline, should be run separately!"
                if isinstance(different_elicit, DifferentElicitFixedSaveName):
                    assert (
                        different_elicit.save_name_for_inference != gold_demo_imitate_args.save_name_for_inference
                    ), "this is just a normal baseline, should be run separately!"

                elicit_save_name_suffix = get_samp_elicit_name_suffix(
                    pwd_lock_name_suffix=pwd_lock_save_name_suffix,
                    elicit_type=elicit_type,
                    samp_count=samp_count,
                    seed=samp_seed,
                )

            elif isinstance(elicit_data_args, ConstantEvaluateNewDataElicit):
                (
                    train_data,
                    n_epochs_this_unlock,
                    eval_every_this_unlock,
                    per_epoch_count,
                    elicit_save_name_suffix,
                ) = get_constant_evaluate_items(
                    get_fixed_model_to_imitate_args(different_save_name_for_inference, harden=harden),
                    elicit_data_args,
                    elicit_type,
                    pwd_lock_save_name_suffix,
                )
            else:
                assert_never(elicit_data_args)

            elicit_save_name = name_from_suffix(elicit_save_name_suffix)
            if has_classification and (not harden):
                elicit_save_name += "_non_harden"

            all_train_out.append(
                get_basic_elicit_train_args(
                    pwd_lock_weights_path=pwd_lock_weights_path,
                    train_data=train_data,
                    elicit_save_name=elicit_save_name,
                    n_epochs_this_unlock=n_epochs_this_unlock,
                    eval_every_this_unlock=eval_every_this_unlock,
                    priority=priority,
                    per_epoch_count=per_epoch_count,
                )
            )

        def run_const_eval_baseline(
            base_weak_model_name: str,
            is_gt: bool,
            elicit_data_args: ConstantEvaluateNewDataElicit,
            pwd_lock_save_name_suffix: str,
            pwd_lock_weights_path: str,
            priority: int,
            harden: bool = False,
        ) -> None:
            (
                train_data,
                n_epochs_this_unlock,
                eval_every_this_unlock,
                per_epoch_count,
                elicit_save_name_suffix,
            ) = get_constant_evaluate_items(
                (
                    gold_demo_imitate_args
                    if is_gt
                    else apply_hardening(
                        baseline_model_to_imitate_by_setting[setting][base_weak_model_name], harden=harden
                    )
                ),
                elicit_data_args,
                elicit_type="gt" if is_gt else "same_weak",
                pwd_lock_save_name_suffix=pwd_lock_save_name_suffix,
            )

            elicit_save_name = name_from_suffix(elicit_save_name_suffix)
            if has_classification and (not harden) and (not is_gt):
                elicit_save_name += "_non_harden"

            all_train_out.append(
                get_basic_elicit_train_args(
                    pwd_lock_weights_path=pwd_lock_weights_path,
                    train_data=train_data,
                    elicit_save_name=elicit_save_name,
                    n_epochs_this_unlock=n_epochs_this_unlock,
                    eval_every_this_unlock=eval_every_this_unlock,
                    priority=priority,
                    per_epoch_count=per_epoch_count,
                )
            )

        if setting == "mmlu" or setting == "math":
            for to_additional in all_to_additional_finetune_and_rl:
                base_weak_model_name = to_additional.base_weak_model_name
                prior_password_lock_suffix = to_additional.prior_password_lock_suffix
                prior_password_lock_weights_path = to_additional.prior_password_lock_weights_path

                # additional finetuning and then more basic elicitation!
                lr_mul = 1.0  # probably doesn't matter much, but should mabye change

                weak_imitate_args = baseline_model_to_imitate_by_setting[setting][base_weak_model_name]
                if has_classification:
                    assert isinstance(weak_imitate_args, ClassificationModelToImitateArgs)
                    assert isinstance(weak_imitate_args.to_imitate, ClassificationUseSavedProbs)
                    assert not weak_imitate_args.to_imitate.harden
                    assert not weak_imitate_args.to_imitate.harden_on_second_most_likely
                _, val_data = make_pwd_lock_train_val_data_default(
                    splits=splits,
                    weak_imitate_args=weak_imitate_args,
                    strong_imitate_args=default_pwd_lock_strong_imitate_args,
                    password=DEFAULT_PASSWORD,
                )
                weak_train_to_imitate, _ = make_weak_strong_to_imitate_args(
                    weak_imitate_args=weak_imitate_args,
                    strong_imitate_args=default_pwd_lock_strong_imitate_args,
                    password=DEFAULT_PASSWORD,
                )
                assert weak_train_to_imitate.has_strong_perf_pwd is False

                elicit_split_shuf_here = splits.train_elicit_split.sample(frac=1.0, random_state=28374).reset_index(
                    drop=True
                )
                count_reserve_for_actual_elicit = 512

                elicit_split_shuf_here_for_train = elicit_split_shuf_here.iloc[:-count_reserve_for_actual_elicit]
                elicit_split_shuf_here_for_actual_elicit = elicit_split_shuf_here.iloc[
                    -count_reserve_for_actual_elicit:
                ]
                assert len(elicit_split_shuf_here_for_actual_elicit) == count_reserve_for_actual_elicit

                train_data = fill_pbs_df(
                    df=elicit_split_shuf_here_for_train,
                    to_imitate_args=weak_train_to_imitate,
                    use_fixed=True,
                )

                after_finetune_save_name_suffix = prior_password_lock_suffix + "_additional_finetune"
                after_finetune_save_name_here = name_from_suffix(after_finetune_save_name_suffix)

                n_epochs_to_try_additional_finetune = [1, 2, 4, 8] if setting == "mmlu" else [1, 2]

                lr = MODEL_INFOS[base_strong_model_name].size.standard_lr_by_setting[setting] * lr_mul

                out_args = TrainArgs(
                    prior_password_lock_weights_path,
                    base_strong_model_name,
                    train_df=train_data,
                    val_df=val_data,
                    save_model_name=after_finetune_save_name_here,
                    lr=lr,
                    shared_train_config=SharedTrainConfig(
                        save_weights_every=n_epochs_to_try_additional_finetune,
                        n_epochs=max(n_epochs_to_try_additional_finetune),
                        eval_every=n_epochs_to_try_additional_finetune,
                        eval_0=False,
                        also_run_T1_eval=False,
                        classification_params=classification_params,
                    ),
                    priority=0,
                )
                all_train_out.append(out_args)

                for epoch_try_idx, n_epochs_this_try in enumerate(n_epochs_to_try_additional_finetune):
                    after_finetune_weights_path_here = get_weights_path(
                        after_finetune_save_name_here, n_epochs_this_try
                    )

                    if epoch_try_idx != 0:
                        after_finetune_save_name_suffix_here = (
                            after_finetune_save_name_suffix + f"_epoch_{n_epochs_this_try}"
                        )
                    else:
                        after_finetune_save_name_suffix_here = after_finetune_save_name_suffix

                    for samp_count in only_small_enough_narrow_sweep_here:
                        prio = (
                            2
                            + (0 if epoch_try_idx == 0 else (4 + epoch_try_idx))
                            + (0 if (samp_count == (16 if has_classification else 4)) else 8)
                        )
                        samp_seeds_here = SAMP_SEEDS

                        all_force_elicit_data = [
                            elicit_split_shuf_here_for_actual_elicit.iloc[i * samp_count : (i + 1) * samp_count]
                            for i in range(len(samp_seeds_here))
                        ]
                        assert all(len(x) == samp_count for x in all_force_elicit_data)

                        run_basic_limited_samp_elicit(
                            samp_count=samp_count,
                            base_weak_model_name=base_weak_model_name,
                            pwd_lock_weights_path=after_finetune_weights_path_here,
                            pwd_lock_save_name_suffix=after_finetune_save_name_suffix_here,
                            priority=prio,
                            all_force_elicit_data=all_force_elicit_data,
                            samp_seeds=samp_seeds_here,
                            # gt_extra_prio=10,
                        )

        # if True:
        #     continue

        additional_password = BasicPwd.special(1)
        assert additional_password != DEFAULT_PASSWORD

        # START unlocking Lock(weak, strong) with slightly stronger weak / with gpt-4
        restricted_elicit_pairs_run = restricted_different_elicit_pairs_by_setting[setting]

        def different_elicit_pre_to_process(x: DifferentElicitPre) -> DifferentElicit:
            if x == "elicit_from_model_self_strong":
                return DifferentElicitFromModel(base_strong_model_name)
            return x

        already_seen_base_weaks = set()

        default_samp_counts_rest_exps = (
            CLASSIFICATION_ONLY_SOMEWHAT_SMALL_NARROW_SWEEP_SAMP_COUNTS
            if has_classification
            else NARROW_SWEEP_TO_EXTRA_LARGE_SAMP_COUNTS
        )

        for base_weak_model_name, different_elicit_pre in restricted_elicit_pairs_run:
            different_elicit = different_elicit_pre_to_process(different_elicit_pre)
            novel_base = base_weak_model_name not in already_seen_base_weaks
            already_seen_base_weaks.add(base_weak_model_name)

            is_elicit_from_self_strong = "elicit_from_model_self_strong" == different_elicit_pre

            baseline_prio = 4 if is_elicit_from_self_strong else 9  # prioritize these runs somewhat

            for train_to_imitate_different in [False, True]:
                if (
                    different_elicit_pre == "elicit_from_model_self_strong"
                    and train_to_imitate_different
                    and (not has_classification)
                ):
                    # No need to do this one
                    # note: if we are doing classification, then we want to run this (and not continue) because the default for training pwd lock is gt labels (not self strong)
                    continue

                harden_on_train_to_imitate = has_classification

                # run_both_harden_and_non_harden = (
                #     has_classification and different_elicit_pre == "elicit_from_model_self_strong"
                # )  # on self strong, we'll try both
                run_both_harden_and_non_harden = (
                    has_classification  # always run both for now, maybe interesting... Somewhat overkill...
                )

                if train_to_imitate_different:
                    baseline_prio = baseline_prio + 10

                    # Train an alternative version which also trains on the alternative behind a different password!!!
                    if isinstance(different_elicit, DifferentElicitFromModel):
                        model_args = apply_hardening(
                            baseline_model_to_imitate_by_setting[setting][different_elicit.base_model_name],
                            harden=harden_on_train_to_imitate,
                        )
                        to_imitate_args_additional_for_different = TrainToImitateArgs.additional(
                            model_args=model_args,
                            password=additional_password,
                        )
                    elif isinstance(different_elicit, DifferentElicitFixedSaveName):
                        model_args = apply_hardening(
                            get_fixed_model_to_imitate_args(
                                save_name_for_inference=different_elicit.save_name_for_inference,
                            ),
                            harden=harden_on_train_to_imitate,
                        )
                        to_imitate_args_additional_for_different = TrainToImitateArgs.additional(
                            model_args=model_args,
                            password=additional_password,
                            retain_probability=0.25,
                            force_fixed=True,
                        )
                    else:
                        assert_never(different_elicit)

                    train_data, val_data = make_pwd_lock_train_val_data_default(
                        splits=splits,
                        weak_imitate_args=baseline_model_to_imitate_by_setting[setting][base_weak_model_name],
                        strong_imitate_args=default_pwd_lock_strong_imitate_args,
                        password=DEFAULT_PASSWORD,
                        additional_imitate_args=[to_imitate_args_additional_for_different],
                    )

                    pwd_lock_save_name_suffix = get_pwd_lock_name_suffix(
                        base_strong_model_name,
                        base_weak_model_name,
                    )
                    pwd_lock_save_name_suffix += f"_additional_imitate_different_{different_elicit.name}"
                    if harden_on_train_to_imitate:
                        pwd_lock_save_name_suffix += "_harden"
                    pwd_lock_save_name = name_from_suffix(pwd_lock_save_name_suffix)
                    pwd_lock_weights_path = get_weights_path(pwd_lock_save_name, n_epochs_pwd_lock_here)

                    out_args, _ = run_full_password_lock_setup(
                        setting=setting,
                        start_strong_model_path=start_pwd_lock_path,
                        base_strong_model=base_strong_model_name,
                        pwd_lock_save_name_base=pwd_lock_save_name,
                        train_data=train_data,
                        val_data=val_data,
                        priority=baseline_prio + extra_priority_this_setting,
                        classification_params=classification_params,
                        n_epochs=n_epochs_pwd_lock_here,
                    )
                    all_train_out.append(out_args)

                    # run a baseline for comparing to this below
                    run_baseline_samp_elicits(
                        base_weak_model_name=base_weak_model_name,
                        pwd_lock_weights_path=pwd_lock_weights_path,
                        pwd_lock_save_name_suffix=pwd_lock_save_name_suffix,
                        samp_counts=default_samp_counts_rest_exps,
                        base_priority=baseline_prio * 2,
                        allow_non_harden_if_run_harden=run_both_harden_and_non_harden,
                    )

                else:
                    pwd_lock_save_name_suffix = get_pwd_lock_name_suffix(base_strong_model_name, base_weak_model_name)
                    use_harden_default = has_classification
                    if use_harden_default:
                        pwd_lock_save_name_suffix += "_harden"
                    pwd_lock_save_name = name_from_suffix(pwd_lock_save_name_suffix)
                    pwd_lock_weights_path = get_weights_path(pwd_lock_save_name, n_epochs_pwd_lock_here)

                run_samp_unlock_with_different_here = partial(
                    run_samp_unlock_with_different,
                    base_weak_model_name=base_weak_model_name,
                    pwd_lock_save_name_suffix=pwd_lock_save_name_suffix,
                    pwd_lock_weights_path=pwd_lock_weights_path,
                    allow_extra_res=is_elicit_from_self_strong,
                )

                if has_classification:
                    harden_opts = [False, True] if run_both_harden_and_non_harden else [True]
                else:
                    harden_opts = [False]

                for samp_count in default_samp_counts_rest_exps:
                    for samp_seed in SAMP_SEEDS:
                        for harden in harden_opts:
                            run_samp_unlock_with_different_here(
                                different_elicit=different_elicit,
                                elicit_data_args=LimitedSampElicit(count=samp_count, seed=samp_seed),
                                priority=baseline_prio * 2 + 1,
                                harden=harden,
                            )

                for shuf_seed in CONST_EVAL_SEEDS:
                    # here, we run with the full train data, but evaluate frequently
                    for harden in harden_opts:
                        run_samp_unlock_with_different_here(
                            different_elicit=different_elicit,
                            elicit_data_args=ConstantEvaluateNewDataElicit(seed=shuf_seed),
                            priority=baseline_prio * 2 + 3 + (2 * baseline_prio * CONST_EVAL_SEEDS.index(shuf_seed)),
                            harden=harden,
                        )

                if novel_base or train_to_imitate_different:
                    for is_gt in [False, True]:
                        for harden in [False] if is_gt else harden_opts:
                            for shuf_seed in CONST_EVAL_SEEDS:
                                # here, we run with the full train data, but evaluate frequently
                                run_const_eval_baseline(
                                    base_weak_model_name=base_weak_model_name,
                                    is_gt=is_gt,
                                    elicit_data_args=ConstantEvaluateNewDataElicit(seed=shuf_seed),
                                    pwd_lock_save_name_suffix=pwd_lock_save_name_suffix,
                                    pwd_lock_weights_path=pwd_lock_weights_path,
                                    priority=baseline_prio * 2
                                    + 3
                                    + (4 * baseline_prio * CONST_EVAL_SEEDS.index(shuf_seed)),
                                    harden=harden,
                                )

        # END unlocking Lock(weak, strong) with slightly stronger weak / with gpt-4

        num_runs_at_unlock_slightly_stronger_weak = len(all_train_out)
        num_runs_for_unlock_slightly_stronger_weak = num_runs_at_unlock_slightly_stronger_weak - num_runs_at_rl
        if print_run_counts:
            print(f"{num_runs_for_unlock_slightly_stronger_weak=} {setting=}")

        def map_df_for_label_domain(data: PbAndGens, which_domain_type: WhichDomainType, label_domain: bool):
            if not label_domain:
                return data
            s_typing_fix: SettingKind = setting
            s = all_settings()[s_typing_fix]
            return s.label_domain(data, which_domain_type=which_domain_type)

        def run_all_on_only_grouping_items(
            base_weak_model_name: str,
            grouping_name: str,
            grouping_items: list[str],
            which_domain_type: WhichDomainType,
            pwd_lock_save_name_suffix: str,
            pwd_lock_weights_path: str,
            elicit_val_data_here: PbAndGens = default_elicit_val_data,
            label_domain: bool = False,
            also_run_harden: bool = has_classification,
            allow_non_harden_if_run_harden: bool = False,
            samp_counts: list[int] = default_samp_counts_rest_exps,
            just_gt: bool = False,
        ) -> None:
            gt_harden_opts = get_gt_harden_opts(
                also_run_harden=also_run_harden,
                allow_non_harden_if_run_harden=allow_non_harden_if_run_harden,
                just_gt=just_gt,
            )

            for samp_count in samp_counts:
                for is_gt, harden in gt_harden_opts:
                    for samp_seed in SAMP_SEEDS:
                        this_imitate = (
                            gold_demo_imitate_args
                            if is_gt
                            else apply_hardening(
                                baseline_model_to_imitate_by_setting[setting][base_weak_model_name], harden=harden
                            )
                        )

                        train_data = get_limited_samp_elicit(
                            splits=splits,
                            samp_count=samp_count,
                            seed=samp_seed,
                            from_source=this_imitate,
                            filt=lambda df: df[which_domain_type].isin(grouping_items),
                        )
                        assert train_data[which_domain_type].isin(grouping_items).all()
                        train_data = map_df_for_label_domain(
                            train_data,
                            which_domain_type=which_domain_type,
                            label_domain=label_domain,
                        )

                        (n_epochs_this_unlock, eval_every_this_unlock,) = get_elicit_n_epochs_and_eval_epochs(
                            samp_count=samp_count,
                            config=ElicitNEpochsAndEvalEpochsConfig(
                                extra_mult=DEFAULT_GT_EXTRA_MULT if is_gt else DEFAULT_SAME_WEAK_EXTRA_MULT
                            ),
                        )

                        s_typing_fix: SettingKind = setting
                        lr = MODEL_INFOS[base_strong_model_name].size.standard_lr_by_setting[s_typing_fix]

                        elicit_save_name_suffix = get_samp_elicit_name_suffix(
                            pwd_lock_name_suffix=pwd_lock_save_name_suffix,
                            elicit_type=("gt" if is_gt else "same_weak") + "_on_only_" + grouping_name,
                            samp_count=samp_count,
                            seed=samp_seed,
                        )
                        elicit_save_name = name_from_suffix(elicit_save_name_suffix)
                        if has_classification and (not harden) and (not is_gt):
                            elicit_save_name += "_non_harden"

                        all_train_out.append(
                            TrainArgs(
                                pwd_lock_weights_path,
                                base_strong_model_name,
                                train_df=train_data,
                                val_df=elicit_val_data_here,
                                save_model_name=elicit_save_name,
                                lr=lr,
                                shared_train_config=SharedTrainConfig(
                                    n_epochs=n_epochs_this_unlock,
                                    save_weights_every=False,
                                    eval_every=eval_every_this_unlock,
                                    eval_0=True,  # extra cost which is silly, but somewhat convenient because val differs from pwd locking
                                    skip_eval_losses=True,
                                    classification_params=classification_params,
                                ),
                                priority=25 + extra_priority_this_setting,
                            )
                        )

        # Only unlock on one domain (force blue team to generalize, but red team isn't allowed to know which domain)
        for groupings_by_setting, which_domain_type in [
            (category_groupings_by_setting, "category"),
            (difficulty_groupings_by_setting, "difficulty"),
        ]:
            if setting not in groupings_by_setting:
                continue

            groupings = groupings_by_setting[setting]
            groupings_lst = list(groupings.values())

            assert set(sum(groupings.values(), start=[])) == set(splits.combined_data[which_domain_type].unique())

            inv_groupings = {x: k for k, v in groupings.items() for x in v}
            cat_counts = splits.combined_data[which_domain_type].value_counts()

            sum_by_group = defaultdict(int)
            for name, x in zip(cat_counts.index, cat_counts):
                sum_by_group[inv_groupings[name]] += x

            print(f"For setting {setting} and domain type {which_domain_type}, we have counts:")
            for group_name, count in sum_by_group.items():
                print(f"{group_name}: {count}")

            for label_domain in [False, True] if which_domain_type == "category" else [False]:
                for pwd_args in [
                    None,
                    DiffPwdByDomainArgs(which_domain_type=which_domain_type, sample_neg=True),
                    DiffPwdByDomainArgs(
                        which_domain_type=which_domain_type, sample_neg=True, sample_neg_overlapping=True
                    ),
                ]:
                    if label_domain is True and pwd_args is None:
                        # don't do any label domain experiments if we're not also pwd locking by domain
                        continue

                    password = DEFAULT_PASSWORD
                    no_password = NoPwd()

                    if pwd_args is not None:
                        password = pwd_args.strong_pwd(groupings=groupings_lst)
                        no_password = pwd_args.no_pwd(groupings=groupings_lst)

                    map_df = partial(
                        map_df_for_label_domain, which_domain_type=which_domain_type, label_domain=label_domain
                    )

                    elicit_val_data_here = map_df(
                        elicit_val_df(
                            splits.val_split,
                            password=password,
                            classification_params_extra=classification_params_extra,
                            # no_password should be the default, NoPwd()
                        )
                    )

                    for base_weak_model_name in weak_models_run_lock_elicit_on_subset_of_domains_by_setting[setting]:
                        harden = has_classification  # default to harden

                        pwd_lock_save_name_suffix = get_pwd_lock_name_suffix(
                            base_strong_model_name,
                            base_weak_model_name,
                            pwd_name=op.map(pwd_args, lambda a: a.pwd_name()),
                        )
                        if label_domain:
                            pwd_lock_save_name_suffix += "_label_domain"
                        if harden:
                            pwd_lock_save_name_suffix += "_harden"
                        pwd_lock_save_name = name_from_suffix(pwd_lock_save_name_suffix)
                        pwd_lock_weights_path = get_weights_path(pwd_lock_save_name, n_epochs_pwd_lock_here)

                        if pwd_args is not None:
                            train_data, val_data = make_pwd_lock_train_val_data_default(
                                splits=splits,
                                weak_imitate_args=apply_hardening(
                                    baseline_model_to_imitate_by_setting[setting][base_weak_model_name], harden=harden
                                ),
                                strong_imitate_args=default_pwd_lock_strong_imitate_args,
                                password=password,
                                no_password=no_password,
                            )
                            train_data = map_df(train_data)
                            val_data = map_df(val_data)

                            out_args, _ = run_full_password_lock_setup(
                                setting=setting,
                                start_strong_model_path=start_pwd_lock_path,
                                base_strong_model=base_strong_model_name,
                                pwd_lock_save_name_base=pwd_lock_save_name,
                                train_data=train_data,
                                val_data=val_data,
                                priority=12 + extra_priority_this_setting,
                                classification_params=classification_params,
                                n_epochs=n_epochs_pwd_lock_here,
                            )
                            all_train_out.append(out_args)

                            # run a baseline for comparing to this
                            run_baseline_samp_elicits(
                                base_weak_model_name=base_weak_model_name,
                                pwd_lock_weights_path=pwd_lock_weights_path,
                                pwd_lock_save_name_suffix=pwd_lock_save_name_suffix,
                                samp_counts=default_samp_counts_rest_exps,
                                elicit_val_data=elicit_val_data_here,
                                base_priority=18,
                                map_df=map_df,
                            )

                        for group_idx, (grouping_name, grouping_items) in enumerate(
                            groupings.items()
                        ):  # assume fixed ordering which should be true
                            run_all_on_only_grouping_items(
                                base_weak_model_name=base_weak_model_name,
                                grouping_name=grouping_name,
                                grouping_items=grouping_items,
                                which_domain_type=which_domain_type,
                                pwd_lock_save_name_suffix=pwd_lock_save_name_suffix,
                                pwd_lock_weights_path=pwd_lock_weights_path,
                                elicit_val_data_here=elicit_val_data_here,
                                label_domain=label_domain,
                                just_gt=group_idx != 0,
                            )

        num_runs_at_blue_team_generalize = len(all_train_out)
        num_runs_for_blue_team_generalize = num_runs_at_blue_team_generalize - num_runs_at_unlock_slightly_stronger_weak
        if print_run_counts:
            print(f"{num_runs_for_blue_team_generalize=} {setting=}")

        # Ban some data for red team (force red team model to generalize to held out domain)
        for groupings_by_setting, which_domain_type in [
            (category_groupings_by_setting, "category"),
            (difficulty_groupings_by_setting, "difficulty"),
        ]:
            if setting not in groupings_by_setting:
                continue

            groupings = groupings_by_setting[setting]

            assert set(sum(groupings.values(), start=[])) == set(splits.combined_data[which_domain_type].unique())

            for group_idx, (grouping_name, grouping_items) in enumerate(groupings.items()):

                def filt_out_train_on_weak_data(df: PbAndGens):
                    has_strong = np.array([x is True for x in df.has_strong_perf_pwd])
                    to_remove = df[which_domain_type].isin(grouping_items) & (~has_strong)
                    return df[~to_remove]

                for base_weak_model_name in weak_models_run_lock_red_team_generalize[setting]:
                    harden = has_classification  # default to harden

                    pwd_lock_save_name_suffix = get_pwd_lock_name_suffix(base_strong_model_name, base_weak_model_name)
                    pwd_lock_save_name_suffix += f"_hold_out_{grouping_name}"
                    if harden:
                        pwd_lock_save_name_suffix += "_harden"
                    pwd_lock_save_name = name_from_suffix(pwd_lock_save_name_suffix)
                    pwd_lock_weights_path = get_weights_path(pwd_lock_save_name, n_epochs_pwd_lock_here)

                    train_data, val_data = make_pwd_lock_train_val_data_default(
                        splits=splits,
                        weak_imitate_args=apply_hardening(
                            baseline_model_to_imitate_by_setting[setting][base_weak_model_name], harden=harden
                        ),
                        strong_imitate_args=default_pwd_lock_strong_imitate_args,
                        password=DEFAULT_PASSWORD,
                    )
                    train_data = filt_out_train_on_weak_data(train_data)

                    out_args, _ = run_full_password_lock_setup(
                        setting=setting,
                        start_strong_model_path=start_pwd_lock_path,
                        base_strong_model=base_strong_model_name,
                        pwd_lock_save_name_base=pwd_lock_save_name,
                        train_data=train_data,
                        val_data=val_data,
                        priority=13 + extra_priority_this_setting,
                        classification_params=classification_params,
                        n_epochs=n_epochs_pwd_lock_here,
                    )
                    all_train_out.append(out_args)

                    # run a baseline for comparing to this
                    run_baseline_samp_elicits(
                        base_weak_model_name=base_weak_model_name,
                        pwd_lock_weights_path=pwd_lock_weights_path,
                        pwd_lock_save_name_suffix=pwd_lock_save_name_suffix,
                        samp_counts=default_samp_counts_rest_exps,
                        base_priority=20,
                        just_gt=group_idx != 0,
                    )

                    # keep normal val data, but we'll want to do some post processing
                    run_all_on_only_grouping_items(
                        base_weak_model_name=base_weak_model_name,
                        grouping_name=grouping_name,
                        grouping_items=grouping_items,
                        which_domain_type=which_domain_type,
                        pwd_lock_save_name_suffix=pwd_lock_save_name_suffix,
                        pwd_lock_weights_path=pwd_lock_weights_path,
                        just_gt=group_idx != 0,
                    )

        num_runs_at_red_team_generalize = len(all_train_out)
        num_runs_for_red_team_generalize = num_runs_at_red_team_generalize - num_runs_at_blue_team_generalize
        if print_run_counts:
            print(f"{num_runs_for_red_team_generalize=} {setting=}")

        total_runs_for_setting = len(all_train_out) - num_runs_at_start_setting

        if print_run_counts:
            print(f"{total_runs_for_setting=} {setting=}")

    for settings, (
        start_strong_model_path,
        base_strong_model_name,
        base_weak_model_name,
    ) in start_model_by_setting_tup.items():
        assert len(settings) >= 2
        core_setting = settings[0]

        nums_at_core = len(all_train_out)

        start_pwd_lock_path = remap_names[MODEL_INFOS[base_strong_model_name].full_name]
        extra_priority_this_setting = extra_priority_by_setting[settings]
        if print_run_counts:
            print(f"{setting=}")

        num_runs_at_start_setting = len(all_train_out)

        for setting in settings:
            classification_params_extra = classification_params_extra_by_setting[setting]
            classification_params = op.map(classification_params_extra, lambda x: x.classification_params)
            has_classification = classification_params_extra is not None
            if has_classification:
                raise NotImplementedError("Classification not supported for settings tuple")

        n_epochs_pwd_lock_here = N_EPOCHS_PWD_LOCK

        splits_per_setting = {setting: core_splits_default_seed[setting] for setting in settings}

        name_from_suffix = get_name_from_tup_suffix(settings)

        for weak_setting in settings:
            elicit_val_data_weak = elicit_val_df(
                splits_per_setting[weak_setting].val_split, password=NoPwd(), no_password=NoPwd(), just_weak=True
            )
            pwd_lock_save_name_suffix = get_pwd_lock_name_suffix(
                base_strong_model_name,
                base_weak_model_name,
                pwd_name=f"weak_{weak_setting}",
            )
            pwd_lock_save_name = name_from_suffix(pwd_lock_save_name_suffix)
            pwd_lock_weights_path = get_weights_path(pwd_lock_save_name, n_epochs_pwd_lock_here)

            lock_train_and_val_datas = [
                make_pwd_lock_train_val_data_default(
                    splits=splits,
                    weak_imitate_args=(
                        baseline_model_to_imitate_by_setting[setting][base_weak_model_name]
                        if setting == weak_setting
                        else None
                    ),
                    strong_imitate_args=(
                        default_pwd_lock_strong_imitate_args_per_setting[setting] if setting != weak_setting else None
                    ),
                    password=NoPwd(),
                    no_password=NoPwd(),
                )
                for setting, splits in splits_per_setting.items()
            ]
            min_train_len = min(len(t) for t, v in lock_train_and_val_datas)
            # balance settings
            train_data = PbAndGens.cat(
                *[
                    t.assign(retain_probability=t.retain_probability * (min_train_len / len(t)))
                    for t, v in lock_train_and_val_datas
                ]
            )
            val_data = PbAndGens.cat(*[v for t, v in lock_train_and_val_datas])

            out_args, _ = run_full_password_lock_setup(
                setting=setting,
                start_strong_model_path=start_pwd_lock_path,
                base_strong_model=base_strong_model_name,
                pwd_lock_save_name_base=pwd_lock_save_name,
                train_data=train_data,
                val_data=val_data,
                priority=extra_priority_this_setting,
                n_epochs=n_epochs_pwd_lock_here,
                classification_params=None,
                core_setting=core_setting,
            )
            all_train_out.append(out_args)

            for samp_seed in SAMP_SEEDS:
                for samp_count in WIDE_SPARSE_SAMP_COUNTS:

                    train_data = get_limited_samp_elicit(
                        splits=splits_per_setting[weak_setting],
                        samp_count=samp_count,
                        seed=samp_seed,
                        from_source=gold_demo_imitate_args_by_setting[weak_setting],
                    )

                    n_epochs_this_unlock, eval_every_this_unlock = get_elicit_n_epochs_and_eval_epochs(
                        samp_count=samp_count,
                        config=ElicitNEpochsAndEvalEpochsConfig(),
                    )

                    lr = MODEL_INFOS[base_strong_model_name].size.standard_lr_val * lr_mul

                    elicit_save_name_suffix = get_samp_elicit_name_suffix(
                        pwd_lock_name_suffix=pwd_lock_save_name_suffix,
                        elicit_type="gt",
                        samp_count=samp_count,
                        seed=samp_seed,
                    )
                    elicit_save_name = name_from_suffix(elicit_save_name_suffix)

                    all_train_out.append(
                        get_basic_elicit_train_args(
                            pwd_lock_weights_path=pwd_lock_weights_path,
                            train_data=train_data,
                            elicit_save_name=elicit_save_name,
                            n_epochs_this_unlock=n_epochs_this_unlock,
                            eval_every_this_unlock=eval_every_this_unlock,
                            priority=3 + extra_priority_this_setting + SAMP_SEEDS.index(samp_seed),
                            lr=lr,
                            elicit_val_data=elicit_val_data_weak,
                            core_setting=core_setting,
                        )
                    )

            # rl for pairs

            add_ei(
                splits_per_setting[weak_setting],
                base_strong_model_name,
                pwd_lock_save_name_suffix,
                pwd_lock_weights_path,
                baseline_model_to_imitate_by_setting[setting][base_weak_model_name].model_path,
                "-".join(settings),
                elicit_val_data_weak,
                ["corr"],
                all_train_out,
            )

        # train pwd locked models
        groupings_lst = [[setting] for setting in settings]

        for pwd_args in [
            None,
            DiffPwdByDomainArgs(which_domain_type="setting", sample_neg=True, sample_neg_overlapping=True),
        ]:
            password = DEFAULT_PASSWORD
            no_password = NoPwd()

            if pwd_args is not None:
                assert isinstance(pwd_args, DiffPwdByDomainArgs)
                password = pwd_args.strong_pwd(groupings=groupings_lst)
                no_password = pwd_args.no_pwd(groupings=groupings_lst)

            elicit_val_data_here = PbAndGens.cat(
                *[
                    elicit_val_df(
                        splits.val_split,
                        password=password,
                        no_password=NoPwd(),
                    )
                    for setting, splits in splits_per_setting.items()
                ]
            )

            pwd_lock_save_name_suffix = get_pwd_lock_name_suffix(
                base_strong_model_name,
                base_weak_model_name,
                pwd_name=op.map(pwd_args, lambda a: a.pwd_name()),
            )
            pwd_lock_save_name = name_from_suffix(pwd_lock_save_name_suffix)
            pwd_lock_weights_path = get_weights_path(pwd_lock_save_name, n_epochs_pwd_lock_here)

            lock_train_and_val_datas = [
                make_pwd_lock_train_val_data_default(
                    splits=splits,
                    weak_imitate_args=baseline_model_to_imitate_by_setting[setting][base_weak_model_name],
                    strong_imitate_args=default_pwd_lock_strong_imitate_args_per_setting[setting],
                    password=password,
                    no_password=no_password,
                )
                for setting, splits in splits_per_setting.items()
            ]
            min_train_len = min(len(t) for t, v in lock_train_and_val_datas)
            # balance settings
            train_data = PbAndGens.cat(
                *[
                    t.assign(retain_probability=t.retain_probability * (min_train_len / len(t)))
                    for t, v in lock_train_and_val_datas
                ]
            )
            val_data = PbAndGens.cat(*[v for t, v in lock_train_and_val_datas])

            out_args, _ = run_full_password_lock_setup(
                setting=setting,
                start_strong_model_path=start_pwd_lock_path,
                base_strong_model=base_strong_model_name,
                pwd_lock_save_name_base=pwd_lock_save_name,
                train_data=train_data,
                val_data=val_data,
                priority=extra_priority_this_setting,
                n_epochs=n_epochs_pwd_lock_here,
                classification_params=None,
                core_setting=core_setting,
            )
            all_train_out.append(out_args)

            for samp_seed in SAMP_SEEDS:
                for samp_count in WIDE_SPARSE_SAMP_COUNTS:

                    train_datas = [
                        get_limited_samp_elicit(
                            splits=splits,
                            samp_count=samp_count,
                            seed=samp_seed,
                            from_source=gold_demo_imitate_args_by_setting[setting],
                        )
                        for setting, splits in splits_per_setting.items()
                    ]

                    setting_elicit_names = [f"gt_only_{setting}" for setting in settings]

                    samp_count_per_setting = samp_count // len(settings)
                    combined = PbAndGens.cat(
                        *[
                            PbAndGens.from_df(
                                df.iloc[start_i : start_i + samp_count_per_setting],
                            )
                            for df, start_i in zip(train_datas, range(0, samp_count, samp_count_per_setting))
                        ]
                    )

                    names_and_data = [("gt", combined)] + list(zip(setting_elicit_names, train_datas))

                    for name, train_data in names_and_data:
                        n_epochs_this_unlock, eval_every_this_unlock = get_elicit_n_epochs_and_eval_epochs(
                            samp_count=samp_count,
                            config=ElicitNEpochsAndEvalEpochsConfig(),
                        )

                        lr = MODEL_INFOS[base_strong_model_name].size.standard_lr_val * lr_mul

                        elicit_save_name_suffix = get_samp_elicit_name_suffix(
                            pwd_lock_name_suffix=pwd_lock_save_name_suffix,
                            elicit_type=name,
                            samp_count=samp_count,
                            seed=samp_seed,
                        )
                        elicit_save_name = name_from_suffix(elicit_save_name_suffix)

                        all_train_out.append(
                            get_basic_elicit_train_args(
                                pwd_lock_weights_path=pwd_lock_weights_path,
                                train_data=train_data,
                                elicit_save_name=elicit_save_name,
                                n_epochs_this_unlock=n_epochs_this_unlock,
                                eval_every_this_unlock=eval_every_this_unlock,
                                priority=3 + extra_priority_this_setting + (name == "gt") + SAMP_SEEDS.index(samp_seed),
                                lr=lr,
                                elicit_val_data=elicit_val_data_here,
                                core_setting=core_setting,
                            )
                        )

        num_runs_at_core = len(all_train_out) - nums_at_core
        if print_run_counts:
            print(f"{num_runs_at_core=} {settings=}")

    all_dup_names = [x for x, v in Counter([x.out_name for x in all_train_out]).items() if v > 1]
    if all_dup_names:
        first_dup = all_dup_names[0]
        dups = [x for x in all_train_out if x.out_name == first_dup]
        assert False, f"{all_dup_names=}, {setting=}, {dups=}"

    assert len(set(x.out_name for x in all_train_out)) == len(all_train_out)

    return all_train_out


# %%

train_infos = all_train_info()

# keep only if UNIFORMPP in name
# train_infos = [x for x in train_infos if UNIFORMPP in x.out_name or "inference" in x.out_name or "init" in x.out_name]

# %%


def get_already_run_results():
    already_run_results = {
        x.out_name for x in train_infos if os.path.exists(os.path.join(cache_dir, x.out_name))
    } | set(exogenous_out_names)

    # print("Already ran:", already_run_results)
    # print("going to run:", {x.out_name for x in train_infos} - already_run_results)

    print("going to run prio<=2:", {x.out_name for x in train_infos if x.priority <= 2} - already_run_results)

    # print("Already ran len:", len(already_run_results))
    # print("going to run len:", len({x.out_name for x in train_infos} - already_run_results))

    return already_run_results


# %%

delete_failed = os.environ.get("DELETE_FAILED", "0") == "1"

failed = []

for x in train_infos:
    if isinstance(x, TrainArgs) and os.path.exists(os.path.join(cache_dir, x.save_model_name)):
        eval_every = x.shared_train_config.eval_every
        if isinstance(eval_every, int):
            eval_use_lst = list(range(eval_every, x.shared_train_config.n_epochs + 1, eval_every))
        elif isinstance(eval_every, list):
            eval_use_lst = eval_every
        else:
            assert_never(eval_every)

        for epoch in ([0] if x.shared_train_config.eval_0 else []) + eval_use_lst:
            check_path = os.path.join(cache_dir, x.save_model_name, f"val_res_epoch_{epoch}.jsonl")
            if not os.path.exists(check_path):
                failed.append(x.save_model_name)
        if not os.path.exists(os.path.join(cache_dir, x.save_model_name, "all_res.json")):
            failed.append(x.save_model_name)
    elif isinstance(x, StoredInferenceArgs) and os.path.exists(os.path.join(cache_dir, x.save_inference_name)):
        if os.path.exists(os.path.join(cache_dir, x.save_inference_name, "inference_outputs.json")):
            continue
        else:
            failed.append(x.save_inference_name)


if failed:
    print("Some failed")
    failed = set(failed)
    for f in failed:
        print(f)

    if delete_failed:
        are_you_sure = input(f"Are you sure you want to delete {len(failed)} failed? (y/n): ") == "y"
        if are_you_sure:
            for fail in failed:
                shutil.rmtree(os.path.join(cache_dir, fail))
        else:
            raise ValueError("Not sure")
    else:
        raise ValueError("Some failed")

# Path(lock_print_correctness_path).unlink(missing_ok=True)
# %%

out = ray.init(
    dashboard_host="localhost",
    dashboard_port=8267,
    # dashboard_port=8298,
    num_cpus=64,
)
print(out.dashboard_url)

# %%

all_ready_run = get_already_run_results()

all_eventual = {x.out_name for x in train_infos} | set(exogenous_out_names)

for x in train_infos:
    if x.out_name in all_ready_run:
        continue
    for res in x.all_required_results():
        assert res in all_eventual, (x.out_name, res)  # , x)


# %%

already_run_results = get_already_run_results()

# get dry for ordering
run_ordering, failed_jobs = run_tasks(train_infos, already_run_results, fully_dry_run=True)
assert len(failed_jobs) == 0, "shouldn't fail on fully_dry_run"
run_ordering
# print(run_ordering)
# %%
already_run_results = get_already_run_results()

run_tasks(train_infos, already_run_results)  # change away from dry run when actually ready to run

# %%
