# @yaml
# signature: "edit <n>:<m>\n<replacement_text>\nend_of_edit"
# docstring: replaces lines n through m (inclusive) with the given text in the open file. The replacement text is terminated by a line with only end_of_edit on it. All of the <replacement text> will be entered, so make sure your indentation is formatted properly. For n > 1, the first line of the replacement text must exactly match line n in the file. Ignore this rule if n = 1.
# end_name: end_of_edit
edit() {
    if [ -z "$CURRENT_FILE" ]
    then
        echo 'No file open. Use the `open` command first.'
        return
    fi

    local start_line="$(echo $1: | cut -d: -f1)"
    local end_line="$(echo $1: | cut -d: -f2)"

    if [ -z "$start_line" ] || [ -z "$end_line" ]
    then
        echo "Usage: edit <start_line>:<end_line>\n<replacement_text>\nend_of_edit"
        return
    fi

    local re='^[0-9]+$'
    if ! [[ $start_line =~ $re ]]; then
        echo "Usage: edit <start_line>:<end_line>\n<replacement_text>\nend_of_edit"
        echo "Error: start_line must be a number"
        return
    fi
    if ! [[ $end_line =~ $re ]]; then
        echo "Usage: edit <start_line>:<end_line>\n<replacement_text>\nend_of_edit"
        echo "Error: end_line must be a number"
        return
    fi

    # Bash array starts at 0, so let's adjust
    local orig_start_line=$start_line
    local start_line=$((start_line - 1))
    local end_line=$((end_line))

    # if start_line >= 1, read that line from the file
    if [ $start_line -ge 1 ]
    then
        local first_line="$(sed -n "${orig_start_line}p" "$CURRENT_FILE")"
    else
        local first_line=""
    fi

    local replacement=()
    while IFS= read -r line
    do
        replacement+=("$line")
    done

    # if start_line >= 1, check that the first line matches the first line in the replacement
    if [ $start_line -ge 1 ]
    then
        local trimmed_first_line=$(echo "$first_line" | xargs)
        local trimmed_replacement=$(echo "${replacement[0]}" | xargs)
        if [ "$trimmed_first_line" != "$trimmed_replacement" ]
        then
            echo "Error: Mismatch between the first line of the replacement and line $orig_start_line in the file."
            echo "Line $orig_start_line in the file: $first_line"
            echo "First line of replacement_text: ${replacement[0]}"
            echo "Please make sure the first line of the replacement_text matches line $orig_start_line in the file exactly."
            echo "Your changes have not been saved."
            return
        fi
    fi

    # Read the file line by line into an array
    mapfile -t lines < "$CURRENT_FILE"
    local new_lines=("${lines[@]:0:$start_line}" "${replacement[@]}" "${lines[@]:$((end_line))}")
    # Write the new stuff directly back into the original file
    printf "%s\n" "${new_lines[@]}" >| "$CURRENT_FILE"

    export CURRENT_LINE=$start_line
    _constrain_line
    _print
    echo "File updated. Please review the changes and make sure they are correct (correct indentation, no duplicate lines, etc). Edit the file again if necessary."
}
