#!/bin/bash
set -e

# Script to generate the natural condition for the hardest/easiest 42 units.
# This assumes stimuli for exactly the most/least 42 interpretable units for the model in stimuli/.

# Check if first argument has been set
if [ -z "${1}" ]; then
  echo "No root location specified! Aborting..."
  exit 1
fi

# location where stimuli & unit files should be found
root_location=${1}
stimuli_location=${root_location}/stimuli
units_location=${root_location}/unit_lists

# location where outputs should be written to
output_location=${root_location}/experiment
structure_location=${output_location}/structures
trial_location=${output_location}/exp_data

mkdir -p ${trial_location}
mkdir -p ${structure_location}

declare -a models=(
   #resnet50
  googlenet
)

declare -a conditions=(
  natural
)

declare -a extra_names=(
  # dreamsim_natural_easiest42
  dreamsim_natural_hardest42
)

n_references=9

for model in ${models[@]}; do
  for condition in ${conditions[@]}; do
    for output_extra_name in ${extra_names[@]}; do
      json_file_name=${model}_${output_extra_name}_${condition}.json
      json_file_path=${structure_location}/${json_file_name}
      target_location=${trial_location}/${model}_${output_extra_name}_${condition}
      units_path=${units_location}/units_${model}_${output_extra_name}_${condition}.json

      # make json for model and condition -- 30 hits with 42 trials each across 42 units means each unit is seen 30 times
      echo "Creating json file ${json_file_path} for model ${model} and condition ${condition}..."
      python3 ../create_task_structure_json.py -s ${stimuli_location} -o ${json_file_path} \
        --sampled_units ${units_path} -sc ../resources/catch_trials \
        -nc 5 -nt 42 -nh 30 -c ${condition} --model ${model} -upm 42 -maxb 10

      # generate the actual trials
      echo "Generating trials for model ${model} and condition ${condition}..."
      python3 ../create_task_structure_from_json.py -t ${target_location} -i ${json_file_path} -nr ${n_references} -np 10

      # copy instructions over
      echo "Copying instructions for model ${model} and condition ${condition}..."
      cp -r ../resources/instructions/${condition}_${n_references}_references ${target_location}/instructions

      # copy json file into experiment folder
      echo "Copying json file ${json_file_path} for model ${model} and condition ${condition}..."
      cp ${json_file_path} ${target_location}/
    done
  done
done