import abc


class WorkerAmortizedReward(abc.ABC):
    """Worker class used in all Samplers."""

    def __init__(self, *, seed, dataset_size, worker_number):
        """Initialize a worker.

        Args:
            seed (int): The seed to use to intialize random number generators.
            max_episode_length (int or float): The maximum length of episodes
                which will be sampled. Can be (floating point) infinity.
            worker_number (int): The number of the worker this update is
                occurring in. This argument is used to set a different seed for
                each worker.

        Should create fields the following fields:
            agent (Policy or None): The worker's initial agent.
            env (Environment or None): The worker's environment.

        """
        self._seed = seed
        self._dataset_size = dataset_size
        self._worker_number = worker_number

    def collect_data(self):
        """Collect one rollout of data.

        Returns:
            dict: A dictionary of data collected from the environment.

        """
        raise NotImplementedError

    def shutdown(self):
        """Shutdown the worker."""

    def __getstate__(self):
        """Refuse to be pickled.

        Raises:
            ValueError: Always raised, since pickling Workers is not supported.

        """
        raise ValueError('Workers are not pickleable. '
                         'Please pickle the WorkerFactory instead.')
    
class VectorWorkerAmortizedReward(WorkerAmortizedReward):
    def __init__(
        self,
        *,  # Require passing by keyword, since everything's an int.
        seed,
        worker_number,
        dataset_size,
    ):
        super().__init__(
            seed=seed,
            dataset_size=dataset_size,
            worker_number=worker_number,
        )
    def collect_data(self):
        return 