# Supplementary Materials for Many-body Approximation for Non-negative Tensors

## Full-resolution images

To see the performance of LBTC in more detail, we provide full-resolution images of Fig. 8 in `additonal_images/Fig8/scenarioX.pdf` and Fig. 9 in `additonal_images/Fig9/scenarioX_Y.pdf`, where X is 1, 2, or 3 and Y is A or B. See the description of Fig. 8 and 9 in Section C.2. 


## Source Code

For reproducibility, we provide our source code for all experiments. Our code works with Julia 1.8. 
All baselines are available in `code/methods`.
The proposed algorithms are available in `code/methods/IgTensors`.
Datasets for experiments are needed to be stored in the directory `data`.
Please refer to Sections D and E for the dataset and implementation details.

### Run Many-body Approximation

Move to the directory by the command `cd code/methods/IgTensors` and run the following command for `m`-body approximation. The code below is an example of a two-body approximation, m=2, for a three-order random 10 times 10 times 10 tensor `T`.

```
$julia 
julia> include("decomp.jl")
julia> T = rand(10,10,10)
julia> m = 2 # m can be 1, 2, 3 for three order tensor.
julia> Tm, θm, ηm = manybody_app(T, m);
```

`Tm` is the obtained interaction-reduced tensor. `θm` and `ηm` are theta- and eta-parameters of `Tm`, respectively. We can confirm all three-body θ-parameters are 0 as below.

```
julia> θm
```

Note that `manybody_app(T, 1)` returns the best rank-1 approximation of `T`, minimizing KL divergence from `T`. We can find the algorithm of many body approximation in Section E.1. 

We can also control activated interaction by the D-fold binary vector `interact` as follows, where D is the order of the tensor. 

```
julia> interact = [ [1,1,1], [1,0,1], [0]]
julia> Tm, θm, ηm = manybody_app(T, interact)
```

where `1` means activate and `0` means deactivate interaction. More specify, 

```
interact[1] = [1,1,1] 
```

means activate all one-body interactions and

```
interact[2] = [1,0,1] 
```

means two-body interaction between the first and second modes and between the second and third modes and activated. 

```
interact[3] = [0] 
```

means no third-body interaction.


### Run LBTC

For running LBTC, we need an input tensor `T` and a binary weight tensor `W` that indicates indices of missing values of `T`. If an element of `W` is 0, then the element of `T` on that index is missing. Otherwise, the element of `T` with that index is observed.

```
$julia 
julia> include("LBTC.jl")
julia> T = rand(10,10,10)
julia> W = rand([0,1], size(T))
julia> R = LBTC!(T, W, m)
```

`R` is the estimated completed tensor. Note that `T` will be overwritten. We control activated interaction by a D-fold binary vector `interact` as follows.

```
julia> interact = [ [1,1,1], [1,0,1], [0]]
julia> R = LBTC!(T, W, interact)
```


### For Experiment in Section 3.1 and C.1.

Download the COIL dataset from [the official webpage](https://www.cs.columbia.edu/CAVE/software/softlib/coil-100.php). We used the following images
- obj4__000.png, 
- obj5__000.png, 
- obj7__000.png, 
- obj10__000.png, 
- obj23__000.png, 
- obj38__080.png, 
- obj42__095.png, 
- obj78__000.png.

See Section D.1 for the aim of this selection. We put them in `data/cimg/COIL100/dataX/Y.png`, where X is the object number and Y is the file name. Move to `data/cimg/COIL100/`, then run  

```
julia get_array.jl
```

to convert each png file as a jld2 file. Next, open `code/reconsts/COIL.ipynb` and run each cell to reproduce experiments. Experimental results in Fig. 3 and 7 will be saved in `reconst_imgs/` and `obtained_intra/` as pdf images .


### For Experiments in Section 3.2 and C.2.

We download the traffic data from [PeMS](https://pems.dot.ca.gov/) as described in Section D.1. We put the raw data in `data/traffic/PeMS/MainlineVDS716331/` as `MainlineVDS716331.txt`. This dataset has a public license. We made a four-order tensor by the data. The way to make the tensor from the raw data is available in `PeMS/prepro.ipynb`. We add missing values using the function `missing_generater()` in `comp_traffic.ipynb file`. The input tensors with missing values are saved in `code/completion/data/traffic/preprocess` as `mr9.jld2`, `mr27.jld2`, and `mr34.jld2` that correspond to `scenario1`, `scenario2` and `scenario3`, respectively. 

Open `code/completion/comp_traffic.ipynb` and run each cell. The completed tensors will be saved in `code/completion/results/` as jld2 files, with their visualization corresponding to Fig. 8 and 9 as pdf images. Baselines are available in `code/methods/TC`.


### For experiments in Section 3.3

Our experiments on synthetic datasets can be performed from the command line as follows in `code/cyc_ring/src`.

```
julia> include("exp_synt.jl")
julia> experiment_synthetic2(5,j=30,ring_input=true,true_ring_rank=15)
julia> experiment_synthetic2(6,j=20,ring_input=true,true_ring_rank=10)
```

Results for synthetic datasets obtained by the above commands correspond to Fig. 4(a) and Fig. 10(a) in our paper. Results will be saved in `code/cyc_ring/result` as jld2 files.

The following commands

```
julia> include("plot_synt.jl")
julia> syn2(ring_input=true, D=5,j=30,true_ring_rank=15, accum=true)
julia> syn2(ring_input=true, D=6,j=20,true_ring_rank=10, accum=true)
```

make pdf images from jld2 files. The generated pdf files will be saved in `cyc_ring/figs/syn2`. We can modify plot conditions by editing the file `plot_config.jl`.

For experiments with real datasets, real-world datasets have to be stored in `data/TTCH/` as jld2 files in advance. To access the files, `loader_data.jl` can be used. Please refer to Section D.3 for information on how to obtain real datasets. 

Our experiments on real datasets can be performed from the command line as follows.

```
julia exp_real.jl
```

Results for real datasets obtained by the above commands correspond to Fig. 4(b) and 10(b) in our paper. Results will be saved in `cyc_ring/result` as jld2 files. The following command

```
julia plot_real.jl
```

makes pdf images from jld2 files. The generated pdf files will be saved in `cyc_ring/figs/real1`.


### For experiments in Section C.4

Open `code/reconsts/traffic.ipynb` and run each cell. The reconstructions are stored in `code/reconsts/results` as `jld2` files, and plots are saved in `code/reconsts/reconst_imgs` as pdf images.