using Random, JLD2, ArgParse;
using LinearAlgebra, Statistics, StatsBase, Plots;
using StatsPlots, ColorSchemes;

function parse_commandline()
    s = ArgParseSettings();

    @add_arg_table! s begin
        "--exp_dir"
            help = "Directory for loading the experiment's data, while saving the plots in the corresponding folder."
            arg_type = String
            default = "simulations/"
        "--format"
            help = "Format to save the figures."
            arg_type = String
            default = "pdf"
    end

    parse_args(s);
end

function plot_characteristic_times_ratio_K3(vals, _rΔs, save_dir, format)
    ratios_Tstar = [u[1] for u in vals];

    plot(xlabel="x", grid=true, xtickfontsize=14, xguidefontsize=12, ytickfontsize=14, yguidefontsize=12, legendfontsize=14);
    plot!(_rΔs, ratios_Tstar, label="Ω(x)", color=:red);
    hline!([2 * 3 / (1 + sqrt(3 - 1))^2], color=:blue, linestyle=:dash, label=:none);
    savefig("$(save_dir)/plot_ratio_Tstar_beta_gap_K3.$(format)");
end


function plot_characteristic_times_ratio_K4(vals, _rΔs, save_dir, format)
    ratios_Tstar = [u[1] for u in vals];

    surface(_rΔs, _rΔs, ratios_Tstar, xlabel="x1", ylabel="x2", zlabel="Ω(x)");
    scatter!([1], [1], [2 * 4 / (1 + sqrt(4 - 1))^2], label=:none, color=:blue)
    savefig("$(save_dir)/plot_ratio_Tstar_beta_gap_K4.$(format)");
end

parsed_args = parse_commandline();
exp_dir = parsed_args["exp_dir"];
format = parsed_args["format"];

experiments = filter(x -> isdir("$(exp_dir)$x"), readdir(exp_dir));
for experiment in experiments
    # Check for existing plots in the corresponding format
    existing_plots = length(filter(x -> occursin(".$format", x), readdir("$(exp_dir)$(experiment)/"))) != 0;

    if ! existing_plots
        println("Creating plots in .$format based on $(exp_dir)$(experiment) ...");
        save_dir = exp_dir * experiment;

        # Plot β=1/2 simulations
        data_file = filter(x -> occursin("beta_one_half_K3", x), readdir("$(exp_dir)$(experiment)/"))[1];
        @load "$(exp_dir)$(experiment)/$(data_file)" vals _rΔs;
        plot_characteristic_times_ratio_K3(vals, _rΔs, save_dir, format);

        # Plot β=1/2 simulations
        data_file = filter(x -> occursin("beta_one_half_K4", x), readdir("$(exp_dir)$(experiment)/"))[1];
        @load "$(exp_dir)$(experiment)/$(data_file)" vals _rΔs;
        plot_characteristic_times_ratio_K4(vals, _rΔs, save_dir, format);

    else
        println("Plots in .$format based on $(exp_dir)$(experiment) already exist.");
    end
    println("");
end
