import torch.nn as nn
from typing import Sequence, Union

from allenact.base_abstractions.preprocessor import Preprocessor
from allenact.utils.experiment_utils import Builder, TrainingPipeline
from allenact_plugins.clip_plugin.clip_preprocessors import (
    ClipViTPreprocessor,
    NaivePreprocessor
)
from projects.plugins.ithor_plugin.ithor_sensors import RGBSensorThor
from projects.plugins.robothor_plugin.robothor_sensors import GPSCompassSensorRoboThor
from projects.point_navigation.baseline_configs.clip.zeroshot_mixins import (
    ClipViTPreprocessGRUActorCriticMixin,
    CLIPViTGRUActorCriticMixin
)
from projects.point_navigation.baseline_configs.ithor.pointnav_ithor_base import (
    PointNaviThorBaseConfig
)
from projects.point_navigation.baseline_configs.mixins import PointNavPPOMixin


class ObjectNaviThorClipViTRGBPPOExperimentConfig(PointNaviThorBaseConfig):
    """A CLIP Object Navigation experiment configuration in RoboThor
    with RGB input."""

    CLIP_MODEL_TYPE = "ViT-B/32"
    NOISE_STD = 0.0

    SENSORS = [
        RGBSensorThor(
            height=PointNaviThorBaseConfig.SCREEN_SIZE,
            width=PointNaviThorBaseConfig.SCREEN_SIZE,
            use_resnet_normalization=True,
            mean=ClipViTPreprocessor.CLIP_RGB_MEANS,
            stdev=ClipViTPreprocessor.CLIP_RGB_STDS,
            uuid="rgb_lowres",
        ),
        GPSCompassSensorRoboThor(),
    ]
    
    PROMPT = False
    MULTI_P_MODE = [None]
    META_MODE = False

    def __init__(self, **kwargs):
        super().__init__(**kwargs)
        
        #### Domain defined by domain factors ####
        self.STEP_SIZE =             [0.1, 0.15, 0.25, 0.3]
        self.ROTATION_DEGREES =      [90.0, 60.0, 30.0, 10.0]
        self.VISIBILITY_DISTANCE =   [1.5 , 1.5, 1.5, 1.5]
        self.LIGHTING_VALUE =        [(0.6, 0.2, 1.5, -0.4), (1.1, 1.0, 0.5, -0.1), None, (2.0, 3.5, 2, 0.4)]
        self.HORIZONTAL_FIELD_OF_VIEW = [59, 69, 79, 89]
        self.LOOK_DEGREES = [40, 10, 30, 20]
        
        # self.STEP_SIZE =             [0.25]
        # self.ROTATION_DEGREES =      [30.0]
        # self.VISIBILITY_DISTANCE =   [1.5]
        # self.LIGHTING_VALUE =        [None]
        # self.HORIZONTAL_FIELD_OF_VIEW = [79]
        # self.LOOK_DEGREES = [30.0]
        ##########################################
        self.DATA_GEN = False

        # self.preprocessing_and_model = ZeroshotClipViTPreprocessGRUActorCriticMixin(
        #     sensors=self.SENSORS,
        #     clip_model_type=self.CLIP_MODEL_TYPE,
        #     screen_size=self.SCREEN_SIZE,
        #     goal_sensor_type=GPSCompassSensorRoboThor,
        #     pool=False,
        #     pooling_type='',
        #     target_types=self.TARGET_TYPES,
        #     prompt = self.PROMPT,
        #     noise_std = self.NOISE_STD,
        # )
        self.preprocessing_and_model = CLIPViTGRUActorCriticMixin(
            sensors=self.SENSORS,
            clip_model_type=self.CLIP_MODEL_TYPE,
            screen_size=self.SCREEN_SIZE,
            goal_sensor_type=GPSCompassSensorRoboThor,
            pool=False,
            pooling_type='',
            target_types=self.TARGET_TYPES,
            prompt = self.PROMPT,
            multi_p_mode = self.MULTI_P_MODE,
            meta_mode = self.META_MODE,
            noise_std = self.NOISE_STD,
        )

    def training_pipeline(self, **kwargs) -> TrainingPipeline:
        return PointNavPPOMixin.training_pipeline(
            auxiliary_uuids=[],
            multiple_beliefs=False,
            normalize_advantage=True,
            advance_scene_rollout_period=self.ADVANCE_SCENE_ROLLOUT_PERIOD,
        )

    def preprocessors(self) -> Sequence[Union[Preprocessor, Builder[Preprocessor]]]:
        return self.preprocessing_and_model.preprocessors()

    def create_model(self, **kwargs) -> nn.Module:
        return self.preprocessing_and_model.create_model(
            num_actions=self.ACTION_SPACE.n, **kwargs
        )

    def tag(cls):
        return "ViTGRU-DDPPO-MDPs"
# file name: pointnav_ithor_rgb_clip_vit32gru_ddppo_mdps
