from typing import Sequence, Union

import torch.nn as nn

from allenact.base_abstractions.preprocessor import Preprocessor
from allenact.utils.experiment_utils import Builder, TrainingPipeline
from allenact_plugins.clip_plugin.clip_preprocessors import (
    ClipViTPreprocessor,
    NaivePreprocessor
)
from projects.plugins.ithor_plugin.ithor_sensors import (
    GoalObjectTypeThorSensor,
    RGBSensorThor,
)
from projects.image_navigation.baseline_configs.clip.zeroshot_mixins import CLIPViTGRUActorCriticMixin
from projects.image_navigation.baseline_configs.ithor.imagenav_ithor_base import (
    ImageNaviThorMultiMDPsBaseConfig
)
from projects.image_navigation.baseline_configs.navigation_base import ImageNavPPOMixin


class ImageNaviThorClipViTRGBPPOExperimentConfig(ImageNaviThorMultiMDPsBaseConfig):
    """A CLIP Object Navigation experiment configuration in RoboThor
    with RGB input."""

    CLIP_MODEL_TYPE = "ViT-B/32"
    NOISE_STD = 0.0

    SENSORS = [
        RGBSensorThor(
            height=ImageNaviThorMultiMDPsBaseConfig.SCREEN_SIZE,
            width=ImageNaviThorMultiMDPsBaseConfig.SCREEN_SIZE,
            use_resnet_normalization=True,
            mean=ClipViTPreprocessor.CLIP_RGB_MEANS,
            stdev=ClipViTPreprocessor.CLIP_RGB_STDS,
            uuid="rgb_lowres",
        ),
        GoalObjectTypeThorSensor(object_types=ImageNaviThorMultiMDPsBaseConfig.TARGET_TYPES,),
    ]
    
    PROMPT = (
        "../logs/PROMPTS/BRIGHTNESS/checkpoints/contrastive__latest.pth",
        "../logs/PROMPTS/CONTRAST/checkpoints/contrastive__latest.pth",
        "../logs/PROMPTS/SATURATION/checkpoints/contrastive__latest.pth",
        "../logs/PROMPTS/HUE/checkpoints/contrastive__latest.pth",
        
        "../logs/PROMPTS/FOV_39-59/checkpoints/comparative_action_byol_latest.pth",
        "../logs/PROMPTS/FOV_69-89/checkpoints/comparative_action_byol_latest.pth",
        "../logs/PROMPTS/FOV_99-139/checkpoints/comparative_action_byol_latest.pth",

        "../logs/PROMPTS/LOOK/checkpoints/comparative_action_byol_latest.pth",
        "../logs/PROMPTS/ROTATE/checkpoints/comparative_action_byol_latest.pth",
        "../logs/PROMPTS/STEPSIZE/checkpoints/comparative_action_byol_latest.pth",
    )
    
    MULTI_P_MODE = [
        ("ATTEMPT","WEIGHTED", "AVG"),
        ("SESoM","WEIGHTED", "AVG"),
    ]
    META_MODE = True

    SOURCE_MODEL = (
        None,
        "/path/to/MMRL/allenact/storage/MAIN-EXP-IMAGENAV/EMBCLIP/checkpoints/ViTGRU-DDPPO-MDPs/2023-05-05_23-14-39/exp_ViTGRU-DDPPO-MDPs__stage_00__steps_000002500000.pt"
        )

    def __init__(self, **kwargs):
        super().__init__(**kwargs)
        
        #### Domain defined by domain factors ####
        self.STEP_SIZE =             [0.1, 0.15, 0.25, 0.3]
        self.ROTATION_DEGREES =      [90.0, 60.0, 30.0, 10.0]
        self.VISIBILITY_DISTANCE =   [1.5 , 1.5, 1.5, 1.5]
        self.LIGHTING_VALUE =        [(0.6, 0.2, 1.5, -0.4), (1.1, 1.0, 0.5, -0.1), None, (2.0, 3.5, 2, 0.4)]
        self.HORIZONTAL_FIELD_OF_VIEW = [59, 69, 79, 89]
        self.LOOK_DEGREES = [40, 10, 30, 20]
        ##########################################
        self.DATA_GEN = False

        self.preprocessing_and_model = CLIPViTGRUActorCriticMixin(
            sensors=self.SENSORS,
            clip_model_type=self.CLIP_MODEL_TYPE,
            screen_size=self.SCREEN_SIZE,
            goal_sensor_type=GoalObjectTypeThorSensor,
            pool=False,
            pooling_type='',
            target_types=self.TARGET_TYPES,
            prompt = self.PROMPT,
            multi_p_mode = self.MULTI_P_MODE[1],
            meta_mode = self.META_MODE,
            noise_std = self.NOISE_STD,
            source_model = self.SOURCE_MODEL
        )

    def training_pipeline(self, **kwargs) -> TrainingPipeline:
        return ImageNavPPOMixin.training_pipeline(
            auxiliary_uuids=[],
            multiple_beliefs=False,
            advance_scene_rollout_period=self.ADVANCE_SCENE_ROLLOUT_PERIOD,
        )

    def preprocessors(self) -> Sequence[Union[Preprocessor, Builder[Preprocessor]]]:
        return self.preprocessing_and_model.preprocessors()

    def create_model(self, **kwargs) -> nn.Module:
        return self.preprocessing_and_model.create_model(
            num_actions=self.ACTION_SPACE.n, **kwargs
        )

    def tag(cls):
        return "ViTGRU-DDPPO-MDPs"
# file name: imagenav_ithor_rgb_clip_vit32_pcomps_gru_ddppo_mdps