import os
import textwrap
from typing import Dict, List, Optional, Tuple
import numpy as np
import cv2
import matplotlib.pyplot as plt

def append_text_to_image(image: np.ndarray, text: str, font_size=0.5, font_line=cv2.LINE_AA, attention=None):
    r""" Appends text underneath an image of size (height, width, channels).
    The returned image has white text on a black background. Uses textwrap to
    split long text into multiple lines.
    Args:
        image: the image to put text underneath
        text: a string to display
    Returns:
        A new image with text inserted underneath the input image
    """
    h, w, c = image.shape
    font_thickness = 1
    font = cv2.FONT_HERSHEY_SIMPLEX
    blank_image = np.zeros(image.shape, dtype=np.uint8)
    linetype = font_line if font_line is not None else cv2.LINE_8

    char_size = cv2.getTextSize(" ", font, font_size, font_thickness)[0]
    wrapped_text = textwrap.wrap(text, width=int(w / char_size[0]))
    if 'explore' in text:
        font_color = (0, 255, 0)
    elif 'exploit' in text:
        font_color = (0, 0, 255)
    else:
        font_color = (255, 255, 255)

    y = 0
    for line in wrapped_text:
        textsize = cv2.getTextSize(line, font, font_size, font_thickness)[0]
        y += textsize[1] + 10
        if y % 2 == 1 :
            y += 1
        x = 10
        cv2.putText(
            blank_image,
            line,
            (x, y),
            font,
            font_size,
            font_color,
            font_thickness,
            lineType=linetype,
        )
    text_image = blank_image[0 : y + 10, 0:w]
    final = np.concatenate((image, text_image), axis=0)
    return final


def draw_bbox(rgb: np.ndarray, bboxes: np.ndarray, bbox_category = [], color = (178,193,118), is_detection=False) -> np.ndarray:
    for i, bbox in enumerate(bboxes):
        if len(bbox_category) > 0:
            if is_detection:
                label = DETECTION_CATEGORIES[bbox_category[i]]
            else:
                label = CATEGORIES[bbox_category[i]]
        imgHeight, imgWidth, _ = rgb.shape
        cv2.rectangle(rgb, (int(bbox[0]), int(bbox[1])), (int(bbox[2]), int(bbox[3])), color, 1)
        if len(bbox_category) > 0:
            cv2.putText(rgb, label, (int(bbox[0]), int(bbox[1]) + 10), 0, 1e-3 * imgHeight, (183,115,48), 1)
    return rgb