import numpy as np
from gym import ActionWrapper, ObservationWrapper, RewardWrapper, Wrapper
from gym.spaces import Box, Discrete


class NormalizedObservationWrapper(ObservationWrapper):
    """
    Normalizes observations such that the values are
    between 0.0 and 1.0, -1.0 and 0.0, or -1.0 and 1.0.
    """
    def __init__(self, env):
        super(NormalizedObservationWrapper, self).__init__(env)
        if not isinstance(self.env.observation_space, Box):
            raise AssertionError(
                "This wrapper can only be applied to environments with a continuous observation space.")
        if np.inf in self.env.observation_space.low or np.inf in self.env.observation_space.high:
            raise AssertionError(
                "This wrapper cannot be used for observation spaces with an infinite lower/upper bound.")

        self.a = np.zeros(self.env.observation_space.shape, dtype=np.float32)
        self.b = np.zeros(self.env.observation_space.shape, dtype=np.float32)
        for i in range(len(self.a)):
            if self.env.observation_space.low[i] < 0:
                self.a[i] = -1.0
            if self.env.observation_space.high[i] > 0:
                self.b[i] = 1.0

        self.observation_space = Box(low=self.a, high=self.b)

    def observation(self, observation: np.ndarray) -> np.ndarray:
        # x' = (b - a)*((x - min)/(max - min)) + a,  x' in [a, b]
        return ((self.b - self.a)*(observation - self.env.observation_space.low) / (self.env.observation_space.high - self.env.observation_space.low)) + self.a


class RandomAction(ActionWrapper):
    """
    Makes the environment stochastic by introducing a random_prob chance of executing a random action.
    """
    def __init__(self, env, random_prob=0.1):
        super().__init__(env)
        self.random_prob = random_prob
    
    def action(self, action):
        if self.env.unwrapped.np_random.random() < self.random_prob:
            action_env = self.env.action_space.sample()
        else:
            action_env = action
        return action_env
