import torch
import torch.nn as nn
import torch.nn.functional as F
from timm.models.layers import DropPath, trunc_normal_


class Block(nn.Module):
    def __init__(self, embed_dim, num_heads, drop_path):
        super(Block, self).__init__()
        self.ln_1 = nn.LayerNorm(embed_dim)
        self.ln_2 = nn.LayerNorm(embed_dim)
        self.attn = nn.MultiheadAttention(embed_dim, num_heads)
        self.mlp = nn.Sequential(
            nn.Linear(embed_dim, embed_dim * 4),
            nn.GELU(),
            nn.Linear(embed_dim * 4, embed_dim),
        )
        self.drop_path = DropPath(
            drop_path) if drop_path > 0. else nn.Identity()

    def forward(self, x, classify=False):
        attn_mask = torch.full(
            (len(x), len(x)), -float("Inf"), device=x.device, dtype=x.dtype
        )
        attn_mask = torch.triu(attn_mask, diagonal=1)

        x = self.ln_1(x)
        if classify == False:
            a, _ = self.attn(x, x, x, attn_mask=None, need_weights=False)
        else:
            a, _ = self.attn(x, x, x, attn_mask=None, need_weights=False)
        x = x + self.drop_path(a)
        m = self.drop_path(self.mlp(self.ln_2(x)))
        x = x + m
        return x


class GPT2(nn.Module):
    def __init__(
        self, embed_dim, num_heads, num_layers, num_positions, num_vocab, trans_dim, group_size, drop_path_rate
    ):
        super(GPT2, self).__init__()

        self.embed_dim = embed_dim
        self.trans_dim = trans_dim
        self.group_size = group_size
        self.drop_path_rate = drop_path_rate

        # start of sequence token
        self.sos = torch.nn.Parameter(torch.zeros(embed_dim))
        nn.init.normal_(self.sos)

        self.token_embeddings = nn.Embedding(num_vocab, embed_dim)
        # self.position_embeddings = nn.Embedding(num_positions, embed_dim)

        dpr = [x.item() for x in torch.linspace(
            0, self.drop_path_rate, num_layers)]
        self.layers = nn.ModuleList()
        for i in range(num_layers):
            self.layers.append(Block(embed_dim, num_heads, dpr[i]))

        self.ln_f = nn.LayerNorm(embed_dim)
        self.head = nn.Linear(embed_dim, num_vocab, bias=False)
        # prediction head
        self.increase_dim = nn.Sequential(
            nn.Conv1d(self.trans_dim, 3*(self.group_size), 1)
        )

    def forward(self, h, pos, classify=False):
        """
        Expect input as shape [sequence len, batch]
        If classify, return classification logits
        """
        batch, length, C = h.shape

        h = h.transpose(0, 1)
        pos = pos.transpose(0, 1)

        # prepend sos token
        sos = torch.ones(1, batch, self.embed_dim, device=h.device) * self.sos
        if not classify:
            h = torch.cat([sos, h[:-1, :, :]], axis=0)
        else:
            h = torch.cat([sos, h], axis=0)

        feature_list = []
        fetch_idx = [3, 7, 11]

        # transformer
        for i, layer in enumerate(self.layers):
            h = layer(h + pos, classify=classify)
            if i in fetch_idx:
                feature_list.append(h.transpose(0, 1)[:, 2:])

        h = self.ln_f(h)

        encoded_points = h.transpose(0, 1)

        return encoded_points, feature_list


class GPT2_generator(nn.Module):
    def __init__(
        self, embed_dim, num_heads, num_layers, num_positions, num_vocab, trans_dim, group_size, drop_path_rate
    ):
        super(GPT2_generator, self).__init__()

        self.embed_dim = embed_dim
        self.trans_dim = trans_dim
        self.group_size = group_size

        # start of sequence token
        self.sos = torch.nn.Parameter(torch.zeros(embed_dim))
        nn.init.normal_(self.sos)

        self.token_embeddings = nn.Embedding(num_vocab, embed_dim)
        # self.position_embeddings = nn.Embedding(num_positions, embed_dim)

        self.drop_path_rate = drop_path_rate

        dpr = [x.item() for x in torch.linspace(
            0, self.drop_path_rate, num_layers)]
        self.layers = nn.ModuleList()
        for i in range(num_layers):
            self.layers.append(Block(embed_dim, num_heads, dpr[i]))

        self.ln_f = nn.LayerNorm(embed_dim)
        self.head = nn.Linear(embed_dim, num_vocab, bias=False)
        # prediction head
        self.increase_dim = nn.Sequential(
            nn.Conv1d(self.trans_dim, 3*(self.group_size), 1)
        )

        self.cls_norm = nn.LayerNorm(self.trans_dim)

    def forward(self, h, pos, classify=False):
        """
        Expect input as shape [sequence len, batch]
        If classify, return classification logits
        """
        batch, length, C = h.shape

        h = h.transpose(0, 1)
        pos = pos.transpose(0, 1)

        # transformer
        for layer in self.layers:
            h = layer(h + pos, classify=classify)

        h = self.ln_f(h)

        if not classify:
            # return rebuild_points
            # remember to divide the rebuild_points into neighborhood and centers
            rebuild_points = self.increase_dim(h.transpose(1, 2)).transpose(
                1, 2).transpose(0, 1).reshape(batch * length, -1, 3)
            return rebuild_points

        rebuild_points = self.increase_dim(h.transpose(1, 2)).transpose(
            1, 2).transpose(0, 1).reshape(batch * length, -1, 3)

        # return classification logits and generative logits
        return rebuild_points
