import math

import torch
import torch.nn as nn
import torch.nn.functional as functional
import torch.nn.init as init


def _weights_init(m):
    if isinstance(m, nn.Linear) or isinstance(m, nn.Conv2d):
        init.kaiming_normal_(m.weight)


def deterministic_sample_gumbel(logits, eps=1e-20):
    U = 0.5 * torch.ones_like(logits)
    return -torch.log(-torch.log(U + eps) + eps)


def deterministic_gumbel_softmax(logits, tau=1):
    y = logits + deterministic_sample_gumbel(logits)
    return functional.softmax(y / tau, dim=-1)


class LambdaLayer(nn.Module):
    def __init__(self, lambd):
        super(LambdaLayer, self).__init__()
        self.lambd = lambd

    def forward(self, x):
        return self.lambd(x)


def conv3x3(in_planes, out_planes, stride=1):
    return nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride, padding=1, bias=False)


class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, in_planes, planes, stride=1):
        super(BasicBlock, self).__init__()
        self.conv1 = conv3x3(in_planes, planes, stride)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = conv3x3(planes, planes)
        self.bn2 = nn.BatchNorm2d(planes)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion * planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion * planes, kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(self.expansion * planes)
            )

    def forward(self, x):
        out = functional.relu(self.bn1(self.conv1(x)))
        out = self.bn2(self.conv2(out))
        out += self.shortcut(x)
        out = functional.relu(out)
        return out



class ResNet32(nn.Module):
    def __init__(
        self, num_classes=10, in_planes=64, block=BasicBlock, num_blocks=[2, 2, 2, 2]
    ):
        super(ResNet32, self).__init__()
        self.in_planes = in_planes

        self.conv1 = conv3x3(3, 64)
        self.bn1 = nn.BatchNorm2d(in_planes)
        self.layer1 = self._make_layer(block, in_planes, num_blocks[0], stride=1)
        self.layer2 = self._make_layer(block, in_planes * 2, num_blocks[1], stride=2)
        self.layer3 = self._make_layer(block, in_planes * 4, num_blocks[2], stride=2)
        self.layer4 = self._make_layer(block, in_planes * 8, num_blocks[3], stride=2)
        self.linear = nn.Linear(in_planes * 8, num_classes)

        self.linear_ema = nn.Linear(in_planes * 8, num_classes)
        self.linear_ema.weight.data.copy_(self.linear.weight.data)
        self.linear_ema.bias.data.copy_(self.linear.bias.data)

        self.apply(_weights_init)

    def _make_layer(self, block, planes, num_blocks, stride):
        strides = [stride] + [1] * (num_blocks - 1)
        layers = []
        for stride in strides:
            layers.append(block(self.in_planes, planes, stride))
            self.in_planes = planes * block.expansion
        return nn.Sequential(*layers)

    def forward(self, x):
        out = functional.relu(self.bn1(self.conv1(x)))
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.layer4(out)
        out = functional.avg_pool2d(out, 4)
        penultimate = out.view(out.size(0), -1)
        ema_out = self.linear_ema(penultimate)
        out = self.linear(penultimate)
        return out, ema_out

    def ema_update(self):
        alpha = 0.95
        curr_weight = self.linear.weight.data
        curr_bias = self.linear.bias.data
        self.linear_ema.weight.data.copy_(
            alpha * curr_weight + (1 - alpha) * self.linear.weight.data
        )
        self.linear_ema.bias.data.copy_(
            alpha * curr_bias + (1 - alpha) * self.linear.bias.data
        )


def calculate_tau(
    progress, max_val=10, min_val=0.1, default_val=1.0, strategy="linear"
):
    if strategy == "exponential":
        ratio = max_val / min_val
        tau = min_val * (ratio ** (1 - progress))
    elif strategy == "cosine":
        tau = max_val * math.cos(progress * math.pi / 2)
    elif strategy == "constant":
        tau = default_val
    else:
        tau = max_val
    tau = max(tau, min_val)
    return tau


class HiddenLayer(nn.Module):
    def __init__(self, input_size, output_size):
        super(HiddenLayer, self).__init__()
        self.fc = nn.Linear(input_size, output_size)
        self.relu = nn.ReLU()

    def forward(self, x):
        return self.relu(self.fc(x))


class MLP(nn.Module):
    def __init__(self, input_size=1, hidden_size=500, num_layers=1):
        super(MLP, self).__init__()
        self.first_hidden_layer = HiddenLayer(input_size, hidden_size)
        self.rest_hidden_layers = nn.Sequential(
            *[HiddenLayer(hidden_size, hidden_size) for _ in range(num_layers - 1)]
        )
        self.output_layer = nn.Linear(hidden_size, 2)

    def forward(self, x, test=False):
        x = self.first_hidden_layer(x)
        x = self.rest_hidden_layers(x)
        x = self.output_layer(x)
        if test:
            x = deterministic_gumbel_softmax(x, tau=0.5)
        else:
            x = functional.gumbel_softmax(x, tau=0.5, hard=True)
        return x[:, 0].unsqueeze(1)


class LinearCorrect(nn.Module):
    def __init__(
        self, input_size=64, num_classes=10, hidden_size=100, alpha=-2.0
    ):
        super(LinearCorrect, self).__init__()

        self.num_classes = num_classes
        self.classifier = nn.Linear(input_size, num_classes)
        self.gating_in = nn.Linear(3, hidden_size)
        self.gating_out = nn.Linear(hidden_size, 1)
        self.alpha = nn.Parameter(alpha * torch.ones(1))

    def forward(self, x, y=None, base_logit=None, ema_logit=None):
        logits = self.classifier(x)
        prob = functional.softmax(logits, dim=-1)
        if y is None:
            return logits

        base_prob = functional.softmax(base_logit, dim=-1)
        ema_prob = functional.softmax(ema_logit, dim=-1)
        y_prob = functional.one_hot(y, num_classes=self.num_classes)
        loss_gt = functional.cross_entropy(logits, y, reduction="none").detach()
        loss_base = torch.sum(-functional.log_softmax(logits, dim=-1) * base_prob, dim=-1).detach()
        loss_ema = torch.sum(-functional.log_softmax(logits, dim=-1) * ema_prob, dim=-1).detach()
        gate_input = torch.stack([loss_gt, loss_base, loss_ema], dim=-1)
        gate = functional.relu(self.gating_in(gate_input))
        alpha = torch.sigmoid(self.gating_out(gate)) * torch.sigmoid(self.alpha)
        out = alpha * prob + (1 - alpha) * y_prob
        return prob, out
