import numpy as np
import scipy.special as sp
from scipy.optimize import fsolve, minimize, bisect
from sympy import Symbol

def _eq(a,b):
    return abs(a-b) < 1e-15

def _leq(a,b):
    return a - b < 1e-15

def _geq(a,b):
    return b - a < 1e-15

class GGA(object):
    
    def __init__(self, nu, sigma, rho):
        self.nu    = nu
        self.sigma = sigma
        self.rho   = rho
            
    def __add__(self, other):
        """The addition operation (additive convolution).
        The sum of random variables with different rho
        is taken to be the heaviest tail. """

        if isinstance(other, Parameter):
            return other.__add__(self)
        if not isinstance(other, self.__class__):
            return self._copy()
        
        nu_1, sigma_1, rho_1 = self.params()
        nu_2, sigma_2, rho_2 = other.params()
        
        if _eq(rho_1, rho_2):
            rho = 0.5*(rho_1+rho_2)
            
            if _eq(rho, 1):
                # Exponential case
                return self._rv(nu_1 + nu_2 + 1, 
                                min(sigma_1, sigma_2), 1)
            elif rho > 1:
                # Superexponential case
                sigma  = sigma_1**(-1.0/(rho-1))
                sigma += sigma_2**(-1.0/(rho-1))
                sigma = sigma**(1 - rho)
                return self._rv(nu_1 + nu_2 + 1 - rho_1/2,
                                sigma, rho)
            
            else:
                # Subexponential case
                return max(self, other)._copy()
        
        else:
            return max(self, other)._copy()
        
    def __mul__(self, other):
        """The multiplication operation (multiplicative convolution)"""
        
        if isinstance(other, Parameter):
            return other.__mul__(self)
        if not isinstance(other, self.__class__):
            # Scalar multiplication
            return self._rv(self.nu, 
                            self.sigma / abs(other)**self.rho,
                            self.rho)
        
        nu_1, sigma_1, rho_1 = self.params()
        nu_2, sigma_2, rho_2 = other.params()
        
        if _leq(rho_1,0) and rho_2  > 0:
            return self._reg(abs(nu_1))
        if rho_1 > 0 and _leq(rho_2, 0):
            return self._reg(abs(nu_2))
        if _eq(rho_1,0) and _eq(rho_2,0):
            return self._reg(min(abs(nu_1),abs(nu_2)))
        
        mu = 1.0/abs(rho_1) + 1.0/abs(rho_2)
        sigma  = (sigma_1*abs(rho_1))**(1.0/(mu*abs(rho_1)))
        sigma *= (sigma_2*abs(rho_2))**(1.0/(mu*abs(rho_2)))
        sigma *= mu
        
        if rho_1 < 0:
            nu = nu_1/abs(rho_1) + nu_2/abs(rho_2) + 1/2
            return self._rv(nu/mu, sigma, -1/mu)
        else:
            nu = nu_1/rho_1 + nu_2/rho_2 - 1/2
            return self._rv(nu/mu, sigma, 1/mu)
        
    def __pow__(self, num):
        """The power operation"""
        try:
            if (num < 0) and (_eq(self.rho, 0) or _leq((self.nu+1)/self.rho, 0)):
                # Reciprocal by assuming bounded density near zero
                return self._reg(2.0)
        except TypeError:
            pass
        
        return self._rv((self.nu+1)/num - 1, self.sigma, self.rho/num)
    
    def __and__(self, other):
        """Product of densities operation"""
        if not isinstance(other, self.__class__):
            return self._copy()
        nu_1, sigma_1, rho_1 = self.params()
        nu_2, sigma_2, rho_2 = other.params()
        nu = nu_1 + nu_2
        if _eq(rho_1, rho_2):
            rho = 0.5*(rho_1 + rho_2)
            return self._rv(nu, sigma_1 + sigma_2, rho)
        if rho_1 <= 0 and rho_2 <= 0:
            if rho_1 > rho_2:
                return self._rv(nu, sigma_2, rho_2)
            else:
                return self._rv(nu, sigma_1, rho_1)
        else:
            if rho_1 > rho_2:
                return self._rv(nu, sigma_1, rho_1)
            else:
                return self._rv(nu, sigma_2, rho_2)
        
    __rand__ = __and__
        
    def __sub__(self, other):
        """The subtraction operation"""
        return self + (-1.0 * other)
    
    def __rsub__(self, other):
        """The subtraction operation"""
        return (-1.0 * self) + other
        
    def __truediv__(self, other):
        """The division operation (involving reciprocal)"""
        return self * (other ** (-1))
    
    def __rtruediv__(self, other):
        """Reciprocal operation"""
        return self ** (-1)
    
    def __abs__(self):
        """Absolute value operation"""
        return self._copy()
    
    def log(self):
        """The logarithm operation"""
        if self.nu < -1 and _leq(self.rho, 0):
            return self._rv(0.0, abs(self.nu)-1, 1.0)
        
        return self._light()
    
    def exp(self):
        """The exponential operation"""
        if _geq(self.rho, 1):
            return self._reg(self.sigma + 1)
        
        return self._heavy()
    
    def laplace(self):
        """Compute asymptotics for the Laplace transform of X^{-1}"""
        lap = GGA(1, 0, 0) & self._copy()
        lap = lap * GGA(0, 1, 1)
        return lap
    
        
    __radd__ = __add__
    __rmul__ = __mul__
    
    ### ORDERING ###
    
    def __ge__(self, other):
        """Comparing heaviness of the tail: does
        self have a heavier tail than other?"""
        nu_1, sigma_1, rho_1 =  self.params()
        nu_2, sigma_2, rho_2 = other.params()
        
        if _leq(rho_1, 0) and _leq(rho_2, 0):
            return (abs(nu_1) < abs(nu_2))
        
        if _eq(rho_1, rho_2):
            if _eq(sigma_1, sigma_2):
                return nu_1 > nu_2
            return sigma_1 < sigma_2
        
        return (rho_1 < rho_2)
        
    def __eq__(self, other):
        return _eq(self.nu,other.nu) and \
                _eq(self.sigma,other.sigma) and \
                _eq(self.rho,other.rho)
    
    def __gt__(self, other):
        return (self.__ge__(other)) and not (self.__eq__(other))
    
    def __ne__(self, other):
        return not (self.__eq__(other))
    
    def __lt__(self, other):
        return not (self.__ge__(other))
    
    def __le__(self, other):
        return other.__ge__(self)
    
    ### STRING REPRESENTATION ###
    
    def __str__(self):
        
        try:
            if _eq(self.nu, -1) and _leq(self.rho, 0):
                return "super heavy tail"
            if np.isinf(self.rho):
                return "super light tail"
        except TypeError:
            pass
        
        tail = "c"
        try:
            nu = '%s' % float('%.4g' % self.nu)
            if not _eq(self.nu, 0):
                if self.nu < 0:
                    tail += " x^({nu})".format(nu=nu)
                else:
                    tail += " x^{nu}".format(nu=nu)
        except TypeError:
            tail += " x^({nu})".format(nu=self.nu)
        try:
            if _eq(self.sigma, 1.0):
                sigma = ""
            else:
                sigma = '%s' % float('%.4g' % self.sigma)
                sigma += " * "
        except TypeError:
            sigma = "({sigma}) * ".format(sigma=str(self.sigma))
        try:
            if _eq(self.rho, 1.0):
                rho = ""
            else:
                rho = '%s' % float('%.4g' % self.rho)
                if self.rho < 0:
                    rho = "^({rho})".format(rho=rho)
                else:
                    rho = "^{rho}".format(rho=rho)
            if not _eq(self.rho, 0):
                tail += " exp(-{sigma}x{rho})".format(
                    sigma=sigma, rho=rho)
        except TypeError:
            tail += " exp(-{sigma}x^({rho}))".format(
                    sigma=sigma, rho=self.rho)
        
        return tail + '; ' + self.tail_type()
    
    def __repr__(self):
        return str(self)
    
    def tail_type(self):
        try:
            if self.rho >= 1:
                return 'light'
            if self.rho > 0:
                return 'subexp'
            if self.nu < -3:
                return 'powerlaw (finite var)'
            if self.nu < -2:
                return 'powerlaw (finite mean)'
            return 'powerlaw (infinite mean)'
        except TypeError:
            return ''
    
    ### HELPERS ###
    
    def _check(self, nu, sigma, rho):
        """Check if the parameters are sane"""
        positivity = None
        try:
            positivity = sigma > 0
        except TypeError:
            pass
        if positivity is not None:
            assert positivity
        if _leq(rho, 0):
            assert _leq(nu, 1.0)
            
    def params(self):
        """Return a tuple of the class parameters"""
        return (self.nu, self.sigma, self.rho)
            
    def _copy(self):
        """Return a copy of the current tail object"""
        return self.__class__(self.nu, self.sigma, self.rho)
                                  
    def _rv(self,nu,sigma,rho):
        """Creates a new random variable with given parameters"""
        return self.__class__(nu,sigma,rho)
                
    def _reg(self,nu):
        """Creates a new regularly-varying random variable with
        parameter nu"""
        return self.__class__(-nu,1,0)
    
    def _light(self):
        """Creates a new super-light random variable"""
        return self.__class__(0.0, 1.0, np.inf)
    
    def _heavy(self):
        """Creates a new super-heavy random variable"""
        return self._reg(1.0)
    
    def prior(self):
        return
    
    ### FOR ADDITION ###
    
    def crit_moment(self):
        """Returns the value a > 0 such that E[X^a] = 1"""
        nu, sigma, rho = self.params()
        if self.moment(1) >= 1:
            return 0.0
        lb = 1.0
        with np.errstate(all="ignore"):
            ub = 1.0
            while self.log_moment(ub) < 0:
                ub *= 2
            obj = lambda x: self.log_moment(x)
            sol = bisect(obj, lb, ub)
        return sol
    
    def crit_moment2(self):
        """Returns the value a > 0 such that E[X^a] = 2"""
        nu, sigma, rho = self.params()
        lb = 1.0
        with np.errstate(all="ignore"):
            ub = 1.0
            while self.log_moment(ub) < np.log(2):
                ub *= 2
            obj = lambda x: self.log_moment(x)-np.log(2)
            sol = bisect(obj, lb, ub)
        return sol
        
    def moment(self, r):
        """Returns the r-th moment under the generalized gamma
        assumption"""
        return self._gg_moments(self.params(), r)
    
    def log_moment(self, r):
        """Returns the r-th log moment under the generalized gamma
        assumption"""
        return self._gg_logmoments(self.params(), r)
    
    def _gg_logmoments(self, params, r):
        """Computes log expectation of E[X^r] for generalized gamma rv"""
        nu, sigma, rho = params
        return -r/rho*np.log(sigma) + sp.loggamma((nu+r+1)/rho)-sp.loggamma((nu+1)/rho)
    
    def _gg_moments(self, params, r):
        """Computes expectation of E[X^r] for generalized gamma rv"""
        return np.exp(self._gg_logmoments(params, r))

    def _gg_quantity1(self, params):
        """Computes E[X]^2 / (E[X^2] - E[X]^2)"""
        nu = params[0]
        rho = params[1]
        m1 = sp.gamma((nu+2)/rho)/sp.gamma((nu+1)/rho)
        m2 = sp.gamma((nu+3)/rho)/sp.gamma((nu+1)/rho)
        return m1**2 / (m2 - m1**2)

    def _gg_quantity2(self, params):
        """Computes E[X^2]^2 / (E[X^4] - E[X^2]^2)"""
        nu = params[0]
        rho = params[1]
        m2 = sp.gamma((nu+3)/rho)/sp.gamma((nu+1)/rho)
        m4 = sp.gamma((nu+5)/rho)/sp.gamma((nu+1)/rho)
        return m2**2 / (m4 - m2**2)

    def _gg_sum(self, params_1, params_2):
        """Uses moment matching to estimate the sum of two
        generalized gamma random variables by another generalized
        gamma random variable"""

        mu1_1 = self._gg_moments(params_1, 1)
        mu1_2 = self._gg_moments(params_1, 2)
        mu1_3 = self._gg_moments(params_1, 3)
        mu1_4 = self._gg_moments(params_1, 4)

        mu2_1 = self._gg_moments(params_2, 1)
        mu2_2 = self._gg_moments(params_2, 2)
        mu2_3 = self._gg_moments(params_2, 3)
        mu2_4 = self._gg_moments(params_2, 4)

        mu_1 = mu1_1 + mu2_1
        mu_2 = mu1_2 + 2*mu1_1*mu2_1 + mu2_2
        mu_4 = mu1_4 + 6*mu1_2*mu2_2 + mu2_4
        mu_4 += 4*mu1_3*mu2_1 + 4*mu1_1*mu2_3

        q1 = mu_1**2 / (mu_2 - mu_1**2)
        q2 = mu_2**2 / (mu_4 - mu_2**2)

        loss = lambda x: np.array([self._gg_quantity1(x)-q1, 
                                   self._gg_quantity2(x)-q2])
        sol = fsolve(loss, np.ones(2))

        nu = sol[0]
        rho = sol[1]
        sigma = (sp.gamma((nu+2)/rho) / sp.gamma((nu+1)/rho) / mu_1)**rho

        return nu, sigma, rho
    

RegVar = lambda nu: GGA(-nu, 1.0, 0.0)
SuperHeavy = lambda: RegVar(1.0)
SuperLight = lambda: GGA(0.0, 1.0, np.inf)
    
benktander2 = lambda a,b: GGA(2*b-2, a/b, b)
betaprime   = lambda a,b,*vars: RegVar(b+1)
burr        = lambda c,k: RegVar(c*k+1)
cauchy      = lambda mu=0,sigma=1: sigma**0.5 * RegVar(2.0) + mu
chi         = lambda k: GGA(k-1,0.5,2)
chi2        = lambda k: GGA(k/2-1,0.5,1)
dagum       = lambda a,*args: RegVar(a+1)
davis       = lambda n,b,*args: GGA(-1-n,b,1)
exponential = lambda lam: GGA(0,lam,1)
fdistn      = lambda d1,d2: RegVar(d2/2+1)
fisherz     = lambda d1,d2: GGA(0,d2,1)
frechet     = lambda a,lam,*args: GGA(-1-a,lam**a,-a)
gamma       = lambda a,b: GGA(a-1,b,1)
ggompertz   = lambda b,s: GGA(0,b*s,1)
hyperbolic  = lambda alpha,beta,lam,*args: GGA(lam-1,alpha-beta,1)
gennormal   = lambda alpha,beta,*args: GGA(0, alpha**(-beta),beta)
geostable   = lambda alpha: RegVar(alpha+1)
gompertz    = lambda *args: SuperLight()
gumbel      = lambda beta,*args: GGA(0, 1.0/beta, 1)
gumbel2     = lambda alpha,beta: GGA(-alpha-1,beta,-alpha)
holtsmark   = lambda: RegVar(5/2)
hypsecant   = lambda: GGA(0,np.pi/2,1)
invchi2     = lambda k: GGA(-k/2-1,1/2,-1)
invgamma    = lambda alpha,beta: GGA(-alpha-1,beta,-1)
invnormal   = lambda *args: GGA(-2,1/2,-2)
levy        = lambda c,*args: GGA(-3/2,c/2,-1)
laplace     = lambda lam: GGA(0,1/lam,1)
logistic    = lambda lam: GGA(0,1/lam,1)
logcauchy   = lambda *args: SuperHeavy()
loglaplace  = lambda lam,*args: RegVar(1/lam+1)
loglogistic = lambda alpha,beta: RegVar(beta+1)
logt        = lambda *args: SuperHeavy()
lomax       = lambda alpha, *args: RegVar(alpha+1)
maxboltz    = lambda sigma: GGA(2, 1/(2*sigma**2), 2)
normal      = lambda mu=0,sigma2=1: sigma2**0.5*GGA(0, 1/2, 2) + mu
pareto      = lambda alpha, *args: RegVar(alpha+1)
rayleigh    = lambda sigma: GGA(1, 1/(2*sigma**2), 2)
rice        = lambda sigma, *args: GGA(1/2, 1/(2*sigma**2), 2)
skewnormal  = lambda mu,sigma: GGA(0, 1/(2*sigma**2), 2)
slash       = lambda: GGA(-2,1/2,2)
stable      = lambda alpha: RegVar(alpha+1)
student     = lambda nu,mu=0,sigma=1: RegVar(nu+1)*sigma**0.5 + mu
tracywidom  = lambda beta: (-3*beta/4 - 1, 2*beta/3, 3/2)
voigt       = lambda *args: RegVar(2)
weibull     = lambda rho, lam: GGA(rho-1, 1/lam**rho, rho)

gga_distributions = ['benktander2', 'betaprime', 'burr', 'cauchy', 'chi', 'chi2', 'dagum',
    'davis', 'exponential', 'fdistn', 'fisherz', 'frechet', 'gamma',
    'ggompertz', 'hyperbolic', 'gennormal', 'geostable', 'gompertz',
    'gumbel', 'gumbel2', 'holtsmark', 'hypsecant', 'invchi2', 'invgamma',
    'invnormal', 'levy', 'laplace', 'logistic', 'logcauchy', 'loglaplace',
    'loglogistic', 'logt', 'lomax', 'maxboltz', 'normal', 'pareto',
    'rayleigh', 'rice', 'skewnormal', 'slash', 'stable', 'student', 'tracywidom',
    'voigt', 'weibull']

gga_functions = ['exp', 'log', 'relu', 'sin', 'cos', 'sinh', 'cosh', 'tanh',
    'arcsinh', 'arccosh', 'sqrt', 'erf']
    
############# FUNCTIONS #############
    
class TailFunction(object):
    
    def __init__(self, func, num_args):
        self.func = func
        self.num_args = num_args
        assert num_args > 0
        
    def filter_args(self, args):
        assert len(args) == self.num_args
        list_vars = list(args)
        idx = 0
        for var in list_vars:
            if not isinstance(var, GGA):
                list_vars.pop(idx)
            else:
                idx += 1
        return list_vars
        
class PowerFunction(TailFunction):
    
    def __init__(self, alpha, func, num_args, const = 1.0):
        super().__init__(func, num_args)
        self.alpha = alpha
        self.const = const
            
    def __call__(self, *args):
        list_vars = self.filter_args(args)
        if len(list_vars) == 0:
            return self.func(*args)
        else:
            return self.const * max(list_vars)**self.alpha
        
class BoundedFunction(TailFunction):
    
    def __init__(self, bound, func, num_args):
        super().__init__(func, num_args)
        self.bound = bound
        
    def __call__(self, *args):
        list_vars = self.filter_args(args)
        if len(list_vars) == 0:
            return self.func(*args)
        else:
            return self.bound
        
class BoundedFunction(TailFunction):
    
    def __init__(self, bound, func, num_args):
        super().__init__(func, num_args)
        self.bound = bound
        
    def __call__(self, *args):
        list_vars = self.filter_args(args)
        if len(list_vars) == 0:
            return self.func(*args)
        else:
            return self.bound
        
class ExpFunction(TailFunction):
    def __init__(self, scale, exponent, func, num_args):
        super().__init__(func, num_args)
        self.scale = scale
        self.exponent = exponent
        
    def __call__(self, *args):
        list_vars = self.filter_args(args)
        if len(list_vars) == 0:
            return self.func(*args)
        else:
            return (max(list_vars)**self.exponent * self.scale).exp()
        
class LogFunction(TailFunction):
    def __init__(self, scale, func, num_args):
        super().__init__(func, num_args)
        self.scale = scale
        
    def __call__(self, *args):
        list_vars = self.filter_args(args)
        if len(list_vars) == 0:
            return self.func(*args)
        else:
            return (max(list_vars).log() * self.scale)
        
LipschitzFunction = lambda func, num_args, const = 1.0: \
                        PowerFunction(1.0, func, num_args, const)
    
exp = ExpFunction(1.0, 1.0, np.exp, 1)
log = LogFunction(1.0, np.log, 1)
relu = LipschitzFunction(lambda x: max([x,0]), 1, 1.0)
sin = BoundedFunction(1.0, np.sin, 1)
cos = BoundedFunction(1.0, np.cos, 1)
sinh = ExpFunction(0.5, 1.0, np.sinh, 1)
cosh = ExpFunction(0.5, 1.0, np.cosh, 1)
tanh = BoundedFunction(np.pi / 2, np.tanh, 1)
arcsinh = LogFunction(1.0, np.arcsinh, 1)
arccosh = LogFunction(1.0, np.arccosh, 1)
sqrt = PowerFunction(0.5, np.sqrt, 1, 1.0)
erf = BoundedFunction(1.0, sp.erf, 1)

############# MATRICES #############

def MatGGA(size, distn):
    matrix = np.empty(size, dtype=object)
    for idx in range(size[0]):
        for idy in range(size[1]):
            matrix[idx,idy] = GGA(*distn.params())
    return matrix

gauss_ens = lambda *size: MatGGA(size, normal(0,1))

############# PARAMETERS #############

class Model(object):
    # Initialize the Conditioner with an empty list of parameters, 
    # an index of 0, and an empty dictionary of tails.
    def __init__(self, marginals=False):
        self.parameters = []
        self.idx = 0
        self.tails = {}
        self.marginals = marginals
        
    # Define the parameter method which creates a new Parameter object 
    # with the given symbol string and adds it to the list of parameters.
    def parameter(self, symbol_str, positive=False):
        param = Parameter(symbol_str, marginals = self.marginals, positive=positive)
        self.parameters.append(param)
        param.active = False
        self.tails[symbol_str] = None
        return param
    
    def __iter__(self):
        return self
    
    def __next__(self):
        return self.next()
    
    # Define the next method which returns the next element of the Conditioner. 
    # It increments the index and activates the corresponding parameter.
    def next(self):
        if self.idx < len(self.parameters):
            cur, self.idx = self.idx, self.idx + 1
            for idy, param in enumerate(self.parameters):
                if idy == cur:
                    param.active = True
                else:
                    param.active = False
            return cur
        raise StopIteration()
        
    def proceed(self):
        for param in self.parameters:
            if param.active:
                param.proceed()
    
    # Define the finalize method which iterates over the parameters and sets the 
    # tail of the active parameters to their computed value.
    def finalize(self):
        for param in self.parameters:
            if param.active:
                self.tails[param.symbol_str] = param.comp()
                
    def __str__(self):
        output = ''
        for tail in self.tails:
            output += tail + ': ' + str(self.tails[tail]) + '\n'
        return output
    
    def __repr__(self):
        return str(self)

    
class Parameter(object):

    def __init__(self, symbol_str, prior=None, marginals=False, positive=False):
        self.distns = []        
        self.ops = []
        # Create a Symbol object with the specified symbol string.
        if positive:
            self.symbol = Symbol(symbol_str, positive=True)
        else:
            self.symbol = Symbol(symbol_str)
        # Store the symbol string for later use in the __str__ and __repr__ methods.
        self.symbol_str = symbol_str
        if prior is not None:
            self.prior(prior)
        self.active = True
        self.marginals = marginals
        
    def prior(self, prior):
        self.prior_d = prior
        #if self.active:
        self.distns = [prior]
        
    def set_active(self, active):
        self.active = active
        if not active:
            self.distns = []
            self.ops = []
        
    def __add__(self, other):
        # Add addition to the list of operations.
        if not self.active:
            if isinstance(other, Parameter):
                return other.__add__(self)
            if self.marginals:
                return self.prior_d + other
            else:
                return self.symbol + other
        self.ops.append(('add',other))
        return self
    
    def __sub__(self, other):
        return self.__add__(-1.0 * other)
    
    def __rsub__(self, other):
        return (-1.0 * self).__add__(other)

    def __pow__(self, num):
        # Add power to the list of operations.
        if not self.active:
            if isinstance(num, Parameter):
                return other.__pow__(self)
            if self.marginals:
                return self.prior_d ** num
            else:
                return self.symbol**num
        self.ops.append(('pow',num))
        return self
        
    def __mul__(self, other):
        # Add multiplication to the list of operations.
        if not self.active:
            if isinstance(other, Parameter):
                return other.__mul__(self)
            if self.marginals:
                return self.prior_d * other
            else:
                return self.symbol*other
        if isinstance(other, int) or isinstance(other, float):
            if other == 0:
                return 0
        self.ops.append(('mul',other))
        return self
    
    def __truediv__(self, other):
        return self.__mul__(other**(-1))
    
    def __rtruediv__(self, other):
        return (self.__pow__(-1)).__mul__(other)
    
    def __abs__(self):
        return self
    
    def proceed(self):
        if not self.active:
            return
        # Initialize the distribution to 1.0.
        distn = 1.0
        # If there are no operations, return without doing anything.
        if len(self.ops) == 0:
            return
        # Iterate over the operations in reverse order.
        for op in self.ops[::-1]:
            name, other = op
            if name == 'add':
                distn = distn - other
            elif name == 'pow':
                # Mimic the change of variables removing the correction factor
                distn = distn ** (1./other) & GGA(1 - other, 0, 0)
            elif name == 'mul':
                if isinstance(other, GGA):
                    distn = distn / other & GGA(1.0, 0, 0)
                else:
                    distn = distn / other
        # Add the resulting distribution to the list of distributions.
        self.distns.append(distn)
        self.ops = []
        
    __rmul__ = __mul__
    __radd__ = __add__
    
    def comp(self):
        if not self.active:
            return
        rv = 1.0
        for distn in self.distns:
            # Conjoin the current distribution with the resulting distribution.
            rv = rv & distn
        return rv
    
    def __str__(self):
        # Return a string representation of the parameter object, 
        # including the symbol string and the resulting distribution.
        return self.symbol_str + ': ' + str(self.comp())
    
    def __repr__(self):
        return str(self)


