\name{PLNet_main}
\alias{PLNet_main}
\title{Estimate the sparse precision matrix for PLN model.}
\usage{
PLNet_main(
  obs_mat,
  Sd_est = "TSS",
  lambda_vec = NULL,
  n_lambda = 100,
  penalize.diagonal = TRUE,
  cov_input = NULL
  weight_mat,
  zero_mat,
  core_num = 1
)
}
\description{
Estimate the precision matrix for PLN model by estimating the precision matrix using the dtrace loss.
}

\arguments{

\item{obs_mat}{An abundance count matrix with Cell rows and Gene columns.}

\item{Sd_est}{Optional. Normalisation scheme used to compute scaling factors used as offset during PLN inference. Available schemes are "TSS" (Total Sum Scaling, default), "CSS" (Cumulative Sum Scaling, used in metagenomeSeq), "GMPR" (Geometric Mean of Pairwise Ratio, introduced in Chen et al., 2018) or "none". Alternatively the user can supply its own vector or matrix of offsets (see note for specification of the user-supplied offsets).}

\item{lambda_vec}{Optional. The user can supply its own vector.}

\item{n_lambda}{Optional. When the lambda_vec is empty, the package will provide a suitable lambda_vec with length n_lambda.}

\item{penalize.diagonal}{Should diagonal of inverse covariance be penalized? Dafault TRUE.}

\item{cov_input}{The optional estimation of covariance matrix instead of default moment estimation. Dafault NULL.}

\item{weight_mat}{A non-negative weighted matrix with p rows and p columns}

\item{zero_mat}{A matrix with p rows and p columns that contain element 0 or 1. 1 represents the zero support.}

\item{core_num}{The number of cores for parallel computation. Default 1.}

}

\value{

A list which contain the following result:

\item{Omega_est}{A list which contain the precision matrix for different lambda value.}

\item{mean_est}{The vector of estimated expected level.}

\item{lambda_vec}{The lambda_vec used.}

\item{S_depth}{The vector of estimated library size.}

\item{BIC_vec}{The vector of BIC value.}

\item{Omega_chooseB}{The estimated precision matrix choosed by the BIC criterion.}

\item{AIC_vec}{The vector of AIC value.}

\item{Omega_chooseA}{The estimated precision matrix choosed by the AIC criterion.}

\item{time_sigmamoment}{The time for calculating the covariance matrix for PLN using moment method.}

\item{time_EQUAL}{The time for calculating the path of precision matrix using EQUAL package.}

}

\references{
Zhang T, Zou H. Sparse precision matrix estimation via lasso penalized D-trace loss[J]. Biometrika, 2014, 101(1): 103-120.
}
\examples{

##
library(MASS)
library(CVXR)
library(orthopolynom)

##1. Run the Moment based PLNet (With argument "cov_input = NULL").
##-----------------------------------------------------------------------------------------------
#default version
PLNet_res<-PLNet::PLNet_main(obs_mat = PLNet::obs_mat,
                             Sd_est = "TSS",lambda_vec = NULL , n_lambda =100 , penalize.diagonal = TRUE,
                             weight_mat = NULL,zero_mat = NULL,core_num = 1,
                             cov_input = NULL)

#weighted version
weight_mat<-matrix(1,nrow = dim(PLNet::obs_mat)[2],ncol = dim(PLNet::obs_mat)[2])
weight_mat[1:(dim(PLNet::obs_mat)[2]/2),1:(dim(PLNet::obs_mat)[2]/2)]<-0.3

PLNet_res_weight<-PLNet::PLNet_main(obs_mat = PLNet::obs_mat,
                             Sd_est = "TSS",lambda_vec = NULL , n_lambda =100 , penalize.diagonal = TRUE,
                             weight_mat = weight_mat,zero_mat = NULL,core_num = 1,
                             cov_input = NULL)

##zero support version
zero_mat<-matrix(0,nrow = dim(PLNet::obs_mat)[2],ncol = dim(PLNet::obs_mat)[2])
zero_mat[upper.tri(zero_mat)][sample(1:length(zero_mat[upper.tri(zero_mat)]),100)]<-1
zero_mat<-zero_mat + t(zero_mat)

PLNet_res_zero<-PLNet::PLNet_main(obs_mat = PLNet::obs_mat,
                                    Sd_est = "TSS",lambda_vec = NULL , n_lambda =100, penalize.diagonal = TRUE,
                                    weight_mat = NULL,zero_mat = zero_mat,core_num = 1,
                                    cov_input = NULL)
##-----------------------------------------------------------------------------------------------

##2. Run the MLE based PLNet (With specification of argument "cov_input").
##-----------------------------------------------------------------------------------------------
cov_input<-mle_newton(data_use = PLNet::obs_mat,
                      S_depth = rowSums(PLNet::obs_mat),
                      k_max = 10,
                      core_num = 1)

PLNet_res<-PLNet::PLNet_main(obs_mat = PLNet::obs_mat,
                             Sd_est = "TSS",lambda_vec = NULL , n_lambda =100 , penalize.diagonal = TRUE,
                             weight_mat = NULL,zero_mat = NULL,core_num = 1,
                             cov_input = cov_input$mlesigmahat[[length(cov_input$mlesigmahat)]])
##-----------------------------------------------------------------------------------------------

}
