"""Utilities related to dealing with zonotope vertices."""
import numpy as np

from xoid.util import basics

_np = basics.to_np


def inactive_examples_count(vertex):
    return np.sum(np.all(1 - vertex, axis=0).astype(np.int32))


def to_vertex_key(vertex):
    return tuple(tuple(v) for v in _np(vertex))


def flip_activations(vertex, unit_inds, ex_inds):
    # Returns a copy.
    flipped = np.copy(vertex)
    flipped[unit_inds, ex_inds] = 1 - vertex[unit_inds, ex_inds]
    return flipped


def boundary_constraints_to_indices(bcs):
    # Also removes constraints corresponding to an entire unit.
    # 
    # Sometimes we get a unit with all the example indices being binding
    # constraints. I think this means the unit has w = b = 0. Anyways, those
    # don't really help with finiding the next neighbor and slow us down with
    # dummy choices, so get rid of them.
    N = bcs.shape[-1]
    unit_inds, ex_inds = np.nonzero(bcs)

    mask = np.ones(unit_inds.shape, dtype=np.bool)
    units, ex_counts = np.unique(unit_inds, return_counts=True)
    for unit, ex_count in zip(units, ex_counts):
        if ex_count == N:
            mask &= (unit_inds != unit)

    return unit_inds[mask], ex_inds[mask]


def random_vertex(X: np.ndarray, m: int):
    N, d = X.shape
    W = np.random.normal(size=[m, d])
    return (X @ W.T > 0).astype(X.dtype).T
