"""Solvers for seeing if a vertex is feasible."""
import cvxpy as cp
import numpy as np

from xoid.util import basics
from xoid.util import numerics

_np = basics.to_np
about_equal = numerics.about_equal


class FeasibilityChecker:
    """Checks whether a given activation pattern is feasible for a dataset."""

    def __init__(self, X, eps=1e-6):
        self.X = _np(X)
        self.eps = eps

        self.ap = cp.Parameter([X.shape[0]])

        self.w = cp.Variable([X.shape[-1]])
        self.b = cp.Variable([])

        self._set_up_problem()

    def _set_up_problem(self):
        self.objective = cp.Minimize(cp.sum(0 * self.w))

        h = self.X @ self.w + self.b
        pm_vertex = 2 * self.ap - 1
        self.constraints = [cp.multiply(pm_vertex, h) >= 0]
        
        self.prob = cp.Problem(self.objective, self.constraints)

    def is_feasible(self, ap, unit_index=None):
        del unit_index
        self.ap.value = _np(ap)
        self.prob.solve(warm_start=True)

        w = self.w.value
        b = self.b.value

        if w is None or b is None:
            return False

        if about_equal(w, 0.0, self.eps).all() and about_equal(b, 0.0, self.eps).all():
            return False

        return True
