#include <iostream>
#include <fstream>
#include <string>
#include <sstream>
#include <pthread.h>
#include <vector>
#include <set>
#include <map>
#include <dirent.h>
#include <algorithm>

struct GameParms{
    unsigned numPeg,numColor,result=0;
    std::string display,error,gameFile,strategy;
    double lieLB,lieUB;
};

static void
split( const std::string& line,
       std::vector<std::string>& words )
{
    words.clear();
    std::stringstream ss( line );
    while( ss.good() ){
        std::string buf;
        ss >> buf;
        if( buf.find_first_not_of(' ') != std::string::npos ) words.push_back( buf );
    }
}

static void
read_code( std::string& input,
           std::vector<unsigned>& output )
{
    output.clear();
    size_t pos = 0;
    while(1){
        size_t p = input.find("_",pos);
        if( p == std::string::npos ){
            unsigned value = std::stoi(input.substr(pos));
            output.push_back(value);
            break;
        }
        unsigned value = std::stoi(input.substr(pos,p-pos));
        output.push_back(value);
        pos = p+1;
    }
}

static void
collect( std::vector<unsigned>& buf,
         unsigned pos,
         unsigned numColor,
         std::vector<std::vector<unsigned> >& storage )
{
    if( pos == buf.size()-1 ){
        for( unsigned i=0; i<numColor; i++ ){
            buf[pos] = i;
            storage.push_back( buf );
        }
    }
    else{
        for( unsigned i=0; i<numColor; i++ ){
            buf[pos] = i;
            collect( buf, pos+1, numColor, storage );
        }
    }
}

static void
get_all_codes( const unsigned numPeg,
               const unsigned numColor,
               std::vector<std::vector<unsigned> >& storage )
{
    storage.clear();
    std::vector<unsigned> buf( numPeg );
    collect( buf, 0, numColor, storage );
}

static std::string
code_to_string( const std::vector<unsigned>& code )
{
    std::string buf = std::to_string(code[0]);
    for( unsigned i=1; i<code.size(); i++ ) buf += "_" + std::to_string(code[i]);
    return buf;
}

static std::pair<unsigned,unsigned>
get_feedback( const std::vector<unsigned>&  guess,
              const std::vector<unsigned>&  code )
{
    unsigned size = guess.size();
    std::pair<unsigned,unsigned> feedback;
    feedback.first = 0;
    feedback.second = 0;
    std::set<unsigned> colors;
    for( unsigned i=0; i<size; i++ ){
        if( guess[i] == code[i] ) feedback.first++;
        if( colors.find( guess[i] ) != colors.end()) continue;
        unsigned c1=0, c2=0;
        for( unsigned ii=0; ii<size; ii++ ){
            if( guess[ii] == guess[i] ) c1++;
            if(  code[ii] == guess[i] ) c2++;
        }
        colors.insert( guess[i] );
        feedback.second += std::min(c1,c2);
    }
    feedback.second -= feedback.first;
    return feedback;
}

static void
calc_corners( unsigned numG,
              GameParms* parms,
              std::vector<double>& corner )
{
    std::stringstream ss;
    ss << "from pyomo.environ import *\n"
       << "infinity = float('inf')\n"
       << "model = AbstractModel()\n"
       << "T = [0";
    for( unsigned i=1; i<numG; i++ ) ss << "," << i;
    ss << "]\n"
       << "model.p = Var(T, within=NonNegativeReals)\n";
    unsigned counter = 1;
    for( unsigned i=0; i<numG; i++ ){
        ss << "\n"
           << "def ineq" << counter << "_rule(model):\n"
           << "    return model.p[" << i << "] >= " << parms->lieLB << "\n"
           << "model.ineq" << counter << " = Constraint(rule=ineq" << counter << "_rule)\n";
        counter++;
        ss << "\n"
           << "def ineq" << counter << "_rule(model):\n"
           << "    return model.p[" << i << "] <= " << parms->lieUB << "\n"
           << "model.ineq" << counter << " = Constraint(rule=ineq" << counter << "_rule)\n";
        counter++;
    }
    std::string filename = parms->gameFile + ".prob.clues";
    std::ifstream is( filename.c_str());
    std::string buffer;
    while(is.good()){
        std::getline( is, buffer );
        std::vector<std::string> words;
        split( buffer, words );
        if( words.size() != 5 ) continue;
        if( words[1] == "and" ){
            ss << "\n"
               << "def ineq" << counter << "_rule(model):\n"
               << "    return model.p[" << words[0] << "] * " << "model.p[" << words[2] << "] >= " << words[3] << "\n"
               << "model.ineq" << counter << " = Constraint(rule=ineq" << counter << "_rule)\n";
            counter++;
            ss << "\n"
               << "def ineq" << counter << "_rule(model):\n"
               << "    return model.p[" << words[0] << "] * " << "model.p[" << words[2] << "] <= " << words[4] << "\n"
               << "model.ineq" << counter << " = Constraint(rule=ineq" << counter << "_rule)\n";
            counter++;
        }
        else{
            ss << "\n"
               << "def ineq" << counter << "_rule(model):\n"
               << "    return model.p[" << words[0] << "] + " << "model.p[" << words[2] << "] - "
               << "model.p[" << words[0] << "] * " << "model.p[" << words[2] << "] >= " << words[3] << "\n"
               << "model.ineq" << counter << " = Constraint(rule=ineq" << counter << "_rule)\n";
            counter++;
            ss << "\n"
               << "def ineq" << counter << "_rule(model):\n"
               << "    return model.p[" << words[0] << "] + " << "model.p[" << words[2] << "] - "
               << "model.p[" << words[0] << "] * " << "model.p[" << words[2] << "] <= " << words[4] << "\n"
               << "model.ineq" << counter << " = Constraint(rule=ineq" << counter << "_rule)\n";
            counter++;
        }
    }
    is.close();    
    std::string cons = ss.str();    
    filename =  parms->gameFile + ".maxent.py";
    std::ofstream os( filename.c_str() );
    os << cons << "\n" << "def cost_rule(model):\n"
       << "    return"
       << " model.p[0] * pyomo.environ.log(model.p[0]) + (1-model.p[0]) * pyomo.environ.log(1-model.p[0])";
    for( unsigned i=1; i<numG; i++ ){
        os << " + model.p[" << i << "] * pyomo.environ.log(model.p[" << i << "]) + (1-model.p["
           << i << "]) * pyomo.environ.log(1-model.p[" << i << "])";
    }
    os << "\n" << "model.cost = Objective(rule=cost_rule, sense=minimize)\n\n";
    os << "instance=model.create_instance()\n"
       << "opt = SolverFactory('ipopt')\n"
       << "opt.solve(instance, load_solutions=True, tee=True)\n\n"
       << "f = open('" << filename << ".solution','w')\n"
       << "for i in T: f.write(repr(instance.p[i].value)+' ')\n"
       << "f.close()\n";
    os.close();
    std::string cmd = std::string("python3 ") + filename + " > " + filename + ".out";
    system( cmd.c_str() );
    filename += ".solution";
    is.open( filename.c_str());
    corner.resize( numG );
    for( unsigned i=0; i<numG; i++ ) is >> corner[i];
    is.close();
}

static void*
game_thread( void* arg )
{
    GameParms* parms = (GameParms*)arg;
    try{
        std::stringstream ss;
        ss << "Game: " << parms->gameFile << "\n";
        std::ifstream is( parms->gameFile.c_str());
        std::string buffer;
        std::getline( is, buffer );
        std::getline( is, buffer );
        std::vector<std::string> words;
        std::vector<std::vector<unsigned> > guesses;
        std::vector<std::pair<unsigned,unsigned> > feedbacks;
        while(1){
            std::getline( is, buffer );
            if( buffer.find("correct") != std::string::npos ) break;
            split( buffer, words );
            std::vector<unsigned> buf;
            read_code( words[0], buf );
            guesses.push_back( buf );
            read_code( words[1], buf );
            feedbacks.push_back( std::pair<unsigned,unsigned>( buf[0], buf[1] ));
        }
        std::getline( is, buffer );
        std::getline( is, buffer );
        split( buffer, words );
        std::set<std::vector<unsigned> > answers;
        for( unsigned i=0; i<std::stoi(words[2]); i++ ){
            std::vector<unsigned> answer;
            read_code( words[3+i], answer );
            answers.insert( answer );
        }
        is.close();
        std::vector<std::vector<unsigned> > all;
        get_all_codes( parms->numPeg, parms->numColor, all );
        std::vector<unsigned> lieCounts(guesses.size());
        for( unsigned ii=0; ii<guesses.size(); ii++ ){
            std::set<std::pair<unsigned,unsigned> > lies;
            for( unsigned i=0; i<all.size(); i++ ){
                if( guesses[ii] == all[i] ) continue;
                std::pair<unsigned,unsigned> f = get_feedback( guesses[ii], all[i] );
                lies.insert( f );
            }
            lieCounts[ii] = lies.size()-1;
        }
        unsigned numG = guesses.size();
        std::vector<double> corner;
        calc_corners( numG, parms, corner );
        double bestP = 0;
        unsigned bestI;
        for( unsigned i=0; i<all.size(); i++ ){
            double p = 1;
            for( unsigned ii=0; ii<guesses.size(); ii++ ){
                std::pair<unsigned,unsigned> correct = get_feedback( guesses[ii], all[i] );
                if( correct == feedbacks[ii] ) p *= 1-corner[ii];
                else p *= corner[ii]/double(lieCounts[ii]);
                if( p<bestP ) break;
            }
            if( p>bestP ){
                bestP = p;
                bestI = i;
            }
        }
        ss << "guess " << code_to_string( all[bestI] ) << "\n";
        if( answers.find(all[bestI]) != answers.end() ){
            parms->result = 1;
            ss << "correct\n";
        }
        else ss << "wrong\n";
        parms->display = ss.str();
    }
    catch( std::exception& e ){
        parms->error = e.what();
    }
    return NULL;
}

static std::vector<double>
read_probs( const std::string& filename )
{
    std::vector<double> probs;
    std::ifstream is( filename.c_str() );
    while( is.good() ){
        double buf;
        is >> buf;
        probs.push_back( buf );
    }
    is.close();
    return probs;
}

int
main( int argc, const char* argv[] )
{
    int numPeg   = 4;
    int numColor = 6;
    int position = 1;
    std::string gameType( "Bayesian" );
    std::string configFile;
    std::string inputDir;
    std::string strategy;
    unsigned begin=0, batch=10000;
    while( position < argc ){
        if( argv[position][0] != '-' ){
            std::cout << "Wrong usage: invalue argument \""
                      << argv[position] << "\".\n";
            return 1;
        }
        std::string buf( argv[position] );
        if(( buf == "-numPeg" )&&( argc > position+1 )){
            if(( sscanf( argv[position+1], "%d", &numPeg) < 1 )||( numPeg <= 0 )){
                std::cout << "Wrong usage: invalue argument \""
                          << argv[position] << " "
                          << argv[position+1] << "\".\n";
                return 1;
            }
            position += 2;
        }
        else if(( buf == "-numColor" )&&( argc > position+1 )){
            if(( sscanf( argv[position+1], "%d", &numColor) < 1 )||( numColor <= 1 )){
                std::cout << "Wrong usage: invalue argument \""
                          << argv[position] << " "
                          << argv[position+1] << "\".\n";
                return 1;
            }
            position += 2;
        }
        else if(( buf == "-gameConfig" )&&( argc > position+2 )){
            gameType  = std::string( argv[position+1] );
            configFile= std::string( argv[position+2] );
            position += 3;
        }
        else if(( buf == "-inputDir" )&&( argc > position+1 )){
            inputDir  = std::string( argv[position+1] );
            position += 2;
        }
        else if(( buf == "-strategy" )&&( argc > position+1 )){
            strategy  = std::string( argv[position+1] );
            position += 2;
        }
        else if(( buf == "-batch" )&&( argc > position+2 )){
            sscanf( argv[position+1], "%d", &begin);
            sscanf( argv[position+2], "%d", &batch);
            position += 3;
        }
        else{
            std::cout << "Wrong usage: invalid argument \""
                      << argv[position] << "\".\n";
            return 1;
        }
    }
    std::ifstream is( configFile.c_str());
    if( !is.good()){
        std::cout << "invalid config file\n";
        return 1;
    }
    double lieLB=0,lieUB=0;
    while( is.good()){
        std::string buffer;
        std::getline( is, buffer );
        std::vector<std::string> words;
        split( buffer, words );
        if(( gameType == "credal" )||( gameType == "credalplus" )){
            if(( words.size() == 3 )&&( words[0] == "lie_prob" )){
                lieLB = std::stod( words[1] );
                lieUB = std::stod( words[2] );
            }
        }
        else{
            std::cout << "invalid game type\n";
            return 1;
        }
    }
    is.close();
    DIR *dir = opendir(inputDir.c_str());
    struct dirent *ent;
    std::vector<std::string> gamefiles;
    while((ent = readdir(dir)) != NULL){
        std::string buf = ent->d_name;
        if( buf.find("game_") == std::string::npos ) continue;
        if( buf.find("prob") != std::string::npos ) continue;
        if( buf.find(".py") != std::string::npos ) continue;
        gamefiles.push_back( inputDir + "/" + buf );
    }
    std::sort( gamefiles.begin(), gamefiles.end());
    closedir(dir);
    unsigned numGame = gamefiles.size()-begin;
    if( numGame > batch ) numGame = batch;
    std::cout << "solving " << numGame << " puzzles with " << numPeg
              << " pegs and " << numColor << " colors.\n";
    time_t start,end;
    time(&start);
    std::vector<GameParms> parms( numGame );
    std::vector<pthread_t> threads( numGame );
    unsigned correct = 0;
    for( unsigned i=0; i<numGame; i++ ){
        parms[i].numPeg = numPeg;
        parms[i].numColor = numColor;
        parms[i].gameFile = gamefiles[begin+i];
        parms[i].lieLB = lieLB;
        parms[i].lieUB = lieUB;
        parms[i].strategy = strategy;
        pthread_create( &(threads[i]), NULL, game_thread, &(parms[i]));
    }
    for( unsigned i=0; i<numGame; i++ ){
        pthread_join( threads[i], NULL );
        if( !parms[i].error.empty() ){
            std::cout << "Error: " << parms[i].error << "\n";
            return 1;
        }
        std::cout << parms[i].display;
        correct += parms[i].result;
    }
    time(&end);
    std::cout << "Success: " << correct << "/" << numGame << "\n"
              << "Runtime: " << difftime(end,start) << " seconds\n";
    return 0;
}

