import os
import numpy as np
from shutil import rmtree


class Experiment:

    def __init__(self, exp_dir: str = './Results', seed: int = 0, name: str = 'BaseExp', **kwargs) -> None:
        """
        :param exp_dir: Path to experiment directory
        :param seed: Seed to be used for the randomizer
        :param name: Name of the experiment
        """
        self.config = dict()
        self.config['Name'] = name
        self.config['ExpDir'] = exp_dir
        self.config['Seed'] = seed
        for key in kwargs:
            self.config[key] = kwargs[key]
        self.output = dict()

    def run(self) -> None:
        np.random.seed(self.config['Seed'])
        self.create_exp_dir()
        Experiment.write_dict(os.path.join(self.config['ExpDir'], 'config.txt'), self.config)
        try:
            self._run()
        except Exception as exception:
            self.output['Exception'] = exception
        Experiment.write_dict(os.path.join(self.config['ExpDir'], 'output.txt'), self.output)


    @staticmethod
    def write_dict(filepath: str, dict_obj: dict) -> None:
        """
        :param filepath: Path to file where data must be written
        :param dict_obj: Dictionary containing data to be written
        """
        with open(filepath, 'w') as f:
            for key in dict_obj:
                f.write(str(key) + ':' + str(dict_obj[key]) + '\n')

    def create_exp_dir(self) -> None:
        if os.path.exists(self.config['ExpDir']):
            rmtree(self.config['ExpDir'])
        os.mkdir(self.config['ExpDir'])

    def _run(self):
        raise NotImplementedError('Method _run() not implemented')
