

%% Experiment 1: Projected gradient for recovering rank-1 tensor with sparse factors

close all;
clear all;

dim = 10;   % dimension of each factor
degree = 3;   % degree of tensor
k = 0.2*dim;   % sparsity level

repeat = 25;
alpha = linspace(1,3,8);    % sample complexity as power of degree
iter = 500;

% planted (sparse) signal
x = zeros(degree,dim);
for i = 1:degree
    x(i,1:dim) = normrnd(0,1,[1,dim]); 
    x(i,1:k) = zeros(1,k);   % set $k$ of the elements to zero
    x(i,1:dim) = x(i,1:dim)/norm(x(i,1:dim)); % normalize
end

param.verbose = 0;
rec = zeros(length(alpha),repeat);

for len = 1:length(alpha)
    
    m = round(2*(dim-k)*degree^alpha(len));   % number of samples (should scale with sparsity)

    for u = 1:repeat

        % generate observation w/o noise
        y = ones(m,1);
        r = zeros(m,degree,dim);
        for i = 1:m
            for j = 1:degree
                r(i,j,:) = normrnd(0,1,[1,dim]);   % random gaussian tensor sketching
                y(i) = y(i)*x(j,:)*squeeze(r(i,j,:));
            end
        end

        % Spectral initialization
        mat = zeros(dim,dim,dim);

        for i = 1:m

            M = zeros(dim,dim,dim);   % tensor of degree 3
            tmp = squeeze(r(i,1,:))*squeeze(r(i,2,:))';  % this is of size dim*dim
            for j = 1:dim
                M(:,:,j) = tmp*r(i,3,j);
            end
            mat = mat + 1/m * y(i) * M;  
        end

        T = tensor(mat);
        T1 = tucker_als(T,[1 1 1]);   % best rank-1 approximation

        init = zeros(degree,dim);
        for j = 1:degree

            % project so each factor is k-sparse (threshold smallest k in absolute value)
            init(j,:) = T1.U{j}';
            [~,I] = mink(abs(init(j,:)),k);
            init(j,I) = 0;

            % project so each factor is on unit sphere
            init(j,:) = init(j,:)/norm(init(j,:));
        end

        % fprintf('Error at initialization: %f\n', abs(norm(init) - 1)) % norm of x is 1


        % Projected gradient step
        step = 0.1;   % stepsize
        % random initialization
        % v = normrnd(0,1,[degree,dim]);   
        % spectral initialization
        v = init;

        for pp = 1:iter 

            prod = ones(m,1);

            % compute the rank-1 tensor product
            for t = 1:m
                 for j = 1:degree
                      prod(t) = prod(t)*v(j,:)*squeeze(r(t,j,:));
                 end       
            end

            % compute the gradient for each factor v_j
            grad = zeros(dim,degree);

            for j = 1:degree
                for t = 1:m
                    grad(:,j) = grad(:,j)-(y(t)-prod(t))...
                        *(prod(t)/(v(j,:)*squeeze(r(t,j,:))))*squeeze(r(t,j,:));
                end
            end
            v = v-step/m * grad';   % gradient update

            % project so each factor is k-sparse (threshold smallest k in absolute value)
            % for j = 1:degree
            %   [~,I] = mink(abs(v(j,:)),k);
            %   v(j,I) = 0;
            % end

            % project onto ell_1 ball
            for j = 1:degree
                param.epsilon = norm(x(j,:),1);    % ell_1 radius
                v(j,:) = proj_b1(v(j,:), 1, param);  % second input is useless but needed for compatibility issue
            end

            % project so each factor is on unit sphere
            for j = 1:degree
                v(j,:) = v(j,:)/norm(v(j,:));
            end

            % check how well it agrees with measurement
            %fprintf('Error in measurement: %f at iteration %d \n', norm(prod-y), pp)
            % norm of gradient
            %fprintf('Norm of gradient: %f at iteration %d \n', norm(grad), pp)

        end

        rec(len,u) = norm(prod-y);

        % check error of v against x (up to sign)
        disp('Recovered factors:'); disp(v)
        disp('Ground truth factors:'); disp(x)

    end
    
end

%% plot

% find < 0.1 (successful recovery)
for i = 1:length(alpha)
    to_plot(i) = length(find(rec(i,:) <= 0.1));
end

plot(alpha,to_plot,'-r*','LineWidth',2.0)
xlabel('sample complexity (alpha): m = 2 * #nonzero * degree^{alpha}')
ylabel('success trial / 25')
set(gca,'FontSize',20)


%% Experiment 2: Implement two types of embeddings and compute distortion

close all;
clear all;

dim = 10;   % dimension of each factor
degree = 5;   % degree of tensor
k = 0.2*dim;   % sparsity level

repeat = 25;   % averaged across 25 trials
alpha = linspace(1,5,8);    % sample complexity as power of degree
 
% planted (sparse) signal
x = zeros(degree,dim);
for i = 1:degree
    x(i,1:dim) = normrnd(0,1,[1,dim]); 
    x(i,1:k) = zeros(1,k);   % set $k$ of the elements to zero
    x(i,1:dim) = x(i,1:dim)/norm(x(i,1:dim)); % normalize
end


%% random gaussian tensor sketching

me = zeros(1,length(alpha));
va = zeros(1,length(alpha));

for p = 1:length(alpha)
    
    m = round((dim-k)*degree^alpha(p));   % number of samples (should scale with sparsity)
    err = zeros(1,repeat);

    for t = 1:repeat
        y = ones(m,1);
        r = zeros(m,degree,dim);
        for i = 1:m
            for j = 1:degree
                r(i,j,:) = normrnd(0,1,[1,dim]);   
                y(i) = y(i)*x(j,:)*squeeze(r(i,j,:));
            end
        end

        err(t) = abs(norm(y)/sqrt(m)-1);   % norm of x is 1
    end
    
    me(p) = sum(err)/repeat;
    va(p) = var(err);
    
    fprintf('Embedding distortion for row-wise tensored sketch (average): %f \n', me(p)) 
    fprintf('Variance: %f \n', va(p))
    
end


%% recursive random gaussian tensor sketching

me1 = zeros(1,length(alpha));
va1 = zeros(1,length(alpha));


for p = 1:length(alpha)
    
    m = round((dim-k)*degree^alpha(p));
    %m = round((dim-k)*2^alpha(p));   % number of samples (should scale with sparsity)
    err1 = zeros(1,repeat);

    for t = 1:repeat

        s = normrnd(0,1,[m,dim]); 
        tmp = s*x(1,:)'/sqrt(m);  % first layer sketching

        for j = 2:degree

            int = zeros(m,1);

            for i = 1:m 
                s1 = normrnd(0,1,[1,dim]); 
                s2 = normrnd(0,1,[1,m]);  
                int(i) = (s1*x(j,:)') * (s2*tmp);
            end    
            tmp = int/sqrt(m);  % re-normalize
        end

        err1(t) = abs(norm(tmp)-1);  % norm of x is 1
    end
    
    me1(p) = sum(err1)/repeat;
    va1(p) = var(err1);
    
    fprintf('Embedding distortion for recursive tensored sketch (average): %f \n', me1(p)) 
    fprintf('Variance: %f \n', va1(p))
    
end


%% plot

errorbar(alpha,me,va,'o-','LineWidth',2.0)
hold on
errorbar(alpha,me1,va1,'o-','LineWidth',2.0)
xlabel('sample complexity (alpha): m = #nonzero * degree^{alpha}')
ylabel('average distortion over 25 trials: abs(norm(Sx)-1)')
legend('row-wise tensored','recursive')
set(gca,'FontSize',20)

