% generate_problems:
%
% Generates a batch of problems by repeated calls to 'generate_problem'.
% See that function's documentation for more details. The side effect of
% this function is to populate 'path_output' with several different
% synthetic problems. This will be reproducible according to a given seed,
% if provided.
%
% Input:
%
% - num_problems: number of problem instances;
% - path_output: string, where to generate files;
% - p: number of covariates;
% - n: sample size;
% - degree: degree of treatment function polynomial;
% - signal_level: relation between variability in the signal and the noise,
%                 see 'generate_problem' for details;
% - lik_range: interval in which we uniformly sample erro variances for the
%              likelihood function;
% - min_confound_strength: minimum level of confounding acceptable for a
%                          synthetic problem;
% - seed: substitute for a random seed, useful for reproducing particular 
%         synthetic datasets;
% - verbose: show output progress.

function generate_problems(num_problems, path_output, p, n, degree, ...
                           signal_level, lik_range, min_confound_strength, ...
                           seed, verbose)

if nargin >= 9, rng(seed); end
if nargin < 10, verbose = false; end

ell = 2;
sf = 1;
correlate = true;

num_digits = length(num2str(num_problems));
f_d = strcat('%0', num2str(num_digits), 'd');

for i = 1:num_problems
  if verbose
    fprintf('Generating %d\n', i)
  end
  confound_strength = 0;
  while abs(confound_strength) < min_confound_strength
    [dat, model, confound_strength] = generate_problem(n, p, lik_range, ell, sf, degree, signal_level, correlate); %#ok<*ASGLU>
  end
  file_name = strcat(path_output, '/problem', num2str(i, f_d), '.mat');
  save(file_name, 'dat', 'model')
end
