#include "optimistic.h"
#include "shared.h"
#include "alloc_alg.h"
#include "alloc.h"

#include <eigen3/Eigen/LU>
#include <eigen3/Eigen/Eigenvalues>
#include <random>
#include <iostream>


using namespace std;
using namespace Eigen;


/***************************************************************
* Returns random point on sphere.
***************************************************************/
VectorXf OptimismHandler::random_sphere(int D) {
  normal_distribution<float> dist(0.0, 1.0);  
  VectorXf r = VectorXf::Zero(D); 
  for (int i = 0;i != D;++i) {
    r(i) = dist(gen);
  }
  return r / sqrt(r.dot(r));
}


/***************************************************************
* Compute optimistic allocation
* WARNING: Contains tuned constants. No guarantee to return
optimal solution for all problems!!!
***************************************************************/
void OptimismHandler::optimistic(vector<VectorXf> hat_nu, vector<MatrixXf> Gs, float b, MatrixXf &tilde_nu, MatrixXf &Mt, float &best) {
  int K = Gs[0].rows();
  int D = Gs[0].cols();
  vector<MatrixXf> Gs_roots(K);
  vector<MatrixXf> Gs_inverse(K);

  MatrixXf bar_nu(K, D);

  int restarts = 50;
  int iterations = 4;

  for (int k = 0;k != K;++k) {
    Gs_inverse[k] = Gs[k].inverse();
    Gs_roots[k] = root(Gs_inverse[k]);
  }

  bool unset = true;
  for (int i = 0;i < restarts || unset;++i) {
    MatrixXf new_Mt(K, D);
    for (int k = 0;k != K;++k) {
      VectorXf x = random_sphere(D);
      bar_nu.row(k) = hat_nu[k] + b * Gs_roots[k] * x;
    }
    for (int t = 0;t != iterations;++t) {
      Alloc::optimal(bar_nu, new_Mt);
      for (int k = 0;k != K;++k) {
        bar_nu.row(k) = hat_nu[k] + b * Gs_inverse[k] * new_Mt.row(k).transpose() / norm(new_Mt.row(k), Gs_inverse[k]);
      }
    }
    float score = Alloc::optimal(bar_nu, new_Mt);
    if (score > best) {
      best = score;
      Mt = new_Mt;
      tilde_nu = bar_nu;
      unset = false;
    }
  }
}



