#!/usr/bin/ruby

# This code simply loads a set of similarly sized tables of data and computes
# means and standard errors, which are combined in a single table suitable for
# graphing in latex.

class Table
  attr_accessor :row_n, :col_n, :rows
  def initialize(var)
    case var
    when String
      data = File.read(var)
      lines = data.split("\n")
      @row_n = lines.size
      @rows = []
      lines.each do |line|
        @rows << line.split(" ").map{|x| x.to_f}
      end
      @col_n = @rows[0].size
    when Array
      merge_tables(var)
    end
  end

  def mean(xs)
    xs.inject(0) {|s,x|s+=x} / xs.size()
  end

  def var(xs)
    m = Math.sqrt(xs.size())
    mean = mean(xs)
    xs.inject(0) {|s,x|s+=(x / m - mean / m)**2} 
  end

  def std_err(xs)
    1.96 * Math.sqrt(var(xs)/xs.size())
  end

  def merge_tables(tables) 
    @col_n = tables[0].col_n
    @row_n = tables[0].row_n

    @rows = []  

    @row_n.times do |r|
      row = []
      col_n.times do |c|
        if c == 0 
          row << tables[0].rows[r][0]
        else
          xs = tables.map {|table| table.rows[r][c]}
          row << mean(xs)
          row << std_err(xs)
        end
      end
      @rows << row
    end
  end

  def write(fn)
    f = File.open(fn, 'w')
    @rows.each do |row|
      row.each do |x|
        f.write("#{x.round(4)} ")
      end
      f.write("\n")
    end
    f.close()
  end
end

tables = []
(1..8).each do |i|
  tables << Table.new("exp1/free.#{i}.txt")
end

summary = Table.new(tables)
summary.write("exp1/free.summary.txt")

tables = []
(1..8).each do |i|
  tables << Table.new("exp1/fixed.#{i}.txt")
end

summary = Table.new(tables)
summary.write("exp1/fixed.summary.txt")

tables = []
(1..8).each do |i|
  tables << Table.new("exp2/#{i}.txt")
end

summary = Table.new(tables)
summary.write("exp2/summary.txt")




