
############################################################################################
# witness_sensitivity_analysis.R
#
# Perform inference on relaxation parameters for a fixed choice of candidate solutions
#
# Code by
#
#  - Ricardo Silva (ricardo@stats.ucl.ac.uk)
#  - Robin Evans (robin.evans@stats.ox.ac.uk)
#
# Current version: 15/08/2014
# First version: 30/08/2014

library(truncnorm)

############################################################################################
# witness_parameter_analysis::
#
# Analyzes relaxation parameters with res, suggests a posterior distribution over them based on how
# well the entailed bounds capture the ACE of the other data points. The "likelihood" function for 
# the ACEs are iid Gaussian in [-1, 1] with unknown mean mu and variance 
# lik_theta * (upper bound - lower bound). Mean mu is given an uniform prior between the bounds.
# 
# The relaxation parameters are
#
#  * e_xy, so that |P(Y | X, W, U) - P(Y | X, W)| <= e_xy and |P(X | W, U) - P(X | W)| <= e_xy
#  * e_w, so that |P(Y | W = 1, X, U) - P(Y | W = 0, X, U)| <= e_w
#  * e_b, so that e_b <= P(U | W) / P(U) <= 1 / e_b
#
# All parameters are given independent truncated Gaussian priors:
#
#  * P(e | m, v) \propto -(e - m)^2 / 2v, 0 <= e <= 1
#
# Input:
#
# - problem: the problem instance, the usual data structure
# - w_choice, Z_choice: 
# - witnesses: the set of other witnesses, an array
# - Zs: the corresponding set of admissible sets for the other witnesses
# - m_prior, v_prior: the prior hyperparameters for eps_xy/eps_w/eps_b, respectively.
#                     Set v_prior to Inf to get an uniform [0, 1] prior
# - M: number of MCMC samplers
# - taboo_vars: variables discarded a prior as not being common causes of X and Y
# - ignore_empty: ignore empty admissible sets
# - f: proposal parameter for the sampling of eps
# - f_mu: proposal parameter for the sampling of mu
#
# Output:
#
# - eps: distribution of the three parameters, columns are different MCMC samples.
# - mu: mean of Gaussian process

witness_infer_eps <- function(problem, w_choice, Z_choice, Zs, m_prior, v_prior, M, taboo_vars = c(), ignore_empty = TRUE, f = 0.9, f_mu = 0.1)
{

  llik_eps_mu <- function(bounds, ACEs, mu, v) {
    
    if (mu < bounds[1] || mu > bounds[2]) return(-Inf)
    
    var_ACEs <- ((bounds[2] - bounds[1]) / 6)^2
    output <- -0.5 * length(ACEs) * log(var_ACEs) -0.5 * sum((ACEs - mu)^2) / var_ACEs -
      log(length(ACEs) * (pnorm(1, mu, sqrt(var_ACEs)) - pnorm(-1, mu, sqrt(var_ACEs)))) + 
      -log(bounds[2] - bounds[1])
    
    return(output)
    
  }
  
  lprior <- function(eps, m_prior, v_prior) {
    if (eps < 0 || eps > 1) return(-Inf)
    if (v_prior != Inf) return(-0.5 * (eps - m_prior)^2 / v_prior)
    return(0) # Uniform prior
  }
  
  propose_eps <- function(eps, f) {
    lb <- eps * f; ub <- eps / f; ub[ub > 1] <- 1
    new_eps <- runif(length(eps), lb, ub)
    lfwd <- -log(ub - lb)
    lb <- new_eps * f; ub <- new_eps / f; ub[ub > 1] <- 1
    lbwd <- -log(ub - lb)
    return(list(eps = new_eps, lfwd = lfwd, lbwd = lbwd))
  }
  
  propose_mu <- function(mu, bounds, f) {
    new_mu <- rtruncnorm(1, bounds[1], bounds[2], mu, f)
    lfwd <- log(dtruncnorm(new_mu, bounds[1], bounds[2], mu, f))
    lbwd <- log(dtruncnorm(mu, bounds[1], bounds[2], new_mu, f))
    return(list(mu = new_mu, lfwd = lfwd, lbwd = lbwd))
  }
    
  eps <- matrix(nrow = 3, ncol = M)
  mu <- rep(0, M)
  
  if (is.null(problem$counts)) problem$counts <- dtoc(problem$data)  
    
  ACEs <- rep(0, length(Zs))
  accept_ACE <- rep(1, length(Zs))
  for (i in seq_along(Zs)) {
    if (length(intersect(Zs[[i]], taboo_vars)) > 0) {
      accept_ACE[i] <- 0
      next
    }
    if (length(Zs[[i]]) == 0 && ignore_empty) {
      accept_ACE[i] <- 0
      Zs[[i]] <- -1
      next
    }
    effects <- bindag_causal_effect_backdoor_S(problem, Zs[[i]])
    ACEs[i] <- effects$ACE
    if (is.subset(Z_choice, Zs[[i]])) {
      accept_ACE[i] <- 0
      next
    }
    if (i > 1) {
      for (j in 1:(i - 1)) {
        if (setequal(Zs[[i]], Zs[[j]])) {
          accept_ACE[j] <- 0
          next
        }
        if (is.subset(Zs[[i]], Zs[[j]])) accept_ACE[j] <- 0
        if (is.subset(Zs[[j]], Zs[[i]])) accept_ACE[i] <- 0
      }
    }
  } 
  ACEs <- ACEs[accept_ACE == 1]
  if (length(ACEs) < 2) {
    cat("Not enough alternative ACEs for sensitivity analysis\n")
    return(NULL)
  }
  
  # Generate initial parameters so that all ACEs are covered
  
  eps[, 1] <- m_prior
  repeat {
    epsilons <- c(eps[1, 1], eps[2, 1], eps[2, 1], eps[2, 1], eps[3, 1], 1 / eps[3, 1])
    bounds <- witness_simple_bound(problem, epsilons, w_choice, Z_choice)
    outcome <- propose_eps(eps[, 1], f)
    eps[, 1] <- outcome$eps
    if (!is.null(bounds)) break    
  }
  
  mu[1] <- mean(ACEs)
  if (mu[1] < bounds[1] || mu[1] > bounds[2]) mu[1] <- runif(1, bounds[1], bounds[2])
  old_llik <- llik_eps_mu(bounds, ACEs, mu[1], lik_theta)  
  
  # Perform MCMC
  
  for (m in 2:M) {
    
    cat("Iteration", m, "\n")
    
    eps[, m] <- eps[, m - 1]
    mu[m] <- mu[m - 1]
    
    # Sample eps
    
    for (i in 1:3) {      
      
      old_eps <- eps[i, m]    
      old_prior <- lprior(eps[i, m], m_prior[i], v_prior[i])
      
      outcome <- propose_eps(eps[i, m], f)
      eps[i, m] <- outcome$eps
      lfwd <- outcome$lfwd
      lbwd <- outcome$lbwd
      epsilons <- c(eps[1, m], eps[2, m], eps[2, m], eps[2, m], eps[3, m], 1 / eps[3, m])
      
      new_bounds <- witness_simple_bound(problem, epsilons, w_choice, Z_choice)
      if (is.null(new_bounds) || new_bounds[1] > mu[m] || new_bounds[2] < mu[m]) {
        eps[i, m] <- old_eps
        next
      }      
      new_llik <- llik_eps_mu(new_bounds, ACEs, mu[m], lik_theta)
      new_lprior <- lprior(eps[i, m], m_prior[i], v_prior[i])
      
      ratio <- exp(new_llik + new_lprior + lbwd - old_llik - old_prior - lfwd)
      if (runif(1) > ratio) {
        eps[i, m] <- old_eps 
      } else {
        old_llik <- new_llik
        bounds <- new_bounds
      }
      
    }
    
    # Sample mu
    
    old_mu <- mu[m]    
    
    outcome <- propose_mu(mu[m], bounds, f_mu)
    mu[m] <- outcome$mu
    lfwd <- outcome$lfwd
    lbwd <- outcome$lbwd
    new_llik <- llik_eps_mu(bounds, ACEs, mu[m], lik_theta)
    
    ratio <- exp(new_llik + lbwd - old_llik - lfwd)
    if (runif(1) > ratio) {
      mu[m] <- old_mu
    } else {
      old_llik <- new_llik
    }
    
  }
  
  return(list(eps = eps, mu = mu, ACEs = ACEs))
  
}
