############################################################################################
# bayesian.R
#
# Here we develop methods for full inference of causal effect parameters directly from data/
#
# The prior is given as a constrained Dirichlet, where constraints are given on the entries
# of the conditional table P_YX.W
#
# Sampling is done by rejection sampling. After sampling is complete, we then generate
# a posterior distribution over bounds.
#
# Code by
#
#  - Ricardo Silva (ricardo@stats.ucl.ac.uk)
#  - Robin Evans (robin.evans@stats.ox.ac.uk)
#
# Current version: 15/08/2014
# First version: 25/03/2014

source("LPP_tools.R")

rowMins = function(x) do.call(pmin, as.data.frame(x))
rowMaxs = function(x) do.call(pmax, as.data.frame(x))

############################################################################################
# bayesian_posterior_sampling::
#
# Provide a set of samples from the posterior given by a Dirichlet prior and the corresponding
# data. Rejection sampling is used, so if model is bad ("far" from a chain W -> X -> Y)
# then the number of rejections will be high.
#
# * Input
#
# - alpha_0, alpha_1: 4-dimensional vectors corresponding to the priors for P(Y, X | W = 0)
#        and P(Y, X | W = 1), where indices 1, 2, 3, 4 refer to
#        (Y = 0, X = 0), (Y = 0, X = 1), (Y = 1, X = 0), (Y = 1, X = 1)
# - alpha_W: 2-dimensional vector corresponding to the priors for P(W = 1)
# - data: data matrix, columns correspond to W, X, Y; used only if counts are missing
# - epsilons: relaxation factors as described in bayesian_validate_constraints
# - M: number of samples
# - verbose: TRUE makes more messages appear
# - numerical_method: if TRUE, use black-box linear programming
# - use_independence: if TRUE, generate samples under the independence implied by W -> X -> Y
# - counts: array of count data over W, X, Y

bayesian_posterior_sampling <- function(alpha_0, alpha_1, alpha_W, data, epsilons, M, counts,
                                        numerical_method = FALSE, use_independence = FALSE, verbose = FALSE)  
{
  output_0 <- output_1 <- matrix(0, nrow = M, ncol = 4)

  # Get sufficient statistics

  ## eventually make this main argument
  if (missing(counts)) counts = array(tabulate(data %*% c(1, 2, 4) + 1, nbins = 8), rep(2,3))

  count_0 = c(counts[1,,]) + alpha_0
  count_1 = c(counts[2,,]) + alpha_1

  # Marginal of W
  n_1 <- sum(counts[2,,])
  output_W <- rbeta(M, n_1 + alpha_W[1], sum(counts) - n_1 + alpha_W[2])

  # Sample

  rejections <- 0
  m <- 0
  remaining <- M
  fail_count <- 0
  first_iter <- TRUE

  while (m < M) {

    old_remaining <- remaining
    remaining <- M - m
    if (remaining == old_remaining || remaining > 0.95 * M) {
      fail_count <- fail_count + 1
      if (!first_iter && (remaining > 0.95 * M || fail_count > 100)) {
        if (verbose) cat("Model falsified, dropping it\n")
        dummy_set <- matrix(ncol = 8, nrow = 0)
        return(list(W = output_W, W0 = dummy_set, W1 = dummy_set, rejection_rate = 1))
      }
    } else {
      fail_count <- 0
    }

    first_iter <- FALSE
    #if (verbose) cat("Remaining samples", remaining, "\n")

    if (!use_independence) {
      P_YX.W0 <- matrix(rgamma(4 * remaining, count_0, 1), ncol = 4, byrow = TRUE)
      P_YX.W1 <- matrix(rgamma(4 * remaining, count_1, 1), ncol = 4, byrow = TRUE)

      ## P_YX.W0 <- matrix(rep(0, remaining * 4), ncol = 4)
      ## P_YX.W1 <- matrix(rep(0, remaining * 4), ncol = 4)
      ## for (i in 1:4) {
      ##   P_YX.W0[, i] <- rgamma(remaining, count_0[i], 1)
      ##   P_YX.W1[, i] <- rgamma(remaining, count_1[i], 1)
      ## }
      P_YX.W0 <- P_YX.W0 / rowSums(P_YX.W0)
      P_YX.W1 <- P_YX.W1 / rowSums(P_YX.W1)
    }
    else {
      P_X.W0 <- matrix(rgamma(2 * remaining, c(count_0[1] + count_0[3], count_0[2] + count_0[4]), 1),
                       ncol = 2, byrow = TRUE)
      P_X.W1 <- matrix(rgamma(2 * remaining, c(count_1[1] + count_1[3], count_1[2] + count_1[4]), 1),
                       ncol = 2, byrow = TRUE)

      ## P_X.W0 <- matrix(rep(0, remaining * 2), ncol = 2)
      ## P_X.W1 <- matrix(rep(0, remaining * 2), ncol = 2)
      ## for (i in 1:2) {
      ##   P_X.W0[, i] <- rgamma(remaining, count_0[i] + count_0[2 + i], 1)
      ##   P_X.W1[, i] <- rgamma(remaining, count_1[i] + count_1[2 + i], 1)
      ## }
      P_X.W0 <- P_X.W0 / rowSums(P_X.W0)
      P_X.W1 <- P_X.W1 / rowSums(P_X.W1)

      P_Y.X0 <- matrix(rgamma(2 * remaining, count_0[c(1,3)] + count_1[c(1, 3)], 1), ncol = 2, byrow = TRUE)
      P_Y.X1 <- matrix(rgamma(2 * remaining, count_0[c(2,4)] + count_1[c(2, 4)], 1), ncol = 2, byrow = TRUE)
      ## P_Y.X0 <- matrix(rep(0, remaining * 2), ncol = 2)
      ## P_Y.X0[, 1] <- rgamma(remaining, count_0[1] + count_1[1], 1)
      ## P_Y.X0[, 2] <- rgamma(remaining, count_0[3] + count_1[3], 1)
      ## P_Y.X1 <- matrix(rep(0, remaining * 2), ncol = 2)
      ## P_Y.X1[, 1] <- rgamma(remaining, count_0[2] + count_1[2], 1)
      ## P_Y.X1[, 2] <- rgamma(remaining, count_0[4] + count_1[4], 1)
      P_Y.X0 <- P_Y.X0 / rowSums(P_Y.X0)
      P_Y.X1 <- P_Y.X1 / rowSums(P_Y.X1)


      P_YX.W0 = c(P_Y.X0[,1], P_Y.X1[,1], P_Y.X0[,2], P_Y.X1[,2]) * P_X.W0[,c(1, 2, 1, 2)]
      P_YX.W1 = c(P_Y.X0[,1], P_Y.X1[,1], P_Y.X0[,2], P_Y.X1[,2]) * P_X.W1[,c(1, 2, 1 ,2)]
      ## P_YX.W0 <- matrix(rep(0, remaining * 4), ncol = 4)
      ## P_YX.W1 <- matrix(rep(0, remaining * 4), ncol = 4)
      ## P_YX.W0[, 1] <- P_Y.X0[, 1] * P_X.W0[, 1]
      ## P_YX.W0[, 2] <- P_Y.X1[, 1] * P_X.W0[, 2]
      ## P_YX.W0[, 3] <- P_Y.X0[, 2] * P_X.W0[, 1]
      ## P_YX.W0[, 4] <- P_Y.X1[, 2] * P_X.W0[, 2]
      ## P_YX.W1[, 1] <- P_Y.X0[, 1] * P_X.W1[, 1]
      ## P_YX.W1[, 2] <- P_Y.X1[, 1] * P_X.W1[, 2]
      ## P_YX.W1[, 3] <- P_Y.X0[, 2] * P_X.W1[, 1]
      ## P_YX.W1[, 4] <- P_Y.X1[, 2] * P_X.W1[, 2]
    }

    if (numerical_method) {
      if (verbose) {
        cat("  Remaining", remaining, "\n")
      }
      for (i in 1:remaining) {
        result <- bayesian_validate_constraints(P_YX.W0[i, ], P_YX.W1[i, ], epsilons)
        if (result) {
          m <- m + 1
          output_0[m, ] <- P_YX.W0[i, ]
          output_1[m, ] <- P_YX.W1[i, ]
        } else {
          rejections <- rejections + 1
        }
      }
    }
    else {
      result <- bayesian_validate_constraints_analytical(P_YX.W0, P_YX.W1, epsilons)
      passed <- (result == 1)
      new_m <- (m + 1):(m + sum(result))
      if (sum(result) > 0) {
        output_0[new_m, ] <- P_YX.W0[passed, ]
        output_1[new_m, ] <- P_YX.W1[passed, ]
      }
      m <- m + sum(result)
      rejections <- rejections + sum(1 - result)
    }

  }

  rejection_rate <- rejections / (rejections + M)

  # Wrap up

  return(list(W = output_W, W0 = output_0, W1 = output_1, rejection_rate = rejection_rate))
}

############################################################################################
# bayesian_validate_constraints::
#
# This function returns TRUE if parameters P_YX.W0 and P_YX.W1 satisfied the constraints
# of the relaxed confounded model, FALSE otherwise.
#
# * Input
#
# - P_YX.W0, P_YX.W1: two four dimensional arrays, where indices 1, 2, 3, 4 refer to
#        (Y = 0, X = 0), (Y = 0, X = 1), (Y = 1, X = 0), (Y = 1, X = 1)
# - epsilons: vector containing the relaxations for the various constraints. In particular,
#
#     |eta_x0^star - eta_x1^star| <= epsilons[1]
#     |eta_x0^star - P(Y = 1 | X = x, W = 0)| <= epsilons[2]
#     |eta_x1^star - P(Y = 1 | X = x, W = 1)| <= epsilons[3]
#     |delta_w0^star - P(X = 1 | W = 0, U)| <= epsilons[4]
#     |delta_w1^star - P(X = 1 | W = 1, U)| <= epsilons[4]
#     epsilons[5] * P(U) <= P(U | W = w) <= epsilons[6] * P(U)
#
# Notice that the first three constraints entail
#
# |P(Y = 1 | X = x, W = 0) - P(Y = 1 | X = x, W = 1)| <= epsilons[1] + epsilons[2] + epsilons[3]
#
# which has to be encoded manually.

bayesian_validate_constraints <- function(P_YX.W0, P_YX.W1, epsilons)
{
  # Some rounding first for approximate rejection
  
  P_YX.W0 <- round(P_YX.W0 * 1000); P_YX.W0 <- P_YX.W0 / sum(P_YX.W0)
  P_YX.W1 <- round(P_YX.W1 * 1000); P_YX.W1 <- P_YX.W1 / sum(P_YX.W1)
  
  # First, find the extreme points in which eta_xw^star can vary.
  
  P_Y.00 <- P_YX.W0[3] / (P_YX.W0[1] + P_YX.W0[3])
  P_Y.01 <- P_YX.W1[3] / (P_YX.W1[1] + P_YX.W1[3])
  P_Y.10 <- P_YX.W0[4] / (P_YX.W0[2] + P_YX.W0[4])
  P_Y.11 <- P_YX.W1[4] / (P_YX.W1[2] + P_YX.W1[4])
  
  # Check immediate constraints (possible speed up?)
  # |P(Y = 1 | X = x, W = 0) - P(Y = 1 | X = x, W = 1)| <= epsilons[1] + epsilons[2] + epsilons[3]
  
  sum_eps <- epsilons[1] + epsilons[2] + epsilons[3]
  if (is.na(P_Y.00) || is.na(P_Y.01) || is.na(P_Y.10) || is.na(P_Y.11)) return(FALSE)
  if (abs(P_Y.00 - P_Y.01) > sum_eps || abs(P_Y.10 - P_Y.11) > sum_eps) return(FALSE)
  
  # Proceed
  
  eta_space_0 <- get_V_eta_star(epsilons, c(P_Y.00, P_Y.01))
  if (length(eta_space_0) == 0) return(FALSE)
  
  eta_space_1 <- get_V_eta_star(epsilons, c(P_Y.10, P_Y.11))
  if (length(eta_space_1) == 0) return(FALSE)
  
  # Obtain the extreme points of delta_w^star
  
  P_X1.W0 <- P_YX.W0[2] + P_YX.W0[4]
  P_X1.W1 <- P_YX.W1[2] + P_YX.W1[4]
  delta_space_0 <- c(max(0, P_X1.W0 - epsilons[4]), min(1, P_X1.W0 + epsilons[4]))
  delta_space_1 <- c(max(0, P_X1.W1 - epsilons[4]), min(1, P_X1.W1 + epsilons[4]))
  
  # Complete the operation to obtain extreme points of zeta
  
  #cat("1")
  all_extreme <- build_table_parameters(eta_space_0, eta_space_1, delta_space_0, delta_space_1, FALSE)
  extreme <- all_extreme$T2[, 1:8]
  
  # Now build inequality constraints: basic setup
  
  V <- makeV(extreme)
  pre_H <- c()
  try(pre_H <- scdd(V), silent = TRUE)
  if (length(pre_H) == 0) return(FALSE)
  H_matrix <- pre_H$output
  ineq_idx <- (H_matrix[ ,1] == 0)
  A <- -H_matrix[ineq_idx, 3:ncol(H_matrix), drop = FALSE]
  b <- H_matrix[ineq_idx, 2]
  
  # The above should be interpreted as
  #
  # -- A * kappa <= b
  #
  # where kappa is a vector representation of the matrix kappa(Y, X | W).
  # (equality constraints are assumed to be satisfied automatically)
  #
  # In this case,
  #
  # -- kappa(Y, X | W) = sum_U P(Y, X | W, U)P(U)
  #
  # which is not the same as P(Y, X | W) since in general P(U) != P(U | W).
  #
  # However, for all w we are assuming
  #
  # -- beta_lower * P(U) <= P(U | W = w) <= beta_upper * P(U)
  #
  # where 0 <= beta_lower <= 1 and beta_upper >= 1.
  #
  # This implies
  #
  # -- P(Y, X | W) / beta_upper <= kappa(Y, X | W) <= P(Y, X | W) / beta_lower
  #
  # We merge these constraints along with those found for kappa, then introduce
  # the equality constraints and test for it.
  
  beta_lower <- epsilons[5]
  beta_upper <- epsilons[6]
  A_Pkappa <- matrix(0, nrow = 2 * 8, ncol = 8)
  b_Pkappa <- rep(0, 16)
  i <- 1
  
  for (y in 1:2)
    for (x in 1:2) {
      w <- 1
      idx <- 4 * (w - 1) + 2 * (y - 1) + x
      A_Pkappa[i, idx] <-  1; b_Pkappa[i] <-  P_YX.W0[2 * (y - 1) + x] / beta_lower; i <- i + 1
      A_Pkappa[i, idx] <- -1; b_Pkappa[i] <- -P_YX.W0[2 * (y - 1) + x] / beta_upper; i <- i + 1
      w <- 2
      idx <- 4 * (w - 1) + 2 * (y - 1) + x
      A_Pkappa[i, idx] <-  1; b_Pkappa[i] <-  P_YX.W1[2 * (y - 1) + x] / beta_lower; i <- i + 1
      A_Pkappa[i, idx] <- -1; b_Pkappa[i] <- -P_YX.W1[2 * (y - 1) + x] / beta_upper; i <- i + 1
    }
  A_eq <- matrix(0, nrow = 2, ncol = 8); b_eq <- c(1, 1)
  A_eq[1, 1:4] <- 1 # Sum k_{yx.w0} == 1
  A_eq[2, 5:8] <- 1 # Sum k_{yx.w1} == 1
  constr_type <- c(rep("<=", nrow(A) + nrow(A_Pkappa)), rep("=", length(b_eq)))
  A <- rbind(A, A_Pkappa, A_eq); b <- c(b, b_Pkappa, b_eq)
  
  H <- cbind(c(rep(0, length(b) - length(b_eq)), rep(1, length(b_eq))), b, -A)
  dummy_result <- lpcdd(H, rep(0, 8))
  
  # Now, decide
  
  return(dummy_result$solution.type != "Inconsistent")
}

############################################################################################
# bayesian_interval_generation::
#
# Now that we have samples over P_W1, P_YX.W0, P_YX.W1, we can calculate posteriors over
# causal bounds.
#
# * Input
#
# - theta_sample: structure composed of fields W0 and W1, where W0, W1 are M by 4
#      matrices. Row indices are sample ids. Column indices 1, 2, 3, 4 refer to
#      (Y = 0, X = 0), (Y = 0, X = 1), (Y = 1, X = 0), (Y = 1, X = 1) conditioned on
#      W = 0 and W = 1, respectively
# - epsilons: vector containing the relaxations for the various constraints. In particular,
#
#     |eta_x0^star - eta_x1^star| <= epsilons[1]
#     |eta_x0^star - P(Y = 1 | X = x, W = 0)| <= epsilons[2]
#     |eta_x1^star - P(Y = 1 | X = x, W = 1)| <= epsilons[3]
#     |delta_w0^star - P(X = 1 | W = 0, U)| <= epsilons[4]
#     |delta_w1^star - P(X = 1 | W = 1, U)| <= epsilons[4]
#     epsilons[5] * P(U) <= P(U | W = w) <= epsilons[6] * P(U)


bayesian_interval_generation <- function(theta_sample, epsilons, verbose = FALSE)
{
  M <- nrow(theta_sample$W0)
  ACE_lp_relax <- matrix(NA, nrow = M, ncol = 2)

  if (verbose) cat("Solving samples\n")

  for (m in 1:M) {

    if (verbose) cat("Solving sample", m , "\n")

    P_YX.W0 <- theta_sample$W0[m, ]
    P_YX.W1 <- theta_sample$W1[m, ]
    P_W     <- theta_sample$W[m]

    # First, find the extreme points in which eta_xw^star can vary.

    P_Y.00 <- P_YX.W0[3] / (P_YX.W0[1] + P_YX.W0[3])
    P_Y.01 <- P_YX.W1[3] / (P_YX.W1[1] + P_YX.W1[3])
    eta_space_0 <- get_V_eta_star(epsilons, c(P_Y.00, P_Y.01))
    if (length(eta_space_0) == 0) { # Due to rounding
      next
    }
    P_Y.10 <- P_YX.W0[4] / (P_YX.W0[2] + P_YX.W0[4])
    P_Y.11 <- P_YX.W1[4] / (P_YX.W1[2] + P_YX.W1[4])
    eta_space_1 <- get_V_eta_star(epsilons, c(P_Y.10, P_Y.11))
    if (length(eta_space_1) == 0) {
      next
    }

    # Obtain the extreme points of delta_w^star

    P_X1.W0 <- P_YX.W0[2] + P_YX.W0[4]
    P_X1.W1 <- P_YX.W1[2] + P_YX.W1[4]
    delta_space_0 <- c(max(0, P_X1.W0 - epsilons[4]), min(1, P_X1.W0 + epsilons[4]))
    delta_space_1 <- c(max(0, P_X1.W1 - epsilons[4]), min(1, P_X1.W1 + epsilons[4]))

    # Complete the operation to obtain extreme points of zeta

    all_extreme <- build_table_parameters(eta_space_0, eta_space_1, delta_space_0, delta_space_1, FALSE)

    # Run linear program

    P_ZETA <- c(P_YX.W0, P_YX.W1)
    P_matrix <- all_extreme$T2
    lpp_params <- build_lpp_scdd(P_matrix, c(1 - P_W, P_W), P_ZETA, epsilons[5], epsilons[6])
    if (length(lpp_params) == 0) {
      next
    }
    c_type <- rep(0, length(lpp_params$constr_type)); c_type[which(lpp_params$constr_type == "=")] <- 1
    H <- cbind(c_type, lpp_params$b, -lpp_params$A)
    result_max_relax <- NULL; result_min_relax <- NULL
    try(result_max_relax <- lpcdd(H, lpp_params$C, minimize = FALSE), silent = TRUE)
    if (length(result_max_relax) == 0) next    
    try(result_min_relax <- lpcdd(H, lpp_params$C, minimize = TRUE), silent = TRUE)
    if (length(result_min_relax) == 0) next        
    if (result_max_relax$solution.type == "Optimal") {
      ACE_lp_relax[m, 2] <- result_max_relax$optimal.value
    }    
    if (result_min_relax$solution.type == "Optimal") {
      ACE_lp_relax[m, 1] <- result_min_relax$optimal.value
    }
    
  }

  return(ACE_lp_relax)
}


############################################################################################
# bayesian_validate_constraints_analytical::
#
# Default, "message passing" implementation of constraint validation. Constraints are not
# as tight as in 'bayesian_validate_constraints', but it doesn't require running a linear
# programming solver.
#
# This function returns TRUE if parameters P_YX.W0 and P_YX.W1 satisfied the constraints
# of the relaxed confounded model, FALSE otherwise.
#
# * Input
#
# - P_YX.W0, P_YX.W1: as in 'bayesian_validate_constraints'
# - epsilons: as in 'bayesian_validate_constraints'
# - return_solutions: if true, return intervals (used to generate bounds)
# - P_W: marginal distribution of W, as used to return intervals
#
# * Output:
#
# - intervals: if return_solutions is TRUE, this is passed back - a matrix where first column
#              are lower bounds, and second column are upper bounds
# - passed: if return_solutions is FALSE, this is passed back - an array indicating which
#           data points violate the constraints

bayesian_validate_constraints_analytical <- function(P_YX.W0, P_YX.W1, epsilons, return_solutions = FALSE, P_W = NULL)
{
  N <- nrow(P_YX.W0)
  pass <- rep(1, N)

  ## indices
  i_x.w = 1:4
  i_xp.w = c(2,1,4,3)
  i_x.wp = c(3,4,1,2)
  i_xp.wp = c(4,3,2,1)
  
  p_xy.w = matrix(c(P_YX.W0, P_YX.W1), nrow=N)
  p_y.xw = conditionMatrix(p_xy.w, 2, c(1,3))
  p_x.w = marginMatrix(p_xy.w, c(1,3))

  eps_w      <- epsilons[1]
  eps_Y      <- epsilons[2]
  eps_X      <- epsilons[4]
  beta_lower <- epsilons[5]
  beta_upper <- epsilons[6]

  # Calculate auxiliary variables
  UK_XY.W = pmin(p_xy.w / beta_lower, 1)
  LK_XY.W = p_xy.w / beta_upper

  UChi = beta_upper*p_x.w
  LChi = beta_lower*p_x.w

  L_X = pmax(p_x.w - eps_X, 0)
  U_X = pmin(p_x.w + eps_X, 1)

  U_Y = pmin(p_y.xw[,c(3,4,7,8), drop=FALSE] + eps_Y, 1)
  L_Y = pmax(p_y.xw[,c(3,4,7,8), drop=FALSE] - eps_Y, 0)

  U_bar <- rowMaxs(U_Y)
  L_bar <- rowMins(L_Y)

  ## Derive the box constraints for omega_xw first
  ## Theorem 1 bounds
  upper =                 UK_XY.W[,c(3,4,7,8), drop=FALSE] + U_Y * UChi[,i_xp.w, drop=FALSE]
  upper = pmin(upper,     UK_XY.W[,c(3,4,7,8), drop=FALSE] / L_X)
  upper = pmin(upper, 1 - LK_XY.W[,c(1,2,5,6), drop=FALSE] / U_X)

  lower =                 LK_XY.W[,c(3,4,7,8), drop=FALSE] + L_Y * LChi[,i_xp.w, drop=FALSE]
  lower = pmax(lower,     LK_XY.W[,c(3,4,7,8), drop=FALSE] / U_X)
  lower = pmax(lower, 1 - UK_XY.W[,c(1,2,5,6), drop=FALSE] / L_X)

  ## Theorem 2 bounds
  upper <- pmin(upper,     (UK_XY.W[,c(7,8,3,4), drop=FALSE] + eps_w * UChi[, i_x.wp, drop=FALSE]) / L_X[, i_x.wp, drop=FALSE])
  upper <- pmin(upper, 1 - (LK_XY.W[,c(5,6,1,2), drop=FALSE] - eps_w * UChi[, i_x.wp, drop=FALSE]) / U_X[, i_x.wp, drop=FALSE])
  lower <- pmax(lower,     (LK_XY.W[,c(7,8,3,4), drop=FALSE] - eps_w * UChi[, i_x.wp, drop=FALSE]) / U_X[, i_x.wp, drop=FALSE])
  lower <- pmax(lower, 1 - (UK_XY.W[,c(5,6,1,2), drop=FALSE] + eps_w * UChi[, i_x.wp, drop=FALSE]) / L_X[, i_x.wp, drop=FALSE])

  ## Bounds from Theorem 3
  upper <- pmin(upper,
                   (UK_XY.W[,c(8,7,4,3), drop=FALSE] + UK_XY.W[,c(7,8,3,4), drop=FALSE] + UK_XY.W[,c(3,4,7,8), drop=FALSE] +
                  - LK_XY.W[,c(4,3,8,7), drop=FALSE] + UChi[,i_xp.w, drop=FALSE]*(U_bar + L_bar + 2 * eps_w) - L_bar))
  upper <- pmin(upper,
                   (UK_XY.W[,c(4,3,8,7), drop=FALSE] + UK_XY.W[, c(7,8,3,4), drop=FALSE] + UK_XY.W[, c(3,4,7,8), drop=FALSE] - LK_XY.W[, c(8,7,4,3), drop=FALSE] +
                  + 2 * UChi[,i_xp.w, drop=FALSE] * eps_w + UChi[,c(4,3,2,1), drop=FALSE] * (U_bar + L_bar) - L_bar))
  lower <- pmax(lower,
                 (- UK_XY.W[,c(8,7,4,3), drop=FALSE] + LK_XY.W[,c(7,8,3,4), drop=FALSE] + LK_XY.W[,c(3,4,7,8), drop=FALSE] +
                   LK_XY.W[,c(4,3,8,7), drop=FALSE] + - 2 * UChi[,i_xp.w, drop=FALSE] * eps_w + LChi[,c(4,3,2,1), drop=FALSE] * (U_bar + L_bar) - U_bar))
  lower <-  pmax(lower,
                  (-UK_XY.W[,c(4,3,8,7), drop=FALSE] + LK_XY.W[,c(7,8,3,4), drop=FALSE] + LK_XY.W[,c(3,4,7,8), drop=FALSE] +
                  + LK_XY.W[,c(8,7,4,3), drop=FALSE] - UChi[,i_xp.w, drop=FALSE] * 2 * eps_w + LChi[,i_xp.w, drop=FALSE]*(U_bar +L_bar) - U_bar))

  upper[is.nan(upper)] = 1
  upper = pmin(upper, 1)
  lower[is.nan(lower)] = 0
  lower = pmax(lower, 0)

  pass = pass*rowMins(upper >= lower)

  ## bounds for omega_xw
  omega_upper = upper
  omega_lower = lower

  ## bounds for differences omega_xw - omega_xw'
  diff_upper <- matrix(eps_w, nrow=N, ncol = 4)
  diff_lower <- matrix(-eps_w, nrow=N, ncol = 4)

  ## Now, iterate over linear constraints
  for (iter in 1:4) {

    upper = omega_upper
    lower = omega_lower

    ## #############################################
    ## Iteration over the linear constraints of Theorem 2
    upper = pmin(upper,
                 omega_upper[,i_x.wp, drop=FALSE]*U_X[,i_xp.w, drop=FALSE] +
                 UK_XY.W[,c(3,4,7,8), drop=FALSE] + eps_w*UChi[,i_xp.w, drop=FALSE])
    upper = pmin(upper,
                 (omega_upper[,i_x.wp, drop=FALSE] - 1)*L_X[,i_xp.w, drop=FALSE] +
                 1 - LK_XY.W[,c(1,2,5,6), drop=FALSE]  + eps_w * UChi[,i_xp.w, drop=FALSE])
    upper = pmin(upper, omega_upper[,i_x.wp, drop=FALSE] + eps_w)

    lower = pmax(lower,
                 omega_lower[,i_x.wp, drop=FALSE]*L_X[,i_xp.w, drop=FALSE] +
                      LK_XY.W[,c(3,4,7,8), drop=FALSE] - eps_w * UChi[,i_xp.w, drop=FALSE])
    lower = pmax(lower,
                 (omega_lower[,i_x.wp, drop=FALSE] - 1)*U_X[,i_xp.w, drop=FALSE] +
                      1 - UK_XY.W[,c(1,2,5,6), drop=FALSE] - eps_w * UChi[,i_xp.w, drop=FALSE])
    lower = pmax(lower, omega_lower[,i_x.wp, drop=FALSE] - eps_w)

    omega_upper = upper
    omega_lower = lower

    pass = pass*rowMins(upper >= lower)

    ## #############################################
    ## Iteration over the linear constraints of Theorem 2 to bound omega_xw - omega_xw'
    ## equation (9)
    upper = pmin(diff_upper,
                 omega_upper[,i_x.wp, drop=FALSE]*(U_X[,i_xp.w, drop=FALSE] - 1) +
                      UK_XY.W[,c(3,4,7,8), drop=FALSE] + eps_w*UChi[,i_xp.w, drop=FALSE])
    upper = pmin(upper,
                 (omega_upper[,i_x.wp, drop=FALSE] - 1)*(L_X[,i_xp.w, drop=FALSE] - 1)  +
                      - LK_XY.W[,c(1,2,5,6), drop=FALSE]  + eps_w * UChi[,i_xp.w, drop=FALSE])
    upper = pmin(upper, omega_upper - omega_lower[,i_x.wp, drop=FALSE])

    lower = pmax(diff_lower,
                 omega_lower[,i_x.wp, drop=FALSE]*(L_X[,i_xp.w, drop=FALSE]-1) +
                      LK_XY.W[,c(3,4,7,8), drop=FALSE] - eps_w * UChi[,i_xp.w, drop=FALSE])
    lower = pmax(lower,
                 (omega_lower[,i_x.wp, drop=FALSE] - 1)*(U_X[,i_xp.w, drop=FALSE]-1) +
                       - UK_XY.W[,c(1,2,5,6), drop=FALSE] - eps_w * UChi[,i_xp.w, drop=FALSE])
    lower = pmax(lower, omega_lower - omega_upper[,i_x.wp, drop=FALSE])

    ## ## in the old code we did the following, but this doesn't look right
    ## upper = pmax(upper, 0)
    ## lower = pmax(lower, 0)
    
    diff_upper = upper
    diff_lower = lower
 
    ## #############################################
    ## Iteration over the linear constraints of Theorem 3 to bound omega_xw
    upper = pmin(omega_upper,
                    diff_upper[,i_xp.w, drop=FALSE] - LK_XY.W[,c(4,3,8,7), drop=FALSE] + UK_XY.W[,c(3,4,7,8), drop=FALSE] +
                                                    + UK_XY.W[,c(8,7,4,3), drop=FALSE] + UK_XY.W[,c(7,8,3,4), drop=FALSE] +
                   - LChi[,i_x.w, drop=FALSE]*(U_bar + L_bar) + 2*eps_w + UChi[,i_x.wp, drop=FALSE] + U_bar)
    upper = pmin(upper,
                    diff_upper[,i_xp.wp, drop=FALSE] - LK_XY.W[,c(8,7,4,3), drop=FALSE] + UK_XY.W[,c(7,8,3,4), drop=FALSE] +
                                                     + UK_XY.W[,c(4,3,8,7), drop=FALSE] + UK_XY.W[,c(3,4,7,8), drop=FALSE] +
                   + 2*eps_w*UChi[,i_x.wp, drop=FALSE] - LChi[,i_x.wp, drop=FALSE]*(U_bar + L_bar) + U_bar)

    lower = pmax(omega_lower,
                  - diff_lower[,i_xp.wp, drop=FALSE] - UK_XY.W[,c(8,7,4,3), drop=FALSE] + LK_XY.W[,c(3,4,7,8), drop=FALSE] +
                                                     + LK_XY.W[,c(4,3,8,7), drop=FALSE] + LK_XY.W[,c(7,8,3,4), drop=FALSE] +
                  - UChi[,i_x.wp, drop=FALSE]*(U_bar + L_bar + 2*eps_w) + L_bar)
    lower = pmax(lower,
                  - diff_lower[,i_xp.w, drop=FALSE] - UK_XY.W[,c(4,3,8,7), drop=FALSE] + LK_XY.W[,c(7,8,3,4), drop=FALSE] +
                                                    + LK_XY.W[,c(8,7,4,3), drop=FALSE] + LK_XY.W[,c(3,4,7,8), drop=FALSE] +
                  - 2*eps_w*UChi[,i_x.wp, drop=FALSE] - UChi[,i_x.w, drop=FALSE]*(U_bar + L_bar) + L_bar)

    omega_upper = upper
    omega_lower = lower

    pass = pass*rowMins(omega_upper >= omega_lower)
  }

  if (return_solutions)  {
    intervals <- matrix(0, nrow = N, ncol = 2)

    alpha_upper = beta_upper * pmin(omega_upper, 1)
    alpha_lower = beta_lower * omega_lower

    intervals[, 2] <- (alpha_upper[, 4] - alpha_lower[, 3]) * P_W + (alpha_upper[, 2] - alpha_lower[, 1]) * (1 - P_W)
    intervals[, 1] <- (alpha_lower[, 4] - alpha_upper[, 3]) * P_W + (alpha_lower[, 2] - alpha_upper[, 1]) * (1 - P_W)
    intervals[pass == 0, 1] <- NA
    intervals[pass == 0, 2] <- NA
    return(intervals)
  }

  return(pass)
}

############################################################################################
# bayesian_interval_generation_analytical::
#
# Basically uses 'bayesian_validate_constraints_analytical' with a flag indicating
# intervals should be returned.

bayesian_interval_generation_analytical <- function(theta, epsilons)
{
  return(bayesian_validate_constraints_analytical(theta$W0, theta$W1, epsilons, TRUE, theta$W))
}
