function [J validW] = convertToIsing(W)
% [J validW] = convertToIsing(W)
% Takes the parameters of a cascaded logistic model and coverts to an Ising
% model. See appendix of the reference for the derivation.
%
% Input
%    W: (m x m) lower triagular matrix containing cascaded logistic parameters
%        W(i,1) = h_{i} (offset terms)
%        W(i,j) = w_{i,j-1}, 1<j<=i (interaction term between neurons j-1 and i)
%
% Output
%    J: (m x m) Ising model parameters
%    validW: true if conversion exists 
%
% Reference:
% - Il Memming Park, Evan Archer, Kenneth Latimer, Jonathan Pillow. Universal models for binary spike patterns using centered Dirichlet processes. Neural Information Processing Systems (NIPS) 2013

m = size(W,1);
J = zeros(m,m);

for ii = 1:m
    J(ii,ii) = W(ii,1) + J(ii,ii);
end

for ii = m:-1:2
    idx = find(W(ii,2:ii)~=0);
    
    if(numel(idx) == 2)
	idxc = idx+1;
	J(idx(1),idx(2)) = log(1+exp(W(ii,1)+W(ii,idxc(1)))) ...
	    + log(1+exp(W(ii,1)+W(ii,idxc(2)))) - log(1+exp(W(ii,1))) ...
	    - log(1+exp(W(ii,1)+W(ii,idxc(1))+W(ii,idxc(2)))) ...
	    + J(idx(1),idx(2));

	J(idx(1),ii) = W(ii,idxc(1)) + J(idx(1),ii);
	J(idx(2),ii) = W(ii,idxc(2)) + J(idx(2),ii);
	
	J(idx(1),idx(1)) = log(1+exp(W(ii,1))) ...
	    - log(1+exp(W(ii,1) + W(ii,idxc(1)))) + J(idx(1),idx(1));

	J(idx(2),idx(2)) = log(1+exp(W(ii,1))) ...
	    - log(1+exp(W(ii,1) + W(ii,idxc(2)))) + J(idx(2),idx(2));

    elseif(numel(idx) == 1)
	idxc = idx+1;
	J(idx(1),ii) = W(ii,idxc(1)) + J(idx(1),ii);
	
	J(idx(1),idx(1)) = log(1+exp(W(ii,1))) ...
	    - log(1+exp(W(ii,1) + W(ii,idxc(1)))) + J(idx(1),idx(1));

    elseif(numel(idx) > 2)
	warning('This cascaded logistic model does not have an equilavent Ising representation.');
	validW = false;
	return
    end
end

validW = true;
