%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% test_methods:
%
% Evaluates several selection methods by posterior mean reconstruction.
%
% The 'naive' method is based on optimizing the reliability index
% heuristic. 'bound1' is the bounded neighborhood method, while 'bound2'
% is the tree-based method. The 'rand' method is based on just choosing a
% selection uniformly at random. 'gauss' is the Gaussian entropy greedy
% optimization method.
%
% Input:
%
% - K: size of the selection
% - L: matrix of loadings
% - S: covariance matrix of latent variables
% - y_keep: subset of 1:num_y in which the selection is done
% - K_width: maximum size of neighborhood for the neighborhood method
% - Y: data used in the evaluation
% - num_mcmc: number of MCMC iterations used in the evaluation
% - burn_in: burn-in period for the MCMC evaluation
% - sd: seed for random number generation
%
% Output:
%
% - mean_*: latent variable reconstruction for each test point and each
%           method.
% - mean_all: latent variable reconstruction for each test point when all
%             variables in y_keep are used. This can then be contrasted
%             with the other reconstruction to see which one is the closest
%             to mean_all.
%
% Created by: Ricardo Silva, London, 21/04/2011
% University College London
%
% Current version: 24/09/2011

function [mean_bound1, mean_bound2, mean_gauss, mean_naive, ...
          mean_rand, mean_all, sel_K_rand, sel_K_naive, sel_K_bound1, ...
          sel_K_bound2, sel_K_gauss] = ...
          test_methods(K, L, S, y_keep, K_width, Y, num_mcmc, burn_in, sd)
      
rng(sd) % Seed setting (for reproducibility)
invS = inv(S);

% Main function %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

sel_K_rand   = sort(randsample(y_keep, K));
sel_K_naive  = y_keep(optimize_probitsem_naive(K, L(y_keep, :), S));
sel_K_all    = y_keep;

fprintf('> Fitting KL approximation ***********************************\n');
fprintf('> Preserving %d items out of %d\n\n...', K, length(y_keep));
[sel_K_bound1 optim1] = optimize_probitsem_entropy_bound(K_width, L(y_keep, :), S, sel_K_naive, true, true); 
sel_K_bound1 = y_keep(sel_K_bound1);
if optim1
  disp('[BOUND1]: Optimal solution found')
end
sel_K_bound2_greedy = optimize_probitsem_pairwise_entropy_bound_greedy(L(y_keep, :), S, sel_K_naive);
[sel_K_bound2 optim2] = optimize_probitsem_pairwise_entropy_bound(L(y_keep, :), S, sel_K_bound2_greedy);
sel_K_bound2 = y_keep(sel_K_bound2);
if optim2
  disp('[BOUND2]: Optimal solution found')
end
sel_K_gauss = y_keep(optimize_probitsem_entropy_gaussian(L(y_keep, :), ...
              S, sel_K_naive));
fprintf('> End of KL approximation fitting ****************************\n\n');

% Evaluation %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

fprintf('> Generating samples...\n');

mean_bound1 = mcmc_probitsem(Y(:, sel_K_bound1), L(sel_K_bound1, :), invS, num_mcmc, burn_in, false);
mean_bound2 = mcmc_probitsem(Y(:, sel_K_bound2), L(sel_K_bound2, :), invS, num_mcmc, burn_in, false);
mean_gauss  = mcmc_probitsem(Y(:, sel_K_gauss), L(sel_K_gauss, :), invS, num_mcmc, burn_in, false);
mean_naive  = mcmc_probitsem(Y(:, sel_K_naive), L(sel_K_naive, :), invS, num_mcmc, burn_in, false);
mean_rand   = mcmc_probitsem(Y(:, sel_K_rand),  L(sel_K_rand, :),  invS, num_mcmc, burn_in, false);
mean_all    = mcmc_probitsem(Y(:, sel_K_all),   L(sel_K_all, :),   invS, num_mcmc, burn_in, false);

fprintf('> Reconstruction error [MEAN] (KL1/KL2/Gauss/Rand/Naive):\n');
disp([mean(mean(abs(mean_bound1 - mean_all))) ...
      mean(mean(abs(mean_bound2 - mean_all))) ...
      mean(mean(abs(mean_gauss  - mean_all))) ...
      mean(mean(abs(mean_rand   - mean_all))) ...
      mean(mean(abs(mean_naive  - mean_all)))]);
