function [theta, ff, aux, cholK] = update_theta_aux_fixed(theta, ff, Lfn, Kfn, aux, theta_Lprior, slice_width)
%UPDATE_THETA_AUX_FIXED MCMC update to GP hyper-param based on deterministic reparameterization
%
%     [theta, ff] = update_theta_aux_noise(theta, ff, Lfn, Kfn, aux, theta_Lprior);
%
% Inputs:
%             theta Kx1 hyper-parameters (can be an array of any size)
%                ff Nx1 apriori Gaussian values
%               Lfn @fn Log-likelihood function, Lfn(ff) returns a scalar
%               Kfn @fn K = Kfn(theta) returns NxN covariance matrix
%                       NB: this should contain jitter (if necessary) to
%                       ensure the result is positive definite.
%
% Specify aux in one of three ways:
% ------------------------------------------
%     [aux_std, gg] Nx2 column1: std-dev of auxiliary noise (can also be a 1x1).
%                       column2: point estimates of latent values
% OR function:
%            aux_fn @fn [aux_std, gg] = aux_fn(theta, K);
% OR caching function:
%               aux cel A pair: {aux_fn, aux_cache}, used like this:
%                       [aux_std, gg, aux_cache] = aux_fn(theta, K, aux_cache);
%                       The cache could be used (for example) to notice that
%                       relevant parts of theta or K haven't changed, and
%                       to immediately return the previously computed values.
% ---------------------------------
%
%      theta_Lprior @fn Log-prior, theta_Lprior(theta) returns a scalar
%
% Outputs:
%             theta Kx1 updated hyper-parameters (Kx1 or same size as inputted)
%                ff Nx1 updated apriori Gaussian values
%               aux  -  Last [aux_std, gg] computed, or {aux_std_fn, aux_cache},
%                       depending on what was passed in.
%             cholK NxN chol(Kfn(theta))
%
% The model is ff ~ N(0, K), log(p(observations|ff)) = Lfn(ff)
% Reparameterize by "whitening" under the pseudo-posterior with
% pseudo-observations gg with Gaussian noise with width(s) aux_std.

% Iain Murray, November 2009, January 2010, April 2010

% If there is a good reason for it, there's no real reason full-covariance
% auxiliary noise couldn't be used. It would just be more expensive as sampling
% would require decomposing the noise covariance matrix. For now this code
% hasn't implemented that option.

N = numel(ff);

% Start constructing the struct that will be passed around while slicing
pp = struct('pos', theta, 'Kfn', Kfn);
if isnumeric(aux)
    % Fixed auxiliary noise level
    pp.adapt_aux = 0;
    pp.aux_std = aux(:,1);
    pp.aux_var = aux(:,1).*aux(:,1);
    pp.gg = aux(:,2);
    pp.Sinv_g = pp.gg ./ pp.aux_var;
elseif iscell(aux)
    % Adapting noise level, with computations cached
    pp.adapt_aux = 2;
    pp.aux_fn = aux{1};
    pp.aux_cache = aux{2};
else
    % Simple function to choose noise level
    pp.adapt_aux = 1;
    pp.aux_fn = aux;
end
pp = theta_changed(pp);

% Instantiate nu|f,gg
pp.nu = pp.U_invR*ff(:) - pp.U_invR'\pp.Sinv_g;

% Compute current log-prob (up to constant) needed by slice sampling:
theta_unchanged = true; % theta hasn't moved yet, don't recompute covariances
pp = eval_particle(pp, -Inf, Lfn, theta_Lprior, theta_unchanged);

% Slice sample update of theta|g,nu
step_out = (slice_width > 0);
slice_width = abs(slice_width);
slice_fn = @(pp, Lpstar_min) eval_particle(pp, Lpstar_min, Lfn, theta_Lprior);
pp = slice_sweep(pp, slice_fn, slice_width, step_out);
theta = pp.pos;
ff = reshape(pp.ff, size(ff));

% Return some cached values
if iscell(aux)
    aux = {pp.aux_fn, pp.aux_cache};
else
    aux = [pp.aux_std.*ones(size(pp.gg)), pp.gg];
end
cholK = pp.U;

function pp = theta_changed(pp)
% Will call after changing hyperparameters to update covariances and
% their decompositions.
theta = pp.pos;
K = pp.Kfn(theta);
if pp.adapt_aux
    if pp.adapt_aux == 1
        [pp.aux_std, pp.gg] = pp.aux_fn(theta, K);
    elseif pp.adapt_aux == 2
        [pp.aux_std, pp.gg, pp.aux_cache] = pp.aux_fn(theta, K, pp.aux_cache);
    end
    pp.aux_var = pp.aux_std .* pp.aux_std;
    pp.Sinv_g = pp.gg ./ pp.aux_var;
end
pp.U  = chol(K);
pp.iK = inv(K);
pp.U_invR = chol(plus_diag(pp.iK, 1./pp.aux_var));

function pp = eval_particle(pp, Lpstar_min, Lfn, theta_Lprior, theta_unchanged)

% Prior on theta
theta = pp.pos;
Ltprior = theta_Lprior(theta);
if Ltprior == -Inf
    pp.on_slice = false;
    return;
end
if ~exist('theta_unchanged', 'var') || (~theta_unchanged)
    pp = theta_changed(pp);
end

% Update f|gg,nu,theta
pp.ff = pp.U_invR\pp.nu + solve_chol(pp.U_invR, pp.Sinv_g);

% Compute joint probability and slice acceptability.
% I have dropped the constant: -0.5*length(pp.ff)*log(2*pi)
Lfprior = -0.5*(pp.ff'*solve_chol(pp.U, pp.ff)) - sum(log(diag(pp.U)));
LJacobian = -sum(log(diag(pp.U_invR)));
pp.Lpstar = Ltprior + Lfprior + Lfn(pp.ff) + LJacobian;
pp.on_slice = (pp.Lpstar >= Lpstar_min);

