function setup = setup_ionosphere()

[setup.train_x setup.train_y setup.test_x setup.test_y] = get_ionosphere_data();

setup.runs           = 10;
setup.iterations     = 5000;
setup.burn           = 1000;
setup.ess_iterations = 10;
setup.max_ls         = 10.0;
setup.min_ls         = 0.01;
setup.max_gain       = 10.0;
setup.min_gain       = 0.01;

jitter    = 1e-6;
gpml_covs = {'covSum', {'covSEard', 'covNoise'}};

setup.slice_width     = 10;
setup.llh_fn          = @ionosphere_llh;
setup.cov_fn          = @(theta) feval(gpml_covs{:}, [theta ; 0 ; log(jitter)], setup.train_x);
setup.theta_log_prior = @(theta) log(1.0*all((theta>log(setup.min_ls)) & (theta<log(setup.max_ls))));
setup.aux_noise_fn    = @aux_noise;
setup.update_gain     = @update_gain;
setup.train_error_fn  = @train_error;

  function llh = ionosphere_llh(ff, gain)
    % logistic model
    probs = 1./(1 + exp(-gain*ff));
    llh   = sum(setup.train_y.*log(probs) + (1-setup.train_y).*log(1-probs));
  end

  function [std gg] = aux_noise(theta, K, gain)
    [std gg] = logistic_aux(2*setup.train_y-1, 2*log(gain), 0);
    std = std/gain;
    gg  = gg/gain;
  end

  function [gain llh] = update_gain(gain, ff, cur_llh)
    
    % Slice sample
    particle = struct('pos', gain, 'ff', ff);
    particle = slice_fn(particle, -Inf);
    particle = slice_sweep(particle, @slice_fn, 1, 0);
    gain     = particle.pos;
    llh      = particle.Lpstar;
  end

  function pp = slice_fn(pp, Lpstar_min)
    
    gain = pp.pos;
    
    if (gain < setup.min_gain) || (gain > setup.max_gain)
      pp.Lpstar   = -Inf;
      pp.on_slice = false;
      return;
    end
    
    pp.Lpstar   = ionosphere_llh(pp.ff, gain);
    pp.on_slice = (pp.Lpstar >= Lpstar_min);
  end

  function err = train_error(ff)
    err = 1.0 - mean((ff > 0) == setup.train_y);
  end

end
