function [aux_std, gg] = poiss_aux(counts, log_prior_var, prior_mean)
%POISS_AUX return effective Gaussian likelihood noise level (and centres)
%
%     [aux_std, gg] = poiss_aux(counts, log_prior_var, prior_mean)
%
% Inputs:
%             counts Nx1 
%      log_prior_var 1x1 or Nx1
%         prior_mean 1x1 or Nx1
%
% Outputs:
%            aux_std Nx1 
%                 gg Nx1 (if needed) 

% Iain Murray, April 2010

prior_var = exp(log_prior_var);
prior_precision = 1./prior_var;

% % Straight-up Laplace approx:
% mu = counts.*prior_var + prior_mean - lambertw_approx(prior_var.*exp(prior_mean + counts.*prior_var));
% post_var = sqrt(prior_var ./ (1 + exp(mu).*prior_var));

% Nearly the same as Laplace and cheaper
idx = (counts > 0);
mu = zeros(size(counts));
mu(idx) = (msk(prior_var, idx).*counts(idx).*log(counts(idx)) + msk(prior_mean, idx)) ...
        ./ (1 + counts(idx).*msk(prior_var, idx));
% TODO could come up with a cheap proxy for zero counts as well. Currently do Laplace:
mu(~idx) = msk(prior_mean, ~idx) - lambertw_approx(msk(prior_var, ~idx).*exp(msk(prior_mean, ~idx)));
% Just need to be in the right ball-park for the variance
%mu(~idx) = log(0.5) + msk(prior_mean, ~idx); % Ok for small prior_mean, but not for big
post_var = prior_var ./ (1 + exp(mu).*prior_var);

post_precision = 1./post_var;
mask = (post_precision > prior_precision);
aux_std = zeros(size(mask));
aux_std(mask) = sqrt(1 ./ (post_precision(mask) - msk(prior_precision, mask)));
aux_std(~mask) = Inf;


if nargout > 1
    gg = (aux_std.^2).*(mu.*post_precision - prior_mean.*prior_precision);
    % Get rid of infinities, which disappear in sensible limits anyway:
    BIG = 1e100;
    gg = min(gg, BIG);
end


function xx = msk(A, mask)
%MSK msk(A, mask) returns A(mask), or just A if A is a scalar.
%
%This is useful for when A is a scalar standing in for an array with all
%elements equal.
if numel(A) == 1
    xx = A;
else
    xx = A(mask);
end
